<?php

namespace Tests\Feature;

use App\Data\CreateStockTakeData;
use App\Helpers;
use App\Http\Requests\StoreInventoryAdjustment;
use App\Models\AccountingTransactionLine;
use App\Models\Currency;
use App\Models\Customer;
use App\Models\IntegrationInstance;
use App\Models\InventoryAdjustment;
use App\Models\Product;
use App\Models\SalesCredit;
use App\Models\SalesCreditLine;
use App\Models\SalesCreditReturnLine;
use App\Models\SalesOrderFulfillmentLine;
use App\Models\SalesOrderLine;
use App\Models\Setting;
use App\Models\StockTake;
use App\Models\StockTakeItem;
use App\Models\User;
use App\Models\Warehouse;
use App\Services\Accounting\AccountingTransactionManager;
use App\Services\StockTake\StockTakeManager;
use Carbon\Carbon;
use Exception;
use Laravel\Sanctum\Sanctum;
use Plannr\Laravel\FastRefreshDatabase\Traits\FastRefreshDatabase;
use Queue;
use Tests\TestCase;
use Throwable;

class NominalCodeTest extends TestCase
{
    use FastRefreshDatabase;

    /**
     * @throws Exception
     * @throws Throwable
     */
    public function test_it_can_override_nominal_codes_from_integration_settings(): void
    {
        Queue::fake();

        /** @var IntegrationInstance $integrationInstance */
        $integrationInstance = IntegrationInstance::query()->first();
        // Mid-level specificity, overrides general but not product level
        $integrationInstance->integration_settings = [
            'sales_nominal_code_id' => 2,
            'cogs_nominal_code_id' => 4,
        ];
        $integrationInstance->save();

        Setting::query()->where('key', Setting::KEY_NC_MAPPING_SALES_ORDERS)->update(['value' => 3]);
        Setting::query()->where('key', Setting::KEY_NC_MAPPING_COGS)->update(['value' => 5]);

        /** @var Warehouse $warehouse */
        $warehouse = Warehouse::factory()->create()->withDefaultLocation();

        /** @var Product $product_1 */
        $product_1 = Product::factory()->create([
            'sku' => 'product-1',
            'name' => 'Product-1',
            'type' => Product::TYPE_STANDARD,
            'sales_nominal_code_id' => 1,  // Most specific, overrides all
            'cogs_nominal_code_id' => 1,  // Most specific, overrides all
        ]);

        /** @var Product $product_2 */
        $product_2 = Product::factory()->create([
            'sku' => 'product-2',
            'name' => 'Product-2',
            'type' => Product::TYPE_STANDARD,
            'sales_nominal_code_id' => null,
        ]);

        // Get stock in so the products can be fulfilled

        $product_1->setInitialInventory($warehouse->id, 100, 5.00);
        $product_2->setInitialInventory($warehouse->id, 100, 5.00);

        /** @var Product $product_3 */
        $product_3 = Product::factory()->create([
            'sku' => 'product-3',
            'name' => 'Product-3',
            'type' => Product::TYPE_STANDARD,
            'sales_nominal_code_id' => null,
        ]);

        /** @var User $user */
        $user = User::query()->first();
        Sanctum::actingAs($user);

        /** @var Customer $customer */
        $customer = Customer::factory()->create();

        $response = $this->postJson('/api/sales-orders', [
            'order_status' => 'open',
            'currency_code' => 'USD',
            'order_date' => Carbon::now(),
            'customer_id' => $customer->id,
            'sales_order_lines' => [
                [
                    'product_id' => $product_1->id,
                    'description' => $product_1->name,
                    'quantity' => 1,
                    'amount' => 150,
                    'warehouse_id' => $warehouse->id,
                ],
                [
                    'product_id' => $product_2->id,
                    'description' => $product_2->name,
                    'quantity' => 1,
                    'amount' => 150,
                    'warehouse_id' => $warehouse->id,
                ],
            ],
        ])->assertOk();

        $salesOrderData = $response->json()['data'];

        $this->assertDatabaseHas((new SalesOrderLine())->getTable(), [
            'id' => $salesOrderData['item_info'][0]['sales_order_line_id'],
            'nominal_code_id' => $product_1->sales_nominal_code_id,
        ]);

        $this->assertDatabaseHas((new SalesOrderLine())->getTable(), [
            'id' => $salesOrderData['item_info'][1]['sales_order_line_id'],
            'nominal_code_id' => $integrationInstance['integration_settings']['sales_nominal_code_id'],
        ]);

        /*
         * COGS Nominal code settings only happen when accounting transactions are generated.  So need to generate accounting
         * transaction here to test.  COGS occurs when a product is fulfilled, adjusted, returned, stock taken.
         * We should test each scenario.
         */

        // Fulfillment
        $this->postJson("/api/sales-orders/{$salesOrderData['id']}/fulfill", [
            'sales_order_id' => 1,
            'fulfilled_at' => Carbon::now(),
            'fulfillment_type' => 'manual',
            'warehouse_id' => $warehouse->id,
            'fulfillment_lines' => [
                [
                    'sales_order_line_id' => $salesOrderData['item_info'][0]['sales_order_line_id'],
                    'quantity' => 1,
                ],
                [
                    'sales_order_line_id' => $salesOrderData['item_info'][1]['sales_order_line_id'],
                    'quantity' => 1,
                ],
            ],
        ])->assertOk();

        // Adjustment
        $adjustment1 = $this->postJson('/api/inventory-adjustments', [
            'adjustment_date' => now(),
            'product_id' => $product_1->id,
            'warehouse_id' => $warehouse->id,
            'unit_cost' => 2.50,
            'quantity' => 50,
            'adjustment_type' => StoreInventoryAdjustment::ADJUSTMENT_TYPE_INCREASE,
        ])->assertOk();
        $adjustment2 = $this->postJson('/api/inventory-adjustments', [
            'adjustment_date' => now(),
            'product_id' => $product_2->id,
            'warehouse_id' => $warehouse->id,
            'unit_cost' => 2.50,
            'quantity' => 50,
            'adjustment_type' => StoreInventoryAdjustment::ADJUSTMENT_TYPE_INCREASE,
        ])->assertOk();

        // Credit and Return
        $response = $this->postJson('/api/sales-credits', [
            'sales_order_id' => $salesOrderData['id'],
            'order_status' => SalesCredit::CREDIT_STATUS_OPEN,
            'credit_date' => now(),
            'customer_id' => $customer->id,
            'currency_code' => Currency::default()->code,
            'sales_credit_lines' => [
                [
                    'sales_order_line_id' => $salesOrderData['item_info'][0]['sales_order_line_id'],
                    'amount' => 5.00,
                    'quantity' => 1,
                ],
                [
                    'sales_order_line_id' => $salesOrderData['item_info'][1]['sales_order_line_id'],
                    'amount' => 5.00,
                    'quantity' => 1,
                ],
            ],
        ])->assertSuccessful();

        $salesCreditData = $response->json()['data'];

        $response = $this->postJson('/api/sales-credits/receive', [
            'sales_credit_id' => $salesCreditData['id'],
            'warehouse_id' => $warehouse->id,
            'received_at' => now(),
            'return_lines' => [
                [
                    'sales_credit_line_id' => $salesCreditData['item_info'][0]['id'],
                    'action' => SalesCreditReturnLine::ACTION_ADD_TO_STOCK,
                    'reason_id' => 1,
                    'quantity' => 1,
                ],
                [
                    'sales_credit_line_id' => $salesCreditData['item_info'][1]['id'],
                    'action' => SalesCreditReturnLine::ACTION_ADD_TO_STOCK,
                    'reason_id' => 1,
                    'quantity' => 1,
                ],
            ],
        ])->assertOk();

        $salesCreditLine1 = SalesCreditLine::find($salesCreditData['item_info'][0]['id']);
        $salesCreditLine2 = SalesCreditLine::find($salesCreditData['item_info'][1]['id']);
        $salesCreditReturnLine1 = $salesCreditLine1->salesCreditReturnLines()->first();
        $salesCreditReturnLine2 = $salesCreditLine2->salesCreditReturnLines()->first();

        // Stock Take
        $stockTakeManager = app(StockTakeManager::class);
        $stockTake = $stockTakeManager->createStockTake(CreateStockTakeData::from([
            'warehouse_id' => $warehouse->id,
            'date_count' => now(),
            'items' => [
                [
                    'product_id' => $product_1->id,
                    'qty_counted' => 500,
                    'unit_cost' => 2.50,
                ],
                [
                    'product_id' => $product_2->id,
                    'qty_counted' => 500,
                    'unit_cost' => 2.50,
                ],
            ],
        ]));
        $stockTakeManager->initiateCount($stockTake);
        $stockTakeManager->finalizeStockTake($stockTake);

        app(AccountingTransactionManager::class)->sync();

        // Assertions... all types should be credit (except fulfillment) on cogs since we increased inventory for each scenario

        // Fulfillment
        //dd(AccountingTransactionLine::query()->where('link_type', SalesOrderFulfillmentLine::class)->get()->toArray());
        $this->assertDatabaseHas((new AccountingTransactionLine())->getTable(), [
            'nominal_code_id' => $product_1->cogs_nominal_code_id,
            'type' => AccountingTransactionLine::TYPE_DEBIT,
            #'link_id' => 1,
            'link_type' => SalesOrderFulfillmentLine::class,
        ]);

        $this->assertDatabaseHas((new AccountingTransactionLine())->getTable(), [
            'nominal_code_id' => $integrationInstance->integration_settings['cogs_nominal_code_id'],
            'type' => AccountingTransactionLine::TYPE_DEBIT,
            #'link_id' => 2,
            'link_type' => SalesOrderFulfillmentLine::class,
        ]);

        // Adjustment

        $this->assertDatabaseHas((new AccountingTransactionLine())->getTable(), [
            'nominal_code_id' => $product_1->cogs_nominal_code_id,
            'type' => AccountingTransactionLine::TYPE_CREDIT,
            #'link_id' => $adjustment1->json('data.id'),
            'link_type' => InventoryAdjustment::class,
        ]);

        $this->assertDatabaseHas((new AccountingTransactionLine())->getTable(), [
            'nominal_code_id' => Helpers::setting(Setting::KEY_NC_MAPPING_COGS), // No integration mapping for adjustment
            'type' => AccountingTransactionLine::TYPE_CREDIT,
            #'link_id' => $adjustment2->json('data.id'),
            'link_type' => InventoryAdjustment::class,
        ]);

        // Return

        $this->assertDatabaseHas((new AccountingTransactionLine())->getTable(), [
            'nominal_code_id' => $product_1->cogs_nominal_code_id,
            'type' => AccountingTransactionLine::TYPE_CREDIT,
            'link_id' => $salesCreditReturnLine1->id,
            'link_type' => SalesCreditReturnLine::class,
        ]);

        $this->assertDatabaseHas((new AccountingTransactionLine())->getTable(), [
            'nominal_code_id' => $integrationInstance->integration_settings['cogs_nominal_code_id'],
            'type' => AccountingTransactionLine::TYPE_CREDIT,
            'link_id' => $salesCreditReturnLine2->id,
            'link_type' => SalesCreditReturnLine::class,
        ]);

        // Stock Take

        $this->assertDatabaseHas((new AccountingTransactionLine())->getTable(), [
            'nominal_code_id' => $product_1->cogs_nominal_code_id,
            'type' => AccountingTransactionLine::TYPE_CREDIT,
            #'link_id' => 1,
            'link_type' => StockTakeItem::class,
        ]);

        $this->assertDatabaseHas((new AccountingTransactionLine())->getTable(), [
            'nominal_code_id' => Helpers::setting(Setting::KEY_NC_MAPPING_COGS), // No integration mapping for stock take
            'type' => AccountingTransactionLine::TYPE_CREDIT,
            #'link_id' => 2,
            'link_type' => StockTakeItem::class,
        ]);

        $integrationInstance->integration_settings = [];
        $integrationInstance->save();

        $response = $this->postJson('/api/sales-orders', [
            'order_status' => 'open',
            'currency_code' => 'USD',
            'order_date' => Carbon::now(),
            'customer_id' => $customer->id,
            'sales_order_lines' => [
                [
                    'product_id' => $product_3->id,
                    'description' => $product_3->name,
                    'quantity' => 1,
                    'amount' => 150,
                ],
            ],
        ])->assertOk();

        $salesOrderData = $response->json('data');

        $this->assertDatabaseHas((new SalesOrderLine())->getTable(), [
            'id' => $salesOrderData['item_info'][0]['sales_order_line_id'],
            'nominal_code_id' => Helpers::setting(Setting::KEY_NC_MAPPING_SALES_ORDERS),
        ]);
    }
}
