<?php

namespace Tests\Feature;

use App\Http\Requests\StoreInventoryAdjustment;
use App\Models\FifoLayer;
use App\Models\Product;
use App\Models\SalesOrder;
use App\Models\User;
use App\Models\Warehouse;
use Laravel\Sanctum\Sanctum;
use Plannr\Laravel\FastRefreshDatabase\Traits\FastRefreshDatabase;
use Tests\TestCase;

class InventoryManagerTest extends TestCase
{

    use FastRefreshDatabase;

    public function test_it_charges_fifo_when_it_takes_stock_from_other_sources(): void
    {
        Sanctum::actingAs(User::factory()->create());
        $product = Product::factory()->create();
        $warehouse = Warehouse::factory()->create()->withDefaultLocation();

        // Make stock available
        $this->postJson('/api/inventory-adjustments', [
           'product_id' => $product->id,
            'warehouse_id' => $warehouse->id,
            'adjustment_type' => StoreInventoryAdjustment::ADJUSTMENT_TYPE_INCREASE,
            'unit_cost' => 10,
            'quantity' => 1,
            'adjustment_date' => now()
        ])->assertSuccessful();

        // Create sales order
        $this->postJson('/api/sales-orders', [
            'order_date' => now(),
            'order_status' => SalesOrder::STATUS_OPEN,
            'currency_code' => 'USD',
            'sales_order_lines' => [
                [
                    'product_id' => $product->id,
                    'warehouse_id' => $warehouse->id,
                    'description' => $product->name,
                    'amount' => 5,
                    'quantity' => 1
                ]
            ]
        ])->assertSuccessful();

        // Create negative adjustment to take from stock
        $this->postJson('/api/inventory-adjustments', [
            'product_id' => $product->id,
            'warehouse_id' => $warehouse->id,
            'adjustment_type' => StoreInventoryAdjustment::ADJUSTMENT_TYPE_DECREASE,
            'quantity' => 1,
            'adjustment_date' => now()
        ])->assertSuccessful();

        // Fifo should be used up
        $this->assertDatabaseHas('fifo_layers', [
            'product_id' => $product->id,
            'warehouse_id' => $warehouse->id,
            'original_quantity' => 1,
            'fulfilled_quantity' => 1
        ]);
    }

    public function test_it_changes_fifo_layer_total_cost_when_adjustment_edited(): void
    {
        Sanctum::actingAs(User::first());
        $product = Product::factory()->create();
        $warehouse = Warehouse::factory()->create()->withDefaultLocation();

        $response = $this->postJson(route('inventory-adjustments.store'), [
            'product_id' => $product->id,
            'warehouse_id' => $warehouse->id,
            'adjustment_type' => StoreInventoryAdjustment::ADJUSTMENT_TYPE_INCREASE,
            'unit_cost' => 10,
            'quantity' => 10,
            'adjustment_date' => now()
        ])->assertSuccessful();

        $inventoryAdjustmentId = $response->json()['data']['id'];

        $this->assertDatabaseHas((new FifoLayer())->getTable(), [
            'product_id' => $product->id,
            'warehouse_id' => $warehouse->id,
            'original_quantity' => 10,
            'total_cost' => 100
        ]);

        $this->putJson(route('inventory-adjustments.update', $inventoryAdjustmentId), [
            'adjustment_type' => StoreInventoryAdjustment::ADJUSTMENT_TYPE_INCREASE,
            'adjustment_date' => now(),
            'unit_cost' => 10,
            'quantity' => 20,
            'warehouse_id' => $warehouse->id,
        ])->assertSuccessful();

        $this->assertDatabaseCount((new FifoLayer())->getTable(), 1);
        $this->assertDatabaseHas((new FifoLayer())->getTable(), [
            'product_id' => $product->id,
            'warehouse_id' => $warehouse->id,
            'original_quantity' => 20,
            'total_cost' => 200
        ]);
    }
}
