<?php

namespace Tests\Feature;

use App\Console\Commands\Inventory\Integrity\InvalidFifoLayerFulfilledQuantityCache;
use App\Exceptions\InsufficientStockException;
use App\Models\BackorderQueue;
use App\Models\FifoLayer;
use App\Models\InventoryAdjustment;
use App\Models\InventoryMovement;
use App\Models\Product;
use App\Models\SalesOrder;
use App\Models\SalesOrderFulfillmentLine;
use App\Models\SalesOrderLine;
use App\Models\Warehouse;
use App\Services\InventoryManagement\Actions\FixInvalidFifoLayerFulfilledQuantityCache;
use App\Services\StockTake\OpenStockTakeException;
use Illuminate\Support\Facades\Queue;
use Plannr\Laravel\FastRefreshDatabase\Traits\FastRefreshDatabase;
use Tests\TestCase;
use Throwable;

class InventoryIntegrityTest extends TestCase
{
    use FastRefreshDatabase;

    /**
     * @throws OpenStockTakeException
     * @throws InsufficientStockException
     * @throws Throwable
     */
    public function test_it_can_calculate_and_fix_invalid_fifo_layer_cache(): void
    {
        Queue::fake();

        // Create inventory movements with usages more than the available quantity

        $warehouse = Warehouse::first();

        $product = Product::factory()->create([
            'sku' => 'SKU-1',
        ]);

        // Originating event

        $originatingEvent = InventoryAdjustment::factory()->create([
            'product_id' => $product->id,
            'warehouse_id' => $warehouse->id,
            'quantity' => 10,
        ]);

        $overSubscribedFifoLayer = FifoLayer::factory()->create([
            'product_id' => $product->id,
            'original_quantity' => 10,
            'fulfilled_quantity' => 10,
            'link_id' => $originatingEvent->id,
            'link_type' => InventoryAdjustment::class,
            'warehouse_id' => $warehouse->id,
        ]);

        $originatingMovement = InventoryMovement::factory()->create([
            'product_id' => $product->id,
            'warehouse_id' => $warehouse->id,
            'quantity' => $originatingEvent->quantity,
            'type' => InventoryMovement::TYPE_ADJUSTMENT,
            'inventory_status' => InventoryMovement::INVENTORY_STATUS_ACTIVE,
            'link_id' => $originatingEvent->id,
            'link_type' => InventoryAdjustment::class,
            'layer_id' => $overSubscribedFifoLayer->id,
            'layer_type' => FifoLayer::class,
        ]);

        // Usages

        $inventoryAdjustments = InventoryAdjustment::factory(6)->create([
            'product_id' => $product->id,
            'warehouse_id' => $warehouse->id,
            'quantity' => -1,
        ]);

        $inventoryAdjustments->each(function (InventoryAdjustment $inventoryAdjustment) use ($overSubscribedFifoLayer) {
            InventoryMovement::factory()->create([
                'product_id' => $inventoryAdjustment->product_id,
                'warehouse_id' => $inventoryAdjustment->warehouse_id,
                'quantity' => $inventoryAdjustment->quantity,
                'type' => InventoryMovement::TYPE_ADJUSTMENT,
                'inventory_status' => InventoryMovement::INVENTORY_STATUS_ACTIVE,
                'link_id' => $inventoryAdjustment->id,
                'link_type' => InventoryAdjustment::class,
                'layer_id' => $overSubscribedFifoLayer->id,
                'layer_type' => FifoLayer::class,
            ]);
        });

        $salesOrder = SalesOrder::factory()->create();

        $salesOrderLines = SalesOrderLine::factory(6)->create([
            'sales_order_id' => $salesOrder->id,
            'product_id' => $product->id,
            'warehouse_id' => $warehouse->id,
            'quantity' => 1,
        ]);

        $salesOrderLines->each(function (SalesOrderLine $salesOrderLine) use ($overSubscribedFifoLayer) {
            InventoryMovement::factory()->create([
                'product_id' => $salesOrderLine->product_id,
                'warehouse_id' => $salesOrderLine->warehouse_id,
                'quantity' => -$salesOrderLine->quantity,
                'type' => InventoryMovement::TYPE_SALE,
                'inventory_status' => InventoryMovement::INVENTORY_STATUS_ACTIVE,
                'link_id' => $salesOrderLine->id,
                'link_type' => SalesOrderLine::class,
                'layer_id' => $overSubscribedFifoLayer->id,
                'layer_type' => FifoLayer::class,
            ]);
        });

        $manager = app(InvalidFifoLayerFulfilledQuantityCache::class);

        $this->assertEquals([
            'id' => $overSubscribedFifoLayer->id,
            'available_quantity' => 0,
            'usage_quantity' => -2,
            'shortage_quantity' => 2,
        ], $manager->invalidFifoLayerFulfilledQuantityCache()
            ->first()
            ->only([
                'id',
                'available_quantity',
                'usage_quantity',
                'shortage_quantity',
            ]));

        // Run the remedy and assert that it fixed the data

        app(FixInvalidFifoLayerFulfilledQuantityCache::class)->fix($overSubscribedFifoLayer);

        $newestSalesOrderLineInventoryMovements = InventoryMovement::query()
            ->where('link_type', SalesOrderLine::class)
            ->orderBy('inventory_movement_date', 'desc')
            ->take(2)
            ->get();

        $this->assertDatabaseHas((new BackorderQueue())->getTable(), [
            'backordered_quantity' => 1,
            'sales_order_line_id' => $newestSalesOrderLineInventoryMovements->first()->link_id,
        ]);

        $this->assertDatabaseHas((new BackorderQueue())->getTable(), [
            'backordered_quantity' => 1,
            'sales_order_line_id' => $newestSalesOrderLineInventoryMovements->last()->link_id,
        ]);

        $this->assertDatabaseCount((new BackorderQueue())->getTable(), 2);

        $this->assertEquals(2, InventoryMovement::where('layer_type', BackorderQueue::class)->count());

        // Now reduce the original movement quantity

        $originatingMovement->update([
            'quantity' => 5,
        ]);

        $originatingEvent->update([
            'quantity' => 5,
        ]);

        $overSubscribedFifoLayer->update([
            'original_quantity' => 6,
            'fulfilled_quantity' => 6,
        ]);

        // Fix remaining movable usages
        app(FixInvalidFifoLayerFulfilledQuantityCache::class)->fix($overSubscribedFifoLayer);

        $this->assertDatabaseCount((new BackorderQueue())->getTable(), 6);

        $this->assertEquals(6, InventoryMovement::where('layer_type', BackorderQueue::class)->count());

        // Now there are only adjustment usages left, which should delete the adjustments

        $originatingMovement->update([
            'quantity' => 1,
        ]);

        $originatingEvent->update([
            'quantity' => 1,
        ]);

        $overSubscribedFifoLayer->update([
            'original_quantity' => 1,
            'fulfilled_quantity' => 1,
        ]);

        app(FixInvalidFifoLayerFulfilledQuantityCache::class)->fix($overSubscribedFifoLayer);

        // No new backorders created
        $this->assertDatabaseCount((new BackorderQueue())->getTable(), 6);

        $this->assertEquals(6, InventoryMovement::where('layer_type', BackorderQueue::class)->count());


        // Should only be one remaining inventory adjustment
        $this->assertEquals(1, InventoryAdjustment::where('quantity', '<', 0)->count());

        // Now we create a line with qty 2 that uses the fifo layer that is fulfilled.  Making the fifo layer oversubscribed by 2, but can't delete this time.

        $fulfilledSalesOrderLine = SalesOrderLine::factory()->create([
            'sales_order_id' => $salesOrder->id,
            'product_id' => $product->id,
            'warehouse_id' => $warehouse->id,
            'quantity' => 2,
        ]);

        InventoryMovement::factory()->create([
            'product_id' => $fulfilledSalesOrderLine->product_id,
            'warehouse_id' => $fulfilledSalesOrderLine->warehouse_id,
            'quantity' => -$fulfilledSalesOrderLine->quantity,
            'type' => InventoryMovement::TYPE_SALE,
            'inventory_status' => InventoryMovement::INVENTORY_STATUS_ACTIVE,
            'link_id' => $fulfilledSalesOrderLine->id,
            'link_type' => SalesOrderLine::class,
            'layer_id' => $overSubscribedFifoLayer->id,
            'layer_type' => FifoLayer::class,
        ]);

        SalesOrderFulfillmentLine::factory()->create([
            'sales_order_line_id' => $fulfilledSalesOrderLine->id,
            'quantity' => $fulfilledSalesOrderLine->quantity,
        ]);

        // TODO: from manual testing this properly throws an exception, but test fails due to transaction issue
        //app(FixInvalidFifoLayerFulfilledQuantityCache::class)->fix($overSubscribedFifoLayer);

        /*
         * If there are available fifo layers that the sales order could be moved to instead of back ordering (fifo only true),
         * the fifo layer fix should be able to handle this.
         */

        $inventoryAdjustmentForFifoLayerAvailable = InventoryAdjustment::factory()->create([
            'product_id' => $product->id,
            'warehouse_id' => $warehouse->id,
            'quantity' => 2,
        ]);

        $fifoLayerAvailable = FifoLayer::factory()->create([
            'product_id' => $product->id,
            'warehouse_id' => $warehouse->id,
            'original_quantity' => 2,
            'fulfilled_quantity' => 0,
            'link_type' => InventoryAdjustment::class,
            'link_id' => $inventoryAdjustmentForFifoLayerAvailable->id,
        ]);

        InventoryMovement::factory()->create([
            'product_id' => $product->id,
            'warehouse_id' => $warehouse->id,
            'quantity' => 2,
            'type' => InventoryMovement::TYPE_ADJUSTMENT,
            'inventory_status' => InventoryMovement::INVENTORY_STATUS_ACTIVE,
            'link_id' => $inventoryAdjustmentForFifoLayerAvailable->id,
            'link_type' => InventoryAdjustment::class,
            'layer_id' => $fifoLayerAvailable->id,
            'layer_type' => FifoLayer::class,
        ]);

        app(FixInvalidFifoLayerFulfilledQuantityCache::class)->fix($overSubscribedFifoLayer);

        // Backorder count hasn't changed
        $this->assertDatabaseCount((new BackorderQueue())->getTable(), 6);

        $this->assertDatabaseHas((new InventoryMovement())->getTable(), [
            'quantity' => -2,
            'type' => InventoryMovement::TYPE_SALE,
            'inventory_status' => InventoryMovement::INVENTORY_STATUS_ACTIVE,
            'link_id' => $fulfilledSalesOrderLine->id,
            'link_type' => SalesOrderLine::class,
            'layer_id' => $fifoLayerAvailable->id,
            'layer_type' => FifoLayer::class,
        ]);

        $fifoLayerAvailable->refresh();
        $this->assertEquals(2, $fifoLayerAvailable->fulfilled_quantity);
    }
}
