<?php

namespace Tests\Feature;

use App\Exceptions\InventoryAssemblyStockException;
use App\Exceptions\KitWithMovementsCantBeChangedException;
use App\Exceptions\ProductBundleException;
use App\Http\Requests\StoreInventoryAdjustment;
use App\Http\Requests\StoreInventoryAssembly;
use App\Models\InventoryMovement;
use App\Models\Product;
use App\Models\SalesOrder;
use App\Models\User;
use App\Models\Warehouse;
use Illuminate\Foundation\Testing\DatabaseTransactions;
use Illuminate\Foundation\Testing\WithFaker;
use Laravel\Sanctum\Sanctum;
use Tests\RefreshDatabase;
use Tests\TestCase;

class InventoryAssemblyControllerTest extends TestCase
{
    use DatabaseTransactions;


    public function test_it_can_assemble_kit_from_components(){

        Sanctum::actingAs(User::factory()->create());

        /** @var Warehouse $warehouse */
        $warehouse = Warehouse::factory()->create()->withDefaultLocation();

        /** @var Product $component1 */
        $component1 = Product::factory()->create(['type' => Product::TYPE_STANDARD]);
        /** @var Product $component2 */
        $component2 = Product::factory()->create(['type' => Product::TYPE_STANDARD]);

        // Add stock to components
        $this->postJson('/api/inventory-adjustments', [
            'adjustment_date' => now(),
            'quantity' => 6,
            'product_id' => $component1->id,
            'warehouse_id' => $warehouse->id,
            'unit_cost' => 10,
            'adjustment_type' => StoreInventoryAdjustment::ADJUSTMENT_TYPE_INCREASE
        ])->assertSuccessful();
        $this->postJson('/api/inventory-adjustments', [
            'adjustment_date' => now(),
            'quantity' => 5,
            'product_id' => $component2->id,
            'warehouse_id' => $warehouse->id,
            'unit_cost' => 10,
            'adjustment_type' => StoreInventoryAdjustment::ADJUSTMENT_TYPE_INCREASE
        ])->assertSuccessful();


        /** @var Product $kit */
        $kit = Product::factory()->create(['type' => Product::TYPE_KIT]);

        $kit->setKitComponents([
            ['id' => $component1->id, 'quantity' => 3],
            ['id' => $component2->id, 'quantity' => 5],
        ]);

        // Create sales order with backorder queue for kit
        $response = $this->postJson('/api/sales-orders', [
            'currency_code' => 'USD',
            'order_date' => now(),
            'order_status' => SalesOrder::STATUS_OPEN,
            'sales_order_lines' => [
                [
                    'product_id' => $kit->id,
                    'description' => $kit->name,
                    'amount' => 10,
                    'quantity' => 1,
                    'warehouse_id' => $warehouse->id,
                ]
            ]
        ])->assertSuccessful();

        $this->assertDatabaseCount('backorder_queues', 1);
        $this->assertDatabaseHas('backorder_queues', [
            'sales_order_line_id' => $response->json('data.item_info.0.sales_order_line_id'),
            'backordered_quantity' => 1,
            'released_quantity' => 0
        ]);

        // Create the assembly
        $this->postJson('/api/assemblies', [
           'action_date' => now(),
            'product_id'            => $kit->id,
            'warehouse_id'          => $warehouse->id,
            'quantity'              => 1,
            'action'                => StoreInventoryAssembly::ACTION_TYPE_ASSEMBLE,
        ])->assertSuccessful();

        // Fifo layers should be moved from components
        // and that of the kit should be created.
        $this->assertDatabaseCount('fifo_layers', 3);
        $this->assertDatabaseHas('fifo_layers', [
            'product_id' => $component1->id,
            'original_quantity' => 6,
            'fulfilled_quantity' => 3,
            'warehouse_id' => $warehouse->id,
        ]);
        $this->assertDatabaseHas('fifo_layers', [
            'product_id' => $component2->id,
            'original_quantity' => 5,
            'fulfilled_quantity' => 5,
            'warehouse_id' => $warehouse->id,
        ]);

        // Kit fifo layer should be used to release backorder queue
        $this->assertDatabaseHas('fifo_layers', [
            'product_id' => $kit->id,
            'original_quantity' => 1,
            'fulfilled_quantity' => 1,
            'warehouse_id' => $warehouse->id,
        ]);
        $this->assertDatabaseCount('backorder_queues', 1);
        $this->assertDatabaseHas('backorder_queues', [
            'sales_order_line_id' => $response->json('data.item_info.0.sales_order_line_id'),
            'backordered_quantity' => 1,
            'released_quantity' => 1
        ]);

        // Inventory movements should be created.
        $this->assertDatabaseCount('inventory_movements', 7);
        $this->assertDatabaseHas('inventory_movements', [
            'product_id' => $component1->id,
            'inventory_status' => InventoryMovement::INVENTORY_STATUS_ACTIVE,
            'quantity' => -3,
            'warehouse_id' => $warehouse->id,
            'type' => InventoryMovement::TYPE_ASSEMBLY
        ]);
        $this->assertDatabaseHas('inventory_movements', [
            'product_id' => $component2->id,
            'inventory_status' => InventoryMovement::INVENTORY_STATUS_ACTIVE,
            'quantity' => -5,
            'warehouse_id' => $warehouse->id,
            'type' => InventoryMovement::TYPE_ASSEMBLY
        ]);
        $this->assertDatabaseHas('inventory_movements', [
            'product_id' => $kit->id,
            'inventory_status' => InventoryMovement::INVENTORY_STATUS_ACTIVE,
            'quantity' => 1,
            'warehouse_id' => $warehouse->id,
            'type' => InventoryMovement::TYPE_ASSEMBLY
        ]);

    }

    /**
     * @throws KitWithMovementsCantBeChangedException
     * @throws ProductBundleException
     */
    public function test_it_does_not_pick_wrong_fifo_layer_when_assembling_kit(){

        Sanctum::actingAs(User::factory()->create());

        /** @var Warehouse $warehouse1 */
        $warehouse1 = Warehouse::factory()->create()->withDefaultLocation();
        $warehouse2 = Warehouse::factory()->create()->withDefaultLocation();

        /** @var Product $component */
        $component = Product::factory()->create(['type' => Product::TYPE_STANDARD]);

        // Add stock to components
        $this->postJson('/api/inventory-adjustments', [
            'adjustment_date' => now(),
            'quantity' => 3,
            'product_id' => $component->id,
            'warehouse_id' => $warehouse1->id,
            'unit_cost' => 10,
            'adjustment_type' => StoreInventoryAdjustment::ADJUSTMENT_TYPE_INCREASE
        ])->assertSuccessful();
        // Add stock to components
        $this->postJson('/api/inventory-adjustments', [
            'adjustment_date' => now(),
            'quantity' => 1,
            'product_id' => $component->id,
            'warehouse_id' => $warehouse2->id,
            'unit_cost' => 10,
            'adjustment_type' => StoreInventoryAdjustment::ADJUSTMENT_TYPE_INCREASE
        ])->assertSuccessful();

        /** @var Product $kit */
        $kit = Product::factory()->create(['type' => Product::TYPE_KIT]);

        $kit->setKitComponents([
            ['id' => $component->id, 'quantity' => 4],
        ]);

        // Create the assembly
        $response = $this->postJson(route('assemblies.store'), [
            'action_date' => now(),
            'product_id'            => $kit->id,
            'warehouse_id'          => $warehouse1->id,
            'quantity'              => 1,
            'action'                => StoreInventoryAssembly::ACTION_TYPE_ASSEMBLE,
        ])->assertBadRequest();

        $this->assertEquals("Not enough component inventory to assemble 4 kits of $component->sku.", $response->json('message'));
    }
}
