<?php

namespace Tests\Feature;

use App\Enums\FinancialLineClassificationEnum;
use App\Enums\FinancialLineProrationStrategyEnum;
use App\Exceptions\SalesOrder\InvalidProductWarehouseRouting;
use App\Http\Requests\StoreInventoryAdjustment;
use App\Models\Currency;
use App\Models\FifoLayer;
use App\Models\Product;
use App\Models\ReportingDailyFinancial;
use App\Models\SalesChannel;
use App\Models\SalesCredit;
use App\Models\SalesCreditLine;
use App\Models\SalesCreditReturn;
use App\Models\SalesCreditReturnLine;
use App\Models\SalesOrder;
use App\Models\SalesOrderFulfillment;
use App\Models\SalesOrderLine;
use App\Models\Setting;
use App\Models\Supplier;
use App\Models\SupplierPricingTier;
use App\Models\SupplierProduct;
use App\Models\SupplierProductPricing;
use App\Models\User;
use App\Models\Warehouse;
use App\Repositories\FinancialLineRepository;
use App\Repositories\InventoryForecastRepository;
use App\Repositories\ProductRepository;
use App\Repositories\SalesOrderLineFinancialsRepository;
use App\Repositories\SettingRepository;
use App\Services\FinancialManagement\DailyFinancialManager;
use App\Services\FinancialManagement\SalesOrderLineFinancialManager;
use App\Services\InventoryForecasting\ForecastManager;
use App\Services\InventoryManagement\InventoryManager;
use App\Services\PurchaseOrder\PurchaseOrderBuilder\PurchaseOrderBuilderRepository;
use App\Services\SalesOrder\SalesOrderManager;
use Carbon\Carbon;
use Exception;
use Illuminate\Foundation\Testing\WithFaker;
use Illuminate\Support\Facades\Event;
use Illuminate\Support\Facades\Queue;
use Illuminate\Support\Facades\Redis;
use Laravel\Sanctum\Sanctum;
use Plannr\Laravel\FastRefreshDatabase\Traits\FastRefreshDatabase;
use Tests\TestCase;
use Throwable;

class FinancialReportingTest extends TestCase
{
    use FastRefreshDatabase;
    use WithFaker;

    public function test_it_can_successfully_run_calculate_command(): void
    {
        $this->artisan('sku:financials:calculate')->assertSuccessful();
    }

    public function test_it_cannot_access_reporting_orders_as_non_power_user(): void
    {
        Sanctum::actingAs(User::factory()->create(['is_power_user' => 0]));

        $this->getJson('/api/financials/summary-by-period?period=day')->assertForbidden();
    }

    public function test_it_can_request_reporting_orders_as_power_user(): void
    {
        Sanctum::actingAs(User::factory()->create(['is_power_user' => 1]));

        $this->getJson('/api/financials/summary-by-period?period=day')->assertSuccessful();
    }

    public function test_it_can_request_sales_order_proforma_financials(): void
    {
        Sanctum::actingAs(User::factory()->create());

        Queue::fake();

        $salesOrder = SalesOrder::factory()->hasSalesOrderLines()->create();

        $this->getJson('/api/sales-orders/'.$salesOrder->id.'/financials')->assertSuccessful();
    }

    public function test_it_can_invalidate_caches_updates(): void
    {
        $salesOrder = SalesOrder::factory()->hasSalesOrderLines()->create();

        Queue::fake();

        (new SalesOrderLineFinancialManager())->calculate();

        $this->assertEquals(0, Redis::scard(SalesOrderLine::INVALID_FINANCIALS_KEY));

        // Sales Order Line Update
        $salesOrderLine = $salesOrder->salesOrderLines()->first();
        $salesOrderLine->quantity += 1;
        $salesOrderLine->update();

        $this->assertContains($salesOrderLine->id, Redis::smembers(SalesOrderLine::INVALID_FINANCIALS_KEY));

        (new SalesOrderLineFinancialManager())->calculate();

        $this->assertNotContains($salesOrderLine->id, Redis::smembers(SalesOrderLine::INVALID_FINANCIALS_KEY));

        // Sales Order Update
        $salesOrder->sales_order_number = 'new';
        $salesOrder->update();

        $this->assertContains($salesOrderLine->id, Redis::smembers(SalesOrderLine::INVALID_FINANCIALS_KEY));

        (new SalesOrderLineFinancialManager())->calculate();

        $this->assertNotContains($salesOrderLine->id, Redis::smembers(SalesOrderLine::INVALID_FINANCIALS_KEY));

        // Product Update
        $product = $salesOrder->salesOrderLines()->first()->product;
        $product->unit_cost += 1;
        $product->update();

        $this->assertContains($salesOrderLine->id, Redis::smembers(SalesOrderLine::INVALID_FINANCIALS_KEY));

        (new SalesOrderLineFinancialManager())->calculate();

        $this->assertNotContains($salesOrderLine->id, Redis::smembers(SalesOrderLine::INVALID_FINANCIALS_KEY));
    }

    /**
     * @throws Throwable
     * @throws InvalidProductWarehouseRouting
     */
    public function test_it_stores_financials_in_tenant_currency(): void
    {
        /** @var Currency $currency */
        $currency = Currency::query()->updateOrCreate(
            ['code' => 'CAD'],
            [
                'code' => 'CAD',
                'conversion' => 0.66,
            ]
        );

        /** @var Product $product */
        $product = Product::factory()->create();

        $salesOrderPayload = [
            'is_tax_included' => false,
            'currency_code' => $currency->code,
            'sales_channel_id' => 1,
            'order_date' => now(),
            'order_status' => SalesOrder::STATUS_DRAFT,
            'sales_order_lines' => [
                [
                    'quantity' => 1,
                    'amount' => 3.00,
                    'sku' => $product->sku,
                    'description' => $product->name,
                ],
            ],
        ];

        $salesOrderManager = app(SalesOrderManager::class);
        $salesOrder = $salesOrderManager->createOrder($salesOrderPayload);

        /** @var SalesOrderLine $salesOrderLine */
        $salesOrderLine = $salesOrder->salesOrderLines()->first();

        $salesOrderLineFinancial = $salesOrderLine->salesOrderLineFinancial;

        (new SalesOrderLineFinancialManager())->calculate();
        $salesOrderLineFinancial->refresh();

        $this->assertEquals(
            $currency->conversion * 3.00,
            $salesOrderLineFinancial->total_revenue,
        );
    }

    /**
     * @throws Throwable
     * @throws InvalidProductWarehouseRouting
     */
    public function test_it_can_calculate_correct_profit(): void
    {
        Queue::fake();
        Setting::query()->where('key', Setting::KEY_INVENTORY_START_DATE)->update(['value' => '1900-01-01']);
        $productWithActiveFifoLayerAndFulfilled =
            Product::factory()->create([
                'name' => 'Product with active fifo layer and fulfilled',
            ]);
        $productWithUsedFifoLayerAndDefaultShippingCost = Product::factory()->create([
            'name' => 'Product with used fifo layer and default shipping cost',
            'proforma_shipping_cost' => 1.10,
        ]);

        $productWithSupplierCostAndLandedCost =
            Product::factory()->has(
                SupplierProduct::factory()->has(
                    SupplierProductPricing::factory(null, [
                        'price' => 4.99,
                        'supplier_pricing_tier_id' => SupplierPricingTier::default()->id,
                    ])
                )
            )->create([
                'name' => 'Product with Supplier Cost and Landed Cost',
                'proforma_landed_cost_percentage' => 0.1,
            ]);

        $productWithUnitCostAndDefaultMarketplaceCost =
            Product::factory(null, [
                'name' => 'Product with unit cost and default marketplace cost',
                'unit_cost' => 3.85,
                'proforma_marketplace_cost_percentage' => 12,
            ])->create();

        Sanctum::actingAs(User::factory()->create());

        $warehouse_id = Warehouse::factory()->create()->withDefaultLocation()->id;

        $this->postJson('/api/inventory-adjustments', [
            'adjustment_date' => $this->faker->date(),
            'product_id' => $productWithActiveFifoLayerAndFulfilled->id,
            'warehouse_id' => $warehouse_id,
            'unit_cost' => 2.50,
            'quantity' => 50,
            'adjustment_type' => StoreInventoryAdjustment::ADJUSTMENT_TYPE_INCREASE,
        ]);

        $this->postJson('/api/inventory-adjustments', [
            'adjustment_date' => $this->faker->date(),
            'product_id' => $productWithUsedFifoLayerAndDefaultShippingCost->id,
            'warehouse_id' => $warehouse_id,
            'unit_cost' => 3.10,
            'quantity' => 10,
            'adjustment_type' => StoreInventoryAdjustment::ADJUSTMENT_TYPE_INCREASE,
        ]);

        $this->postJson('/api/inventory-adjustments', [
            'adjustment_date' => $this->faker->date(),
            'product_id' => $productWithUsedFifoLayerAndDefaultShippingCost->id,
            'warehouse_id' => $warehouse_id,
            'quantity' => 10,
            'adjustment_type' => StoreInventoryAdjustment::ADJUSTMENT_TYPE_DECREASE,
        ]);

        $integrationInstance = SalesChannel::where('id', 1)->first()->integrationInstance;

        $integrationSettings = is_array($integrationInstance->integration_settings) ?
        $integrationInstance->integration_settings : [];
        $integrationSettings['proforma_payment_cost_percentage'] = 0.022;
        $integrationInstance->integration_settings = $integrationSettings;
        $integrationInstance->update();

        $salesOrderPayload = [
            'is_tax_included' => false,
            'currency_code' => 'USD',
            'sales_channel_id' => 1,
            'order_date' => now(),
            'store_id' => 1,
            'order_status' => SalesOrder::STATUS_DRAFT,
            'sales_order_lines' => [
                [
                    'quantity' => 1,
                    'description' => $productWithActiveFifoLayerAndFulfilled->name,
                    'amount' => 3.00, //subtotal 3.00
                    'product_id' => $productWithActiveFifoLayerAndFulfilled->id,
                    'warehouse_id' => $warehouse_id,
                ],
                [
                    'quantity' => 3,
                    'description' => $productWithUsedFifoLayerAndDefaultShippingCost->name,
                    'amount' => 4.00, //subtotal 12.00
                    'product_id' => $productWithUsedFifoLayerAndDefaultShippingCost->id,
                    'warehouse_id' => $warehouse_id,
                ],
                [
                    'quantity' => 2,
                    'description' => $productWithUnitCostAndDefaultMarketplaceCost->name,
                    'amount' => 5.00, //subtotal 10.00
                    'product_id' => $productWithUnitCostAndDefaultMarketplaceCost->id,
                    'warehouse_id' => $warehouse_id,
                ],
                [
                    'quantity' => 1,
                    'description' => $productWithSupplierCostAndLandedCost->name,
                    'amount' => 6.50, //subtotal 6.50
                    'product_id' => $productWithSupplierCostAndLandedCost->id,
                    'warehouse_id' => $warehouse_id,
                ],
            ],
            'financial_lines' => [
                [
                    'financial_line_type_id' => app(FinancialLineRepository::class)->getOrCreateFinancialLineType('Shipping', FinancialLineClassificationEnum::REVENUE)->id,
                    'description' => 'Shipping charge, product allocated',
                    'quantity' => 1,
                    'amount' => 10.00,
                    'allocate_to_products' => true,
                ],
                [
                    'financial_line_type_id' => app(FinancialLineRepository::class)->getOrCreateFinancialLineType('Shipping', FinancialLineClassificationEnum::REVENUE)->id,
                    'description' => 'Shipping charge, not product allocated',
                    'quantity' => 1,
                    'amount' => 5.00,
                    'allocate_to_products' => false,
                ],
            ],
        ];

        if ($paymentCostPercentage = @$integrationInstance->integration_settings['proforma_payment_cost_percentage']) {
            $salesOrderPayload['financial_lines'][] = [
                'financial_line_type_id' => app(FinancialLineRepository::class)
                    ->getOrCreateFinancialLineType('Payment Fee', FinancialLineClassificationEnum::COST)
                    ->id,
                'description' => 'Payment Fee',
                'amount' => 31.50 * $paymentCostPercentage,
                'proration_strategy' => FinancialLineProrationStrategyEnum::REVENUE_BASED,
                'allocate_to_products' => true,
            ];
        }

        //        $salesOrderPayload['financial_lines'][] = [
        //            'financial_line_type_id'        => app(FinancialLineRepository::class)
        //                ->getOrCreateFinancialLineType('Marketplace Fee', FinancialLineClassificationEnum::COST)
        //                ->id,
        //            'amount'                        => 5.00 * 0.12,
        //            'proration_strategy'            => FinancialLineProrationStrategyEnum::REVENUE_BASED,
        //            'allocate_to_products'          => true,
        //        ];

        $salesOrderManager = app(SalesOrderManager::class);

        $salesOrder = $salesOrderManager->createOrder($salesOrderPayload);

        $this->assertEquals(
            31.50 * $paymentCostPercentage,
            app(FinancialLineRepository::class)
                ->getSalesOrderFinancialLineTotal($salesOrder, 'Payment Fee', FinancialLineClassificationEnum::COST)
        );

        /*
         * TODO: We are missing the functionality to allocate a financial line cost to a specific product
         *  This is needed for both marketplace and landed cost
         */
        //        $this->assertEquals(
        //            5.00 * 0.12,
        //            app(FinancialLineRepository::class)
        //                ->getSalesOrderFinancialLineTotal($salesOrder, 'Marketplace Fee', FinancialLineClassificationEnum::COST)
        //        );

        //$salesOrder->approve();
        //$salesOrder->save();

        $this->postJson('/api/sales-orders/'.$salesOrder->id.'/fulfill', [
            'fulfillment_type' => SalesOrderFulfillment::TYPE_MANUAL,
            'fulfilled_at' => SettingRepository::getInventoryStartDate()->addDay(),
            //TODO: This is not working on dev
            //'cost' => 1.01,
            'status' => SalesOrderFulfillment::STATUS_FULFILLED,
            'fulfillment_lines' => [
                [
                    'sales_order_line_id' => ($line = $salesOrder->salesOrderLines
                        ->where('product_id', $productWithActiveFifoLayerAndFulfilled->id)
                        ->firstOrFail())
                                            ->id,
                    'quantity' => $line->quantity,
                ],
            ],
        ]);

        Queue::fake();

        (new SalesOrderLineFinancialManager())->calculate();

        $salesOrder = $salesOrder->refresh();

        $financials = array_fill_keys(
            [
                'Qty',
                'Revenue',
                'Costs',
                'Profit',
            ],
            0
        );

        /** @var SalesOrderLine $salesOrderLine */
        foreach ($salesOrder->salesOrderLines as $salesOrderLine) {
            $salesOrderLineFinancial = $salesOrderLine->salesOrderLineFinancial;
            $financials['Qty'] += $salesOrderLine->quantity;
            $financials['Revenue'] += $salesOrderLineFinancial->total_revenue;
            $financials['Costs'] += $salesOrderLineFinancial->total_cost;
            $financials['Profit'] += $salesOrderLineFinancial->profit;

            switch ($salesOrderLine->description) {
                case 'Product with active fifo layer and fulfilled':
                    // 3.00 + 0.9524 - 2.50 - 0.066 = 1.3864
                    $this->assertEquals(3.9524, round($salesOrderLineFinancial->total_revenue, 4));
                    $this->assertEquals(0.9524, round($salesOrderLineFinancial->revenue_allocated, 4));
                    $this->assertEquals(2.5000, round($salesOrderLineFinancial->cogs, 4));
                    $this->assertEquals(.022 * 3.00, round($salesOrderLineFinancial->cost_allocated, 4));
                    $this->assertEquals(2.5660, round($salesOrderLineFinancial->total_cost, 4));
                    $this->assertEquals(1.3864, round($salesOrderLineFinancial->profit, 4));
                    break;
                case 'Product with used fifo layer and default shipping cost':
                    // 12.00 + 3.8095 - (3.10 * 3) - (0.022 * 12.00) = 6.2455
                    $this->assertEquals(15.8095, round($salesOrderLineFinancial->total_revenue, 4));
                    $this->assertEquals(3.10 * 3, round($salesOrderLineFinancial->cogs, 4));
                    $this->assertEquals(.022 * 12.00, round($salesOrderLineFinancial->cost_allocated, 4));
                    $this->assertEquals(9.564, round($salesOrderLineFinancial->total_cost, 4));
                    $this->assertEquals(6.2455, round($salesOrderLineFinancial->profit, 4));
                    break;
                case 'Product with unit cost and default marketplace cost':
                    // 10.00 + 3.1746 - (3.85 * 2) - (0.022 * 10.00) = 5.2546
                    $this->assertEquals(13.1746, round($salesOrderLineFinancial->total_revenue, 4));
                    $this->assertEquals(3.85 * 2, round($salesOrderLineFinancial->cogs, 4));
                    $this->assertEquals(round(.022 * 10.00, 4), round($salesOrderLineFinancial->cost_allocated, 4));
                    $this->assertEquals(7.9200, round($salesOrderLineFinancial->total_cost, 4));
                    $this->assertEquals(5.2546, round($salesOrderLineFinancial->profit, 4));
                    break;
                case 'Product with Supplier Cost and Landed Cost':
                    // 6.50 + 2.0635 - 4.99 - (0.022 * 6.50) = 3.4305
                    $this->assertEquals(8.5635, round($salesOrderLineFinancial->total_revenue, 4));
                    $this->assertEquals(4.99, round($salesOrderLineFinancial->cogs, 4));
                    $this->assertEquals(round(.022 * 6.50, 4), round($salesOrderLineFinancial->cost_allocated, 4));
                    $this->assertEquals(5.133, round($salesOrderLineFinancial->total_cost, 4));
                    $this->assertEquals(3.4305, round($salesOrderLineFinancial->profit, 4));
                    break;
                case 'Shipping charge, product allocated':
                    $this->assertEquals(0.000, round($salesOrderLineFinancial->total_revenue, 4));
                    $this->assertEquals(0, round($salesOrderLineFinancial->profit, 2));
                    break;
                case 'Shipping charge, not product allocated':
                    // 5.00 - (0.022 * 5.00) = 4.89
                    $this->assertEquals(5.000, round($salesOrderLineFinancial->total_revenue, 4));
                    $this->assertEquals(round(.022 * 5.00, 4), round($salesOrderLineFinancial->cost_allocated, 4));
                    $this->assertEquals(4.8900, round($salesOrderLineFinancial->profit, 4));
                    break;
                default:
                    throw new Exception("Unexpected line description: {$salesOrderLine->description}");
            }
        }

        $financials['Revenue'] += 5.000 * (1 - .022); // non allocated shipping revenue

        $this->assertEquals((3.9524 + 15.8095 + 13.1746 + 8.5635) + 4.8900, round($financials['Revenue'], 3)); // 41.50 + 4.8900 = 46.39
        $this->assertEquals(round(2.5660 + 9.564 + 7.9200 + 5.133, 3), round($financials['Costs'], 3)); // 25.183
        $this->assertEquals((1.3864 + 6.2455 + 5.2546 + 3.4305), round($financials['Profit'], 3)); // 16.317
    }

    public function test_it_can_get_average_daily_sales_for_purchase_order_builder(): void
    {
        $totalQtySold = 3;
        $totalDays = 30;

        $supplier = Supplier::factory()->withWarehouse()->create();

        $product = Product::factory()->hasSupplierProducts(1, [
            'supplier_id' => $supplier->id,
        ])->create();

        SalesOrder::factory($totalQtySold)->hasSalesOrderLines(1, [
            'product_id' => $product->id,
            'quantity' => 1,
        ])->create([
            'order_date' => $this->faker->dateTimeBetween(Carbon::now()->subDays($totalDays), Carbon::now()),
        ]);

        Queue::fake();

        (new SalesOrderLineFinancialManager())->calculate();

        $purchaseOrderBuilderRepository = app(PurchaseOrderBuilderRepository::class);

        $dailyVelocity = $purchaseOrderBuilderRepository->getAverageDailySales(
            $product,
            Carbon::now()->subDays($totalDays),
            Carbon::now(),
            []
        );

        $this->assertEquals($totalQtySold / $totalDays, $dailyVelocity);
    }

    /**
     * @throws Exception
     */
    public function test_it_can_make_historical_sales_for_sku(): void
    {
        $totalQtySold = 3;
        $totalDays = 30;

        $supplier = Supplier::factory()->withWarehouse()->create();

        $product = Product::factory()->hasSupplierProducts(1, [
            'supplier_id' => $supplier->id,
        ])->create();

        SalesOrder::factory($totalQtySold)->hasSalesOrderLines(1, [
            'product_id' => $product->id,
            'quantity' => 1,
        ])->create([
            'order_date' => $this->faker->dateTimeBetween(Carbon::now()->subDays($totalDays), Carbon::now()),
        ]);

        Queue::fake();

        (new SalesOrderLineFinancialManager())->calculate();

        $forecastManager = new ForecastManager(app(InventoryForecastRepository::class));

        $historicalData = $forecastManager->makeHistoricalSalesForSku($product);

        $this->assertEquals($totalQtySold, $historicalData->getData()[0]['quantity']);
    }

    public function test_it_can_calculate_product_proforma_financials(): void
    {

        $numOrders = 3;
        $qtyEach = 3;
        $amount = 8;
        $unitCost = 5;

        /** @var Product $product */
        $product = Product::factory()->create(['unit_cost' => $unitCost]);

        SalesOrder::factory($numOrders)->hasSalesOrderLines(1, [
            'product_id' => $product->id,
            'quantity' => $qtyEach,
            'amount' => $amount,
        ])->create();

        Queue::fake();

        (new SalesOrderLineFinancialManager())->calculate();

        (new DailyFinancialManager())->calculate();

        $this->assertEquals([
            'quantity' => $qtyEach * $numOrders,
            'num_orders' => $numOrders,
            'reportable_id' => $product->id,
            'reportable_type' => Product::class,
            'total_revenue' => (float) ($qtyEach * $numOrders * $amount),
            'total_cost' => (float) ($qtyEach * $numOrders * $unitCost),
        ],
            ReportingDailyFinancial::query()
                ->selectRaw('
            sum(quantity) as quantity,
            sum(num_orders) as num_orders,
            reportable_id,
            reportable_type,
            sum(total_revenue) as total_revenue,
            sum(total_cost) as total_cost
        ')->groupBy('reportable_id', 'reportable_type')
                ->first()->toArray());
    }

    public function test_it_can_calculate_correct_supplier_cost(): void
    {
        Queue::fake();
        Sanctum::actingAs(User::factory()->create());

        Setting::where('key', Setting::KEY_INVENTORY_START_DATE)->update(['value' => '1900-01-01']);
        $supplier = Supplier::factory()->withWarehouse()->create();

        $productWithActiveFifoLayerAndFulfilled = Product::factory()->create([
            'name' => 'Product with active fifo layer and fulfilled',
        ]);
        $warehouse_id = Warehouse::factory()->create()->withDefaultLocation()->id;

        $this->postJson('/api/inventory-adjustments', [
            'adjustment_date' => $this->faker->date(),
            'product_id' => $productWithActiveFifoLayerAndFulfilled->id,
            'warehouse_id' => $warehouse_id,
            'unit_cost' => 2.50,
            'quantity' => 50,
            'adjustment_type' => StoreInventoryAdjustment::ADJUSTMENT_TYPE_INCREASE,
        ]);
        $cost = (new ProductRepository())->getCostForSupplier($productWithActiveFifoLayerAndFulfilled, $warehouse_id,
            $supplier->id);
        $this->assertEquals(2.50, $cost);

        $productWithUsedFifoLayerAndDefaultShippingCost = Product::factory()->create([
            'name' => 'Product with used fifo layer and default shipping cost',
            'proforma_shipping_cost' => 1.10,
        ]);

        $this->postJson('/api/inventory-adjustments', [
            'adjustment_date' => $this->faker->date(),
            'product_id' => $productWithUsedFifoLayerAndDefaultShippingCost->id,
            'warehouse_id' => $warehouse_id,
            'unit_cost' => 3.10,
            'quantity' => 10,
            'adjustment_type' => StoreInventoryAdjustment::ADJUSTMENT_TYPE_INCREASE,
        ]);

        $this->postJson('/api/inventory-adjustments', [
            'adjustment_date' => $this->faker->date(),
            'product_id' => $productWithUsedFifoLayerAndDefaultShippingCost->id,
            'warehouse_id' => $warehouse_id,
            'quantity' => 10,
            'adjustment_type' => StoreInventoryAdjustment::ADJUSTMENT_TYPE_DECREASE,
        ]);
        $productWithUsedFifoLayerAndDefaultShippingCost->updateAverageCost();
        $cost = (new ProductRepository())->getCostForSupplier($productWithUsedFifoLayerAndDefaultShippingCost,
            $warehouse_id, $supplier->id);
        $this->assertEquals(3.10, $cost);

        $productWithSupplierCostAndLandedCost
            = Product::factory()->has(
                SupplierProduct::factory()->has(
                    SupplierProductPricing::factory(null, [
                        'price' => 4.99,
                        'supplier_pricing_tier_id' => SupplierPricingTier::default()->id,
                    ])
                )
            )->create([
                'name' => 'Product with Supplier Cost and Landed Cost',
                'proforma_landed_cost_percentage' => 0.1,
            ]);

        $supplier = $productWithSupplierCostAndLandedCost->suppliers->first();
        $tiers = SupplierPricingTier::with([])->where('is_default', true)->pluck('id')->toArray();
        $supplier->attachPricingTiers($tiers);

        $cost = (new ProductRepository())->getCostForSupplier($productWithSupplierCostAndLandedCost, $warehouse_id,
            $supplier->id);
        $this->assertEquals(4.99, $cost);

        $productWithUnitCostAndDefaultMarketplaceCost
            = Product::factory(null, [
                'name' => 'Product with unit cost and default marketplace cost',
                'unit_cost' => 3.85,
                'proforma_marketplace_cost_percentage' => 12,
            ])->create();

        $cost = (new ProductRepository())->getCostForSupplier($productWithUnitCostAndDefaultMarketplaceCost,
            $warehouse_id, $supplier->id);
        $this->assertEquals(3.85, $cost);
    }

    public function test_it_can_delete_product_financials_summary_when_sales_order_deleted(): void
    {
        Queue::fake();

        $qtySold = 3;
        $amount = 8;
        $unitCost = 5;

        /** @var Supplier $supplier */
        $supplier = Supplier::factory()->withWarehouse()->create();

        /** @var Product $product */
        $product = Product::factory()->create(['unit_cost' => $unitCost]);

        /** @var SalesOrder $salesOrder */
        $salesOrder = SalesOrder::factory()->hasSalesOrderLines(1, [
            'product_id' => $product->id,
            'quantity' => $qtySold,
            'amount' => $amount,
        ])->create();

        $salesOrderLine = $salesOrder->salesOrderLines()->first();

        (new SalesOrderLineFinancialManager())->calculate();

        (new DailyFinancialManager())->calculate();

        $this->assertEquals([
            'quantity' => $qtySold,
            'num_orders' => 1,
            'reportable_id' => $product->id,
            'reportable_type' => Product::class,
            'total_revenue' => (float) ($qtySold * $amount),
            'total_cost' => (float) ($qtySold * $unitCost),
        ],
            ReportingDailyFinancial::query()
                ->selectRaw('
                sum(quantity) as quantity,
                sum(num_orders) as num_orders,
                reportable_id,
                reportable_type,
                sum(total_revenue) as total_revenue,
                sum(total_cost) as total_cost
            ')->groupBy('reportable_id', 'reportable_type')
                ->first()->toArray());

        $salesOrderLine->delete();

        (new DailyFinancialManager())->calculate();

        $this->assertDatabaseCount((new ReportingDailyFinancial)->getTable(), 0);
    }

    public function test_it_does_not_include_financials_for_draft_sales_orders(): void
    {
        Queue::fake();

        SalesOrder::factory()->hasSalesOrderLines()->create([
            'order_status' => SalesOrder::STATUS_DRAFT,
            'order_date' => Carbon::now(),
        ]);

        (new SalesOrderLineFinancialManager())->calculate();

        $salesOrderLineFinancialsRepository = app(SalesOrderLineFinancialsRepository::class);

        $this->assertEquals(
            0,
            $salesOrderLineFinancialsRepository
                ->getSummaryByPeriod('day', 1)->sum('total_revenue')
        );

        $this->assertEquals(
            0,
            $salesOrderLineFinancialsRepository->getTopProducts('revenue', 1)->sum('total_revenue')
        );
    }

    public function test_it_does_not_include_financials_for_canceled_sales_orders(): void
    {
        Queue::fake();

        SalesOrder::factory()->hasSalesOrderLines()->create([
            'order_status' => SalesOrder::STATUS_CLOSED,
            'canceled_at' => Carbon::now(),
            'order_date' => Carbon::now(),
        ]);

        (new SalesOrderLineFinancialManager())->calculate();

        $salesOrderLineFinancialsRepository = app(SalesOrderLineFinancialsRepository::class);

        $this->assertEquals(
            0,
            $salesOrderLineFinancialsRepository
                ->getSummaryByPeriod('day', 1)->sum('total_revenue')
        );

        $this->assertEquals(
            0,
            $salesOrderLineFinancialsRepository->getTopProducts('revenue', 1)->sum('total_revenue')
        );
    }

    /**
     * @throws Exception
     */
    public function test_it_can_calculate_proforma_sales_credit_and_returned_unit_cost(): void
    {
        // TODO: Update test_it_can_calculate_correct_profit to include a scenario for credits and returns
        Event::fake();

        /*
         * Sales order line qty 4
         * Sales credit line qty 2
         * Sales credit return line qty 1
         */

        Queue::fake();
        $salesOrderLineFinancialManager = new SalesOrderLineFinancialManager();

        /** @var Warehouse $warehouse */
        $warehouse = Warehouse::factory()->create()->withDefaultLocation();

        $unitCost = 5;

        /** @var Product $product */
        $product = Product::factory()->create(['unit_cost' => $unitCost]);

        SalesOrder::factory()->hasSalesOrderLines()->create([
            'order_status' => SalesOrder::STATUS_OPEN,
            'order_date' => Carbon::now(),
        ]);

        SalesCredit::factory()->hasSalesCreditLines()->create([
            'credit_status' => SalesCredit::CREDIT_STATUS_OPEN,
            'credit_date' => Carbon::now(),
        ]);

        /** @var SalesOrderLine $salesOrderLine */
        $salesOrderLine = SalesOrderLine::query()->first();

        $salesOrderLine->update([
            'quantity' => 4,
            'product_id' => $product->id,
        ]);
        $salesOrderLine->refresh();

        /** @var SalesCreditLine $salesCreditLine */
        $salesCreditLine = SalesCreditLine::query()->first();

        $percentCredited = 0.5;
        $salesCreditLine->update([
            'sales_order_line_id' => $salesOrderLine->id,
            'quantity' => $percentCredited * $salesOrderLine->quantity,
            'product_id' => $product->id,
        ]);
        $salesCreditLine->refresh();

        $this->assertEquals(
            $percentCredited * $salesOrderLineFinancialManager->getRevenue($salesOrderLine),
            $salesOrderLineFinancialManager
                ->getProformaSalesCredit($salesOrderLine)
        );

        SalesCreditReturn::factory()->hasSalesCreditReturnLines()->create([
            'status' => SalesCreditReturn::STATUS_CLOSED,
            'sales_credit_id' => $salesCreditLine->salesCredit->id,
            'received_at' => Carbon::now(),
            'warehouse_id' => $warehouse->id,
        ]);

        /** @var SalesCreditReturnLine $salesCreditReturnLine */
        $salesCreditReturnLine = SalesCreditReturnLine::query()->first();

        $percentReturnedOfLine = 0.5;
        $salesCreditReturnLine->update([
            'sales_credit_line_id' => $salesCreditLine->id,
            'quantity' => $salesCreditLine->quantity * $percentReturnedOfLine,
            'product_id' => $product->id,
        ]);
        $salesCreditReturnLine->refresh();

        $inventoryManger = (new InventoryManager($warehouse->id, $product));
        $inventoryManger->addToStock($salesCreditReturnLine->quantity, $salesCreditReturnLine);

        $this->assertDatabaseHas((new FifoLayer())->getTable(), [
            'product_id' => $product->id,
            'warehouse_id' => $warehouse->id,
            'original_quantity' => $salesCreditReturnLine->quantity,
            'total_cost' => $salesCreditReturnLine->quantity * $unitCost,
        ]);

        /** @var FifoLayer $fifoLayer */
        $fifoLayer = FifoLayer::query()->first();

        $this->assertEquals(
            $fifoLayer->avg_cost * $salesCreditReturnLine->quantity,
            $salesOrderLineFinancialManager
                ->getProformaReturnedUnitCost($salesOrderLine)
        );
    }

    public function test_sales_order_line_financial_gets_seeded_with_sales_order_line(): void
    {
        /** @var SalesOrderLine $salesOrderLine */
        $salesOrderLine = SalesOrderLine::factory()->create();

        $salesOrderLine->refresh();

        $this->assertTrue(! is_null($salesOrderLine->salesOrderLineFinancial));
    }

    public function test_it_can_recalculate_sales_order_line_financials(): void
    {
        $salesOrder = SalesOrder::factory()->hasSalesOrderLines(1, [
            'quantity' => 1,
            'amount' => 3.00,
        ])->create();

        $salesOrderLine = $salesOrder->salesOrderLines()->first();

        $salesOrderLine->update([
            'quantity' => 2,
            'amount' => 6.00,
        ]);

        (new SalesOrderLineFinancialManager())->recalculateLines(collect([$salesOrderLine]));

        $this->assertEquals(12.00, $salesOrderLine->salesOrderLineFinancial->total_revenue);
    }
}
