<?php

namespace Tests\Feature;

use App\Enums\FinancialAlertNotificationConditionEnum;
use App\Enums\FinancialAlertTypeEnum;
use App\Enums\FinancialLineClassificationEnum;
use App\Enums\FinancialLineProrationStrategyEnum;
use App\Jobs\ProcessFinancialAlertsJob;
use App\Managers\FinancialAlertManager;
use App\Models\FinancialAlert;
use App\Models\FinancialLine;
use App\Models\FinancialLineType;
use App\Models\SalesOrder;
use App\Models\SalesOrderLineFinancial;
use App\Models\Setting;
use App\Notifications\FinancialAlertEmailNotification;
use App\Notifications\FinancialAlertSlackNotification;
use App\Repositories\SettingRepository;
use App\Services\FinancialManagement\SalesOrderLineFinancialManager;
use Exception;
use Illuminate\Support\Facades\Notification;
use Illuminate\Support\Facades\Queue;
use Plannr\Laravel\FastRefreshDatabase\Traits\FastRefreshDatabase;
use Tests\TestCase;

class FinancialAlertManagerTest extends TestCase
{
    use FastRefreshDatabase;

    private FinancialAlertManager $manager;

    protected function setUp(): void
    {
        parent::setUp();

        Queue::fake();
        Notification::fake();

        app(SettingRepository::class)->set(Setting::KEY_FINANCIAL_ALERTS_ENABLED, true);
        app(SettingRepository::class)->set(Setting::KEY_FINANCIAL_ALERTS_LOW_MARGIN_THRESHOLD, 10);

        $this->manager = app(FinancialAlertManager::class);
    }

    private function createFinancialAlert(): SalesOrder
    {
        $salesOrder = SalesOrder::factory()->hasSalesOrderLines(1, [
            'quantity' => 1,
            'amount' => 3.00,
        ])->create();

        $financialLineType = FinancialLineType::factory()->create([
            'name' => 'Shipping Cost',
            'classification' => FinancialLineClassificationEnum::COST,
        ]);

        FinancialLine::factory()->create([
            'financial_line_type_id' => $financialLineType->id,
            'sales_order_id' => $salesOrder->id,
            'description' => 'Shipping Cost',
            'quantity' => 1,
            'amount' => 4.00,
            'proration_strategy' => FinancialLineProrationStrategyEnum::REVENUE_BASED,
            'allocate_to_products' => 1,
        ]);

        (new SalesOrderLineFinancialManager())->calculate();

        $this->manager->createFinancialAlerts();

        return $salesOrder;
    }

    public function test_it_can_create_financial_alerts(): void
    {
        $salesOrder = $this->createFinancialAlert();

        Queue::assertPushed(ProcessFinancialAlertsJob::class);

        $this->assertDatabaseHas(SalesOrderLineFinancial::class, [
            'profit' => -1.00,
        ]);

        $this->assertDatabaseHas(FinancialAlert::class, [
            'sales_order_line_financial_id' => $salesOrder->salesOrderLines()->first()->salesOrderLineFinancial->id,
            'alert_type' => FinancialAlertTypeEnum::LOSS,
        ]);

        FinancialLine::first()->delete();

        (new SalesOrderLineFinancialManager())->calculate();
        $this->manager->createFinancialAlerts();

        $this->assertDatabaseEmpty(FinancialAlert::class);
    }

    /**
     * @throws Exception
     */
    public function test_it_can_send_slack_summary_notification(): void
    {
        $this->createFinancialAlert();

        app(SettingRepository::class)->set(Setting::KEY_FINANCIAL_ALERTS_NOTIFICATION_TYPE, Setting::KEY_FINANCIAL_ALERTS_NOTIFICATION_TYPE_SLACK);
        app(SettingRepository::class)->set(Setting::KEY_FINANCIAL_ALERTS_NOTIFICATION_DESTINATION, 'https://hooks.slack.com/services/...');

        app(SettingRepository::class)->set(Setting::KEY_FINANCIAL_ALERTS_NOTIFICATION_CONDITIONS,
            json_encode([FinancialAlertNotificationConditionEnum::WEEKLY_SUMMARY->value, FinancialAlertNotificationConditionEnum::DAILY_SUMMARY->value])
        );

        // daily summary
        $this->manager->sendSummaryAlertNotification(FinancialAlertNotificationConditionEnum::DAILY_SUMMARY);
        Notification::assertSentOnDemand(FinancialAlertSlackNotification::class);
        // weekly summary
        $this->manager->sendSummaryAlertNotification(FinancialAlertNotificationConditionEnum::WEEKLY_SUMMARY);
        Notification::assertSentOnDemand(FinancialAlertSlackNotification::class);
    }

    /**
     * @throws Exception
     */
    public function test_it_can_send_email_summary_notification(): void
    {
        $this->createFinancialAlert();

        app(SettingRepository::class)->set(Setting::KEY_FINANCIAL_ALERTS_NOTIFICATION_TYPE, Setting::KEY_FINANCIAL_ALERTS_NOTIFICATION_TYPE_EMAIL);
        app(SettingRepository::class)->set(Setting::KEY_FINANCIAL_ALERTS_NOTIFICATION_DESTINATION, 'me@sku.com');

        app(SettingRepository::class)->set(Setting::KEY_FINANCIAL_ALERTS_NOTIFICATION_CONDITIONS,
            json_encode([FinancialAlertNotificationConditionEnum::WEEKLY_SUMMARY->value, FinancialAlertNotificationConditionEnum::DAILY_SUMMARY->value])
        );

        // daily summary
        $this->manager->sendSummaryAlertNotification(FinancialAlertNotificationConditionEnum::DAILY_SUMMARY);
        Notification::assertSentOnDemand(FinancialAlertEmailNotification::class);
        // weekly summary
        $this->manager->sendSummaryAlertNotification(FinancialAlertNotificationConditionEnum::WEEKLY_SUMMARY);
        Notification::assertSentOnDemand(FinancialAlertEmailNotification::class);
    }

    /**
     * @throws Exception
     */
    public function test_it_can_send_slack_immediate_notification(): void
    {
        app(SettingRepository::class)->set(Setting::KEY_FINANCIAL_ALERTS_NOTIFICATION_TYPE, Setting::KEY_FINANCIAL_ALERTS_NOTIFICATION_TYPE_SLACK);
        app(SettingRepository::class)->set(Setting::KEY_FINANCIAL_ALERTS_NOTIFICATION_DESTINATION, 'https://hooks.slack.com/services/...');
        app(SettingRepository::class)->set(Setting::KEY_FINANCIAL_ALERTS_LAST_NOTIFICATION_DATE, now()->subMinute());

        app(SettingRepository::class)->set(Setting::KEY_FINANCIAL_ALERTS_NOTIFICATION_CONDITIONS,
            json_encode([FinancialAlertNotificationConditionEnum::IMMEDIATELY->value])
        );

        $this->createFinancialAlert();

        Notification::assertSentOnDemand(FinancialAlertSlackNotification::class);
    }

    /**
     * @throws Exception
     */
    public function test_it_can_send_email_immediate_notification(): void
    {
        app(SettingRepository::class)->set(Setting::KEY_FINANCIAL_ALERTS_NOTIFICATION_TYPE, Setting::KEY_FINANCIAL_ALERTS_NOTIFICATION_TYPE_EMAIL);
        app(SettingRepository::class)->set(Setting::KEY_FINANCIAL_ALERTS_NOTIFICATION_DESTINATION, 'me@sku.com');
        app(SettingRepository::class)->set(Setting::KEY_FINANCIAL_ALERTS_LAST_NOTIFICATION_DATE, now()->subMinute());
        app(SettingRepository::class)->set(Setting::KEY_FINANCIAL_ALERTS_NOTIFICATION_CONDITIONS,
            json_encode([FinancialAlertNotificationConditionEnum::IMMEDIATELY->value])
        );

        $this->createFinancialAlert();

        Notification::assertSentOnDemand(FinancialAlertEmailNotification::class);
    }

}