<?php

namespace Tests\Feature\Controllers;

use App\Models\FifoLayer;
use App\Models\InventoryAdjustment;
use App\Models\Product;
use App\Models\ProductBlemished;
use App\Models\ProductInventory;
use App\Models\User;
use App\Models\Warehouse;
use App\Models\WarehouseTransfer;
use App\Models\WarehouseTransferLine;
use App\Models\WarehouseTransferShipmentLine;
use App\Models\WarehouseTransferShipmentReceipt;
use App\Models\WarehouseTransferShipmentReceiptLine;
use Carbon\Carbon;
use Illuminate\Support\Collection;
use Laravel\Sanctum\Sanctum;
use Plannr\Laravel\FastRefreshDatabase\Traits\FastRefreshDatabase;
use Tests\TestCase;

class WarehouseTransferShipmentControllerTest extends TestCase
{
    use FastRefreshDatabase;

    public function setUp(): void
    {
        parent::setUp();
        Sanctum::actingAs(User::first());
    }

    public function test_it_can_receive_blemished_skus_in_warehouse_transfer(): void{

        /** @var WarehouseTransfer $transfer */
        $transfer = WarehouseTransfer::factory()
            ->hasWarehouseTransferLines(2, [
                'quantity' => 10,
            ])
            ->create();
        $lines = $transfer->warehouseTransferLines;

        $this->createInventoryForLines($lines);

        // Open the warehouse transfer.
        $this->putJson(route('warehouses.transfers.open', $transfer->id))->assertSuccessful();


        $this->postJson(route('warehouses.transfers.receive', $transfer->id), [
            'receipt_date' => now(),
            'products' => [
                [
                    'id' => $lines->first()->product_id,
                    'quantity' => 5,
                ],
                [
                    'id' => $lines->last()->product_id,
                    'quantity' => 10,
                ],
            ],
            'blemished' => [
                [
                    'sku' => 'Z_1234',
                    'product_id' => $lines->first()->product_id,
                    'quantity' => 1,
                    'condition' => 'Broken',
                    'reference' => '1234',
                ],
            ],
        ])->assertSuccessful();


        $this->assertDatabaseCount(WarehouseTransfer::class, 1);
        $this->assertDatabaseCount(WarehouseTransferLine::class, 2);

        // Receipt quantities should be complete
        $this->assertDatabaseCount(WarehouseTransferShipmentReceipt::class, 1);
        $this->assertDatabaseCount(WarehouseTransferShipmentReceiptLine::class, 2);
        $this->assertDatabaseHas(WarehouseTransferShipmentReceiptLine::class, [
            'warehouse_transfer_shipment_line_id' => WarehouseTransferShipmentLine::query()
            ->where('warehouse_transfer_line_id', $lines->first()->id)->firstOrFail()->id,
            'quantity' => 5, // 5 received, blemished qty will create an adjustment.
        ]);
        $this->assertDatabaseHas(WarehouseTransferShipmentReceiptLine::class, [
            'warehouse_transfer_shipment_line_id' => WarehouseTransferShipmentLine::query()
                ->where('warehouse_transfer_line_id', $lines->last()->id)->firstOrFail()->id,
            'quantity' => 10,
        ]);

        // Blemished sku must be created.
        $this->assertDatabaseHas(Product::class, [
            'type' => Product::TYPE_BLEMISHED,
        ]);
        $this->assertDatabaseCount(ProductBlemished::class, 1);

        // Necessary inventory adjustments must be created.
        // 2 for initial inventory, 2 for blemished receipt.
        $this->assertDatabaseCount(InventoryAdjustment::class, 4);

        $this->assertDatabaseHas(InventoryAdjustment::class, [
            'product_id' => $lines->first()->product_id,
            'quantity' => -1,
            'link_type' => WarehouseTransferLine::class,
        ]);
        $this->assertDatabaseHas(InventoryAdjustment::class, [
            'product_id' => Product::where('sku', 'Z_1234')->firstOrFail()->id,
            'quantity' => 1,
            'link_type' => WarehouseTransferLine::class
        ]);

    }

    public function test_it_can_delete_receipt_with_blemished_sku(): void{

        /** @var WarehouseTransfer $transfer */
        $transfer = WarehouseTransfer::factory()
            ->hasWarehouseTransferLines(1, [
                'quantity' => 10,
            ])
            ->create();
        $lines = $transfer->warehouseTransferLines;

        $this->createInventoryForLines($lines);

        // Open the warehouse transfer.
        $this->putJson(route('warehouses.transfers.open', $transfer->id))->assertSuccessful();

        $response = $this->postJson(route('warehouses.transfers.receive', $transfer->id), [
            'receipt_date' => now(),
            'products' => [
                [
                    'id' => $lines->first()->product_id,
                    'quantity' => 10,
                ]
            ],
            'blemished' => [
                [
                    'sku' => 'Z_1234',
                    'product_id' => $lines->first()->product_id,
                    'quantity' => 1,
                    'condition' => 'Broken',
                    'reference' => '1234',
                ],
            ],
        ])->assertSuccessful();

        $this->assertDatabaseCount(WarehouseTransfer::class, 1);
        $this->assertDatabaseCount(WarehouseTransferLine::class, 1);

        // Receipt quantities should be complete
        $this->assertDatabaseCount(WarehouseTransferShipmentReceipt::class, 1);
        $this->assertDatabaseCount(WarehouseTransferShipmentReceiptLine::class, 1);
        $this->assertDatabaseHas(WarehouseTransferShipmentReceiptLine::class, [
            'warehouse_transfer_shipment_line_id' => WarehouseTransferShipmentLine::query()
                ->where('warehouse_transfer_line_id', $lines->first()->id)->firstOrFail()->id,
            'quantity' => 10, // 10 received, blemished qty will create an adjustment.
        ]);

        // Blemished sku must be created.
        $this->assertDatabaseHas(Product::class, [
            'type' => Product::TYPE_BLEMISHED,
        ]);
        $this->assertDatabaseCount(ProductBlemished::class, 1); // Each blemished product is unique

        // Necessary inventory adjustments must be created.
        // 1 for initial inventory, 2 for blemished receipt.
        $this->assertDatabaseCount(InventoryAdjustment::class, 3);

        $this->assertDatabaseHas(InventoryAdjustment::class, [
            'product_id' => $lines->first()->product_id,
            'quantity' => -1,
            'link_type' => WarehouseTransferLine::class,
        ]);
        $this->assertDatabaseHas(InventoryAdjustment::class, [
            'product_id' => Product::where('sku', 'Z_1234')->firstOrFail()->id,
            'quantity' => 1,
            'link_type' => WarehouseTransferLine::class
        ]);

        // Delete the receipt
        $this->deleteJson(route('warehouses.transfers.delete-receipt', [$transfer->id, $response->json('data.warehouse_transfer_shipment_receipts.0.id')]))->assertOk();

        // Receipt should be deleted
        $this->assertDatabaseCount(WarehouseTransferShipmentReceipt::class, 0);
        $this->assertDatabaseCount(WarehouseTransferShipmentReceiptLine::class, 0);

        // Blemished products should be deleted
        $this->assertDatabaseCount(ProductBlemished::class, 0);

        // Inventory adjustments should be deleted
        // 1 for initial inventory.
        $this->assertDatabaseCount(InventoryAdjustment::class, 1);


    }

    public function test_it_can_reduce_quantity_of_receipt_with_blemished_sku(): void{

        /** @var WarehouseTransfer $transfer */
        $transfer = WarehouseTransfer::factory()
            ->hasWarehouseTransferLines(1, [
                'quantity' => 10,
            ])
            ->create();
        $lines = $transfer->warehouseTransferLines;

        $this->createInventoryForLines($lines);

        // Open the warehouse transfer.
        $this->putJson(route('warehouses.transfers.open', $transfer->id))->assertSuccessful();


        $response = $this->postJson(route('warehouses.transfers.receive', $transfer->id), [
            'receipt_date' => now(),
            'products' => [
                [
                    'id' => $lines->first()->product_id,
                    'quantity' => 8,
                ]
            ],
            'blemished' => [
                [
                    'sku' => 'Z_1234',
                    'product_id' => $lines->first()->product_id,
                    'quantity' => 1,
                    'condition' => 'Broken',
                    'reference' => '1234',
                ],
            ],
        ])->assertSuccessful();


        $this->assertDatabaseCount(WarehouseTransfer::class, 1);
        $this->assertDatabaseCount(WarehouseTransferLine::class, 1);

        // Receipt quantities should be complete
        $this->assertDatabaseCount(WarehouseTransferShipmentReceipt::class, 1);
        $this->assertDatabaseCount(WarehouseTransferShipmentReceiptLine::class, 1);
        $this->assertDatabaseHas(WarehouseTransferShipmentReceiptLine::class, [
            'warehouse_transfer_shipment_line_id' => WarehouseTransferShipmentLine::query()
                ->where('warehouse_transfer_line_id', $lines->first()->id)->firstOrFail()->id,
            'quantity' => 8, // 8 received, blemished qty will create an adjustment.
        ]);

        // Blemished sku must be created.
        $this->assertDatabaseHas(Product::class, [
            'type' => Product::TYPE_BLEMISHED,
        ]);
        $this->assertDatabaseCount(ProductBlemished::class, 1); // Each blemished product is unique

        // Necessary inventory adjustments must be created.
        // 1 for initial inventory, 2 for blemished receipt.
        $this->assertDatabaseCount(InventoryAdjustment::class, 3);

        $this->assertDatabaseHas(InventoryAdjustment::class, [
            'product_id' => $lines->first()->product_id,
            'quantity' => -1,
            'link_type' => WarehouseTransferLine::class,
        ]);
        $this->assertDatabaseHas(InventoryAdjustment::class, [
            'product_id' => Product::where('sku', 'Z_1234')->firstOrFail()->id,
            'quantity' => 1,
            'link_type' => WarehouseTransferLine::class
        ]);

        // Reduce receipt quantity
        $this->putJson(route("warehouses.transfers.update-receipt", [$transfer->id, $response->json('data.warehouse_transfer_shipment_receipts.0.id')]), [
            'products' => [
                [
                    'id' => $lines->first()->product_id,
                    'quantity' => 5,
                ]
            ],
        ])->assertSuccessful();

        // Receipt should be updated
        $this->assertDatabaseCount(WarehouseTransferShipmentReceipt::class, 1);
        $this->assertDatabaseCount(WarehouseTransferShipmentReceiptLine::class, 1);

        // Blemished products should be retained
        $this->assertDatabaseCount(ProductBlemished::class, 1);

        // Inventory adjustments should be retained
        $this->assertDatabaseCount(InventoryAdjustment::class, 3);
    }

    public function test_it_can_reverse_applicable_blemished_sku_when_transfer_receipt_reduces(): void{

        /** @var WarehouseTransfer $transfer */
        $transfer = WarehouseTransfer::factory()
            ->hasWarehouseTransferLines(1, [
                'quantity' => 10,
            ])
            ->create();

        $lines = $transfer->warehouseTransferLines;

        $this->createInventoryForLines($lines);

        // Open the warehouse transfer.
        $this->putJson(route('warehouses.transfers.open', $transfer->id))->assertSuccessful();


        $response = $this->postJson(route('warehouses.transfers.receive', $transfer->id), [
            'receipt_date' => Carbon::now(),
            'products' => [
                [
                    'id' => $lines->first()->product_id,
                    'quantity' => 8,
                ]
            ],
            'blemished' => [
                [
                    'sku' => 'Z_1234',
                    'product_id' => $lines->first()->product_id,
                    'quantity' => 1,
                    'condition' => 'Broken',
                    'reference' => '1234',
                ],
                [
                    'sku' => 'Z_1234.1',
                    'product_id' => $lines->first()->product_id,
                    'quantity' => 1,
                    'condition' => 'Broken',
                    'reference' => '1234',
                ],
            ],
        ])->assertSuccessful();


        $this->assertDatabaseCount('warehouse_transfers', 1);
        $this->assertDatabaseCount('warehouse_transfer_lines', 1);

        // Receipt quantities should be complete
        $this->assertDatabaseCount(WarehouseTransferShipmentReceipt::class, 1);
        $this->assertDatabaseCount(WarehouseTransferShipmentReceiptLine::class, 1);
        $this->assertDatabaseHas(WarehouseTransferShipmentReceiptLine::class, [
            'warehouse_transfer_shipment_line_id' => WarehouseTransferShipmentLine::query()
                ->where('warehouse_transfer_line_id', $lines->first()->id)->firstOrFail()->id,
            'quantity' => 8, // 8 received, blemished qty will create an adjustment.
        ]);

        // Blemished sku must be created.
        $this->assertDatabaseHas(Product::class, [
            'type' => Product::TYPE_BLEMISHED,
        ]);
        $this->assertDatabaseCount(ProductBlemished::class, 2); // Each blemished product is unique

        // Necessary inventory adjustments must be created.
        // 1 for initial inventory, 4 for blemished receipt.
        $this->assertDatabaseCount(InventoryAdjustment::class, 5);

        $this->assertDatabaseHas(InventoryAdjustment::class, [
            'product_id' => $lines->first()->product_id,
            'quantity' => -1,
            'link_type' => WarehouseTransferLine::class,
        ]);
        $this->assertDatabaseHas(InventoryAdjustment::class, [
            'product_id' => Product::where('sku', 'Z_1234')->firstOrFail()->id,
            'quantity' => 1,
            'link_type' => WarehouseTransferLine::class
        ]);

        $this->assertDatabaseHas(InventoryAdjustment::class, [
            'product_id' => Product::where('sku', 'Z_1234.1')->firstOrFail()->id,
            'quantity' => 1,
            'link_type' => WarehouseTransferLine::class
        ]);


        // Reduce the quantity to be less than the blemished quantity
        $this->putJson(route('warehouses.transfers.update-receipt', [$transfer->id, $response->json('data.warehouse_transfer_shipment_receipts.0.id')]), [
            'products' => [
                [
                    'id' => $lines->first()->product_id,
                    'quantity' => 1,
                ]
            ],
        ])->assertSuccessful();

        // Receipt should be updated
        $this->assertDatabaseCount(WarehouseTransferShipmentReceipt::class, 1);
        $this->assertDatabaseCount(WarehouseTransferShipmentReceiptLine::class, 1);

        // Blemished products should be adjusted
        $this->assertDatabaseCount(ProductBlemished::class, 1);

        // Inventory adjustments should be adjusted
        $this->assertDatabaseCount(InventoryAdjustment::class, 3);

    }

    public function test_it_can_report_on_unknown_items_in_warehouse_transfers(): void
    {
        $fromWarehouse = Warehouse::factory()->create(['name' => 'Warehouse 1']);
        $toWarehouse = Warehouse::factory()->create(['name' => 'Warehouse 2']);

        $warehouseTransfer = WarehouseTransfer::factory()->create([
            'from_warehouse_id' => $fromWarehouse->id,
            'to_warehouse_id' => $toWarehouse->id,
        ]);

        $response = $this->postJson(route('warehouses.transfers.import-lines'), [
            'warehouse_transfer_id' => $warehouseTransfer->id,
            'separator' => ',',
            'escape' => "\"",
            'csvString' => "sku,quantity\nunknownsku1,10\nunknownsku2,20\n",
            'replace' => true,
        ]);

        $this->assertEquals([
            'unknownsku1',
            'unknownsku2',
        ], $response->json('errors.unknown_skus.0.data'));
    }

    public function test_it_can_receive_already_existing_blemished_skus_in_warehouse_transfer(): void
    {
        /** @var WarehouseTransfer $transfer */
        $transfer = WarehouseTransfer::factory()
            ->hasWarehouseTransferLines(2, [
                'quantity' => 10,
            ])
            ->create();
        $lines = $transfer->warehouseTransferLines;

        $this->createInventoryForLines($lines);

        // Open the warehouse transfer.
        $this->putJson(route('warehouses.transfers.open', $transfer->id))->assertSuccessful();

        $firstProduct = $lines->first()->product;

        // Create a blemished product

        $blemishedProduct = Product::factory()->create([
            'type' => Product::TYPE_BLEMISHED,
            'sku' => 'Z_1234',
        ]);

        ProductBlemished::create([
            'product_id' => $blemishedProduct->id,
            'derived_from_product_id' => $firstProduct->id,
            'condition' => 'Broken',
            'reference' => '1234',
        ]);

        $response = $this->postJson(route('warehouses.transfers.receive', $transfer->id), [
            'receipt_date' => now(),
            'products' => [
                [
                    'id' => $lines->first()->product_id,
                    'quantity' => 5,
                ],
                [
                    'id' => $lines->last()->product_id,
                    'quantity' => 10,
                ],
            ],
            'blemished' => [
                [
                    'sku' => 'Z_1234',
                    'product_id' => $lines->first()->product_id,
                    'quantity' => 1,
                    'condition' => 'Broken',
                    'reference' => '1234',
                ],
            ],
        ]);

        $response->assertSuccessful();

        $blemishedProduct = Product::where('sku', 'Z_1234')->firstOrFail();

        $this->assertDatabaseCount(WarehouseTransfer::class, 1);
        $this->assertDatabaseCount(WarehouseTransferLine::class, 2);

        // Receipt quantities should be complete
        $this->assertDatabaseCount(WarehouseTransferShipmentReceipt::class, 1);
        $this->assertDatabaseCount(WarehouseTransferShipmentReceiptLine::class, 2);
        $this->assertDatabaseHas(WarehouseTransferShipmentReceiptLine::class, [
            'warehouse_transfer_shipment_line_id' => WarehouseTransferShipmentLine::query()
                ->where('warehouse_transfer_line_id', $lines->first()->id)->firstOrFail()->id,
            'quantity' => 5, // 5 received, blemished qty will create an adjustment.
        ]);
        $this->assertDatabaseHas(WarehouseTransferShipmentReceiptLine::class, [
            'warehouse_transfer_shipment_line_id' => WarehouseTransferShipmentLine::query()
                ->where('warehouse_transfer_line_id', $lines->last()->id)->firstOrFail()->id,
            'quantity' => 10,
        ]);

        // Blemished sku must be created.
        $this->assertDatabaseHas(Product::class, [
            'type' => Product::TYPE_BLEMISHED,
        ]);
        $this->assertDatabaseCount(ProductBlemished::class, 1);

        // Necessary inventory adjustments must be created.
        // 2 for initial inventory, 2 for blemished receipt.
        $this->assertDatabaseCount(InventoryAdjustment::class, 4);

        $this->assertDatabaseHas(InventoryAdjustment::class, [
            'product_id' => $lines->first()->product_id,
            'quantity' => -1,
            'link_type' => WarehouseTransferLine::class,
        ]);
        $this->assertDatabaseHas(InventoryAdjustment::class, [
            'product_id' => Product::where('sku', 'Z_1234')->firstOrFail()->id,
            'quantity' => 1,
            'link_type' => WarehouseTransferLine::class
        ]);

    }


    private function createInventoryForLines(Collection $lines): void
    {
        /** @var WarehouseTransferLine $line */
        foreach ($lines as $line) {
            $this->postJson(route('inventory-adjustments.store'), [
                'adjustment_date' => now(),
                'adjustment_type' => InventoryAdjustment::TYPE_INCREASE,
                'product_id' => $line->product_id,
                'quantity' => $line->quantity,
                'warehouse_id' => $line->warehouseTransfer->from_warehouse_id,
                'unit_cost' => 10,
            ])->assertSuccessful();
        }
    }
}
