<?php

namespace Tests\Feature\Controllers;

use App\Data\CreateStockTakeData;
use App\Data\StockTakeItemData;
use App\Data\UpdateStockTakeData;
use App\Exceptions\SalesOrder\InvalidProductWarehouseRouting;
use App\Helpers;
use App\Models\FifoLayer;
use App\Models\InventoryMovement;
use App\Models\Product;
use App\Models\SalesOrder;
use App\Models\SalesOrderLine;
use App\Models\StockTake;
use App\Models\StockTakeItem;
use App\Models\User;
use App\Models\Warehouse;
use App\Services\FinancialManagement\SalesOrderLineFinancialManager;
use App\Services\SalesOrder\SalesOrderManager;
use App\Services\StockTake\StockTakeManager;
use Carbon\Carbon;
use Laravel\Sanctum\Sanctum;
use Plannr\Laravel\FastRefreshDatabase\Traits\FastRefreshDatabase;
use Tests\TestCase;
use Throwable;

class StockTakeControllerTest extends TestCase
{
    use FastRefreshDatabase;

    protected function setUp(): void
    {
        parent::setUp();
        Sanctum::actingAs(User::first());
    }

    /**
     * @throws Throwable
     * @throws InvalidProductWarehouseRouting
     */
    public function test_stock_take_controller_for_positive_stock_takes()
    {
        $warehouse = Warehouse::first();
        $product1 = Product::factory()->create();
        $product2 = Product::factory()->create();

        /*
        |--------------------------------------------------------------------------
        | Create Stock Take
        |--------------------------------------------------------------------------
        */

        $response = $this->postJson(route('stock-takes.store'), CreateStockTakeData::from([
            'warehouse_id' => $warehouse->id,
            'date_count' => '2022-01-01',
            'items' => StockTakeItemData::collection([
                [
                    'product_id' => $product1->id,
                    'qty_counted' => 10,
                    'unit_cost' => 100,
                ],
                [
                    'product_id' => $product2->id,
                    'qty_counted' => 20,
                    'unit_cost' => 10,
                ],
            ]),
        ])->toArray());

        $response->assertOk();

        $stockTakeId = $response->json('data')['id'];

        $this->assertDatabaseHas(StockTake::class, [
            'warehouse_id' => $warehouse->id,
            'date_count' => '2022-01-01',
            'status' => StockTake::STOCK_TAKE_STATUS_DRAFT,
        ]);

        $this->assertDatabaseHas(StockTakeItem::class, [
            'product_id' => $product1->id,
            'qty_counted' => 10,
            'unit_cost' => 100,
        ]);

        $this->assertDatabaseHas(StockTakeItem::class, [
            'product_id' => $product2->id,
            'qty_counted' => 20,
            'unit_cost' => 10,
        ]);

        /*
        |--------------------------------------------------------------------------
        | Update Stock Take when Draft
        |--------------------------------------------------------------------------
        */

        $response = $this->putJson(route('stock-takes.update', $stockTakeId), UpdateStockTakeData::from([
            'items' => StockTakeItemData::collection([
                [
                    'product_id' => $product1->id,
                    'qty_counted' => 15,
                    'unit_cost' => 100,
                ],
                [
                    'product_id' => $product2->id,
                    'qty_counted' => 25,
                    'unit_cost' => 10,
                ],
            ]),
        ])->toArray());

        $response->assertOk();

        $this->assertDatabaseHas(StockTakeItem::class, [
            'product_id' => $product1->id,
            'qty_counted' => 15,
            'snapshot_inventory' => null,
            'unit_cost' => 100,
        ]);

        $this->assertDatabaseHas(StockTakeItem::class, [
            'product_id' => $product2->id,
            'qty_counted' => 25,
            'snapshot_inventory' => null,
            'unit_cost' => 10,
        ]);

        $this->assertDatabaseEmpty(InventoryMovement::class);

        /*
        |--------------------------------------------------------------------------
        | Initiate Stock Take
        |--------------------------------------------------------------------------
        */

        $response = $this->postJson(route('stock-takes.initiate', $stockTakeId));

        $response->assertOk();

        $this->assertDatabaseHas(StockTake::class, [
            'id' => $stockTakeId,
            'status' => StockTake::STOCK_TAKE_STATUS_OPEN,
            'value_change' => 1750,
        ]);

        $this->assertDatabaseHas(StockTakeItem::class, [
            'product_id' => $product1->id,
            'snapshot_inventory' => 0,
        ]);

        $this->assertDatabaseHas(StockTakeItem::class, [
            'product_id' => $product2->id,
            'snapshot_inventory' => 0,
        ]);

        $this->assertDatabaseEmpty(InventoryMovement::class);

        /*
        |--------------------------------------------------------------------------
        | Update Stock Take when Open
        |--------------------------------------------------------------------------
        */

        $this->putJson(route('stock-takes.update', $stockTakeId), UpdateStockTakeData::from([
            'items' => StockTakeItemData::collection([
                [
                    'product_id' => $product1->id,
                    'qty_counted' => 20,
                    'unit_cost' => 100,
                ],
                [
                    'product_id' => $product2->id,
                    'qty_counted' => 30,
                    'unit_cost' => 10,
                ],
            ]),
        ])->toArray())->assertOk();

        $this->assertDatabaseHas(StockTake::class, [
            'id' => $stockTakeId,
            'status' => StockTake::STOCK_TAKE_STATUS_OPEN,
            'value_change' => 2300,
        ]);

        $this->assertDatabaseHas(StockTakeItem::class, [
            'product_id' => $product1->id,
            'qty_counted' => 20,
            'snapshot_inventory' => 0,
        ]);

        $this->assertDatabaseHas(StockTakeItem::class, [
            'product_id' => $product2->id,
            'qty_counted' => 30,
            'snapshot_inventory' => 0,
        ]);

        /*
        |--------------------------------------------------------------------------
        | Finalize Stock Take
        |--------------------------------------------------------------------------
        */

        $response = $this->postJson(route('stock-takes.finalize', $stockTakeId));

        $response->assertOk();

        $this->assertDatabaseHas(StockTake::class, [
            'id' => $stockTakeId,
            'status' => StockTake::STOCK_TAKE_STATUS_CLOSED,
            'value_change' => 2300,
        ]);

        $this->assertDatabaseHas(FifoLayer::class, [
            'product_id' => $product1->id,
            'warehouse_id' => $warehouse->id,
            'original_quantity' => 20,
            'total_cost' => 2000,
            'link_type' => StockTakeItem::class,
        ]);

        $this->assertDatabaseHas(FifoLayer::class, [
            'product_id' => $product2->id,
            'warehouse_id' => $warehouse->id,
            'original_quantity' => 30,
            'total_cost' => 300,
            'link_type' => StockTakeItem::class,
        ]);

        $this->assertDatabaseHas(InventoryMovement::class, [
            'product_id' => $product1->id,
            'warehouse_id' => $warehouse->id,
            'quantity' => 20,
            'type' => InventoryMovement::TYPE_STOCK_TAKE,
        ]);

        $this->assertDatabaseHas(InventoryMovement::class, [
            'product_id' => $product2->id,
            'warehouse_id' => $warehouse->id,
            'quantity' => 30,
            'type' => InventoryMovement::TYPE_STOCK_TAKE,
        ]);

        /*
        |--------------------------------------------------------------------------
        | Update Stock Take when Closed (and not initial stock)
        |--------------------------------------------------------------------------
        */

        // Can't delete items
        $response = $this->putJson(route('stock-takes.update', $stockTakeId), UpdateStockTakeData::from([
            'items' => StockTakeItemData::collection([
                [
                    'product_id' => $product1->id,
                    'to_delete' => true,
                ],
            ]),
        ])->toArray());

        $response->assertBadRequest();

        // Can't change quantity
        $response = $this->putJson(route('stock-takes.update', $stockTakeId), UpdateStockTakeData::from([
            'items' => StockTakeItemData::collection([
                [
                    'product_id' => $product1->id,
                    'qty_counted' => 2,
                ],
            ]),
        ])->toArray());

        $response->assertBadRequest();

        // Usage set up

        app(SalesOrderManager::class)->createOrder([
            'order_status' => SalesOrder::STATUS_OPEN,
            'sales_order_lines' => [
                [
                    'product_id' => $product1->id,
                    'description' => 'Test',
                    'quantity' => 1,
                ],
            ],
        ]);

        app(SalesOrderLineFinancialManager::class)->calculate();

        $this->assertEquals(100, SalesOrderLine::first()->salesOrderLineFinancial->cogs);

        // Can change unit cost (and have it update usage)
        $response = $this->putJson(route('stock-takes.update', $stockTakeId), UpdateStockTakeData::from([
            'items' => StockTakeItemData::collection([
                [
                    'product_id' => $product1->id,
                    'unit_cost' => 50,
                ],
            ]),
        ])->toArray());

        $response->assertOk();

        $this->assertDatabaseHas(StockTake::class, [
            'id' => $stockTakeId,
            'status' => StockTake::STOCK_TAKE_STATUS_CLOSED,
            'value_change' => 1300,
        ]);

        $this->assertDatabaseHas(StockTakeItem::class, [
            'product_id' => $product1->id,
            'unit_cost' => 50,
        ]);

        $this->assertDatabaseHas(FifoLayer::class, [
            'product_id' => $product1->id,
            'total_cost' => 1000,
        ]);

        $this->assertEquals(50, SalesOrderLine::first()->salesOrderLineFinancial->cogs);


        /*
        |--------------------------------------------------------------------------
        | Change Date on Positive Stock Change
        |--------------------------------------------------------------------------
        */

        $response = $this->putJson(route('stock-takes.update', $stockTakeId), UpdateStockTakeData::from([
            'date_count' => '2023-01-01',
        ])->toArray());

        $response->assertOk();

        $this->assertDatabaseHas(StockTake::class, [
            'id' => $stockTakeId,
            'date_count' => '2023-01-01',
        ]);

        $this->assertDatabaseHas(InventoryMovement::class, [
            'product_id' => $product1->id,
            'warehouse_id' => $warehouse->id,
            'quantity' => 20,
            'type' => InventoryMovement::TYPE_STOCK_TAKE,
            'inventory_movement_date' => Carbon::parse('2023-01-01')->toDateTimeString(),
        ]);

        $this->assertDatabaseHas(FifoLayer::class, [
            'product_id' => $product1->id,
            'warehouse_id' => $warehouse->id,
            'original_quantity' => 20,
            'total_cost' => 1000,
            'link_type' => StockTakeItem::class,
            'fifo_layer_date' => Carbon::parse('2023-01-01')->toDateTimeString(),
        ]);
    }

    /**
     * @throws Throwable
     */
    public function test_stock_take_controller_for_negative_stock_takes(): void
    {
        $warehouse = Warehouse::first();
        $product1 = Product::factory()->create();
        $product2 = Product::factory()->create();

        /*
        |--------------------------------------------------------------------------
        | Create Positive Stock Take
        |--------------------------------------------------------------------------
        */

        $manager = app(StockTakeManager::class);

        $initialStockTake = $manager->createStockTake(CreateStockTakeData::from([
            'warehouse_id' => $warehouse->id,
            'date_count' => '2022-01-01',
            'items' => StockTakeItemData::collection([
                [
                    'product_id' => $product1->id,
                    'qty_counted' => 10,
                    'unit_cost' => 10,
                ],
                [
                    'product_id' => $product2->id,
                    'qty_counted' => 20,
                    'unit_cost' => 15,
                ],
            ]),
        ]));

        $manager->initiateCount($initialStockTake);
        $manager->finalizeStockTake($initialStockTake);

        $this->assertDatabaseHas(StockTake::class, [
            'warehouse_id' => $warehouse->id,
            'date_count' => '2022-01-01',
            'status' => StockTake::STOCK_TAKE_STATUS_CLOSED,
        ]);

        $this->assertDatabaseCount(FifoLayer::class, 2);
        $this->assertDatabaseCount(InventoryMovement::class, 2);

        /*
        |--------------------------------------------------------------------------
        | Create Negative Stock Take
        |--------------------------------------------------------------------------
        */

        $response = $this->postJson(route('stock-takes.store'), CreateStockTakeData::from([
            'warehouse_id' => $warehouse->id,
            'date_count' => '2022-01-01',
            'items' => StockTakeItemData::collection([
                [
                    'product_id' => $product1->id,
                    'qty_counted' => 8,
                    'unit_cost' => 999, // unit cost should be ignored
                ],
                [
                    'product_id' => $product2->id,
                    'qty_counted' => 18,
                    'unit_cost' => 999, // unit cost should be ignored
                ],
            ]),
        ])->toArray());

        $response->assertOk();

        $negativeStockTakeId = $response->json('data')['id'];
        $negativeStockTake = StockTake::find($negativeStockTakeId);
        $manager->initiateCount($negativeStockTake);
        $manager->finalizeStockTake($negativeStockTake);

        $this->assertDatabaseCount(InventoryMovement::class, 4);

        // Assert avg cost of fifo layer for the negative movements
        $negativeStockTake->refresh()->stockTakeItems->each(function (StockTakeItem $item) use ($initialStockTake) {
            $matchingInitialStockTakeItem = $item->product->stockTakeItems()
                ->where('stock_take_id', $initialStockTake->id)
                ->first();
            $inventoryMovement = $item->inventoryMovements()->first();
            $fifoLayer = $inventoryMovement->fifo_layer;
            $this->assertEquals($matchingInitialStockTakeItem->unit_cost, $fifoLayer->avg_cost);
        });

        /*
        |--------------------------------------------------------------------------
        | Change Date on Negative Stock Change
        |--------------------------------------------------------------------------
        */

        $response = $this->putJson(route('stock-takes.update', $negativeStockTakeId), UpdateStockTakeData::from([
            'date_count' => '2023-01-01',
        ])->toArray());

        $response->assertOk();

        $this->assertDatabaseHas(StockTake::class, [
            'id' => $negativeStockTakeId,
            'date_count' => '2023-01-01',
        ]);

        $this->assertDatabaseHas(InventoryMovement::class, [
            'product_id' => $product1->id,
            'warehouse_id' => $warehouse->id,
            'quantity' => -2,
            'type' => InventoryMovement::TYPE_STOCK_TAKE,
            'inventory_movement_date' => Carbon::parse('2023-01-01', Helpers::getAppTimezone())->utc(),
        ]);
    }

    public function test_it_can_create_stock_take_step_by_step(): void
    {
        $product1 = Product::factory()->create();
        $product2 = Product::factory()->create();
        $warehouse = Warehouse::first();

        $response = $this->postJson(route('stock-takes.store'), CreateStockTakeData::from([
            'warehouse_id' => $warehouse->id,
            'date_count' => '2022-01-01',
        ])->toArray())->assertOk();

        $stockTakeId = $response->json('data')['id'];

        $this->putJson(route('stock-takes.update', $stockTakeId), UpdateStockTakeData::from([
            'items' => StockTakeItemData::collection([
                [
                    'product_id' => $product1->id,
                ],
                [
                    'product_id' => $product2->id,
                ],
            ]),
        ])->toArray())->assertOk();

        $this->putJson(route('stock-takes.update', $stockTakeId), UpdateStockTakeData::from([
            'items' => StockTakeItemData::collection([
                [
                    'product_id' => $product1->id,
                ],
                [
                    'product_id' => $product2->id,
                    'to_delete' => 1,
                ],
            ]),
        ])->toArray())->assertOk();
    }
}
