<?php

namespace Tests\Feature\Controllers;

use App\Helpers;
use App\Models\FifoLayer;
use App\Models\InventoryMovement;
use App\Models\Product;
use App\Models\SalesOrder;
use App\Models\Setting;
use App\Models\StockTake;
use App\Models\StockTakeItem;
use App\Models\User;
use App\Models\Warehouse;
use App\Services\SalesOrder\SalesOrderManager;
use Laravel\Sanctum\Sanctum;
use League\Csv\Exception;
use League\Csv\Writer;
use Plannr\Laravel\FastRefreshDatabase\Traits\FastRefreshDatabase;
use Storage;
use Tests\TestCase;

/*
 * TODO: The only way to modify initial inventory other than uploading is via product controller update (which seems
 *  to be broken in the frontend currently).  Frontend should be updated to hit a method in the InitialInventoryController instead
 */

class InitialInventoryControllerTest extends TestCase
{
    use FastRefreshDatabase;

    protected function setUp(): void
    {
        parent::setUp();
        Sanctum::actingAs(User::first());
    }

    /**
     * @throws Exception
     */
    public function test_it_can_import_initial_inventory_using_stock_take()
    {
        $warehouse = Warehouse::first();

        $product1 = Product::factory()->create(['sku' => 'sku1']);
        $product2 = Product::factory()->create(['sku' => 'sku2']);


        // Write file initial_inventory.csv in storage/imports/models/initial_inventory.csv

        $data = [
            ['sku', 'quantity', 'unit_cost'],
            ['sku1', 10, 100],
            ['sku2', 20, 200],
        ];

        $csv = Writer::createFromString('');
        $csv->insertAll($data);
        $csvContent = $csv->toString();

        Storage::disk('model-imports')->put('initial_inventory.csv', $csvContent);

        $response = $this->postJson(route('initial-inventory.import'), [
            'original_name' => 'initial_inventory.csv',
            'stored_name' => 'initial_inventory.csv',
            'meta' => [
                'warehouse_id' => $warehouse->id,
            ]
        ]);

        $response->assertOk();

        $this->assertDatabaseHas(StockTake::class, [
            'warehouse_id' => $warehouse->id,
            'is_initial_count' => true,
            'date_count' => Helpers::setting(Setting::KEY_INVENTORY_START_DATE),
            'value_change' => 5000,
        ]);

        $this->assertDatabaseHas(StockTakeItem::class, [
            'product_id' => $product1->id,
            'qty_counted' => 10,
            'unit_cost' => 100,
        ]);

        $this->assertDatabaseHas(StockTakeItem::class, [
            'product_id' => $product2->id,
            'qty_counted' => 20,
            'unit_cost' => 200,
        ]);

        $this->assertDatabaseHas(InventoryMovement::class, [
            'product_id' => $product1->id,
            'warehouse_id' => $warehouse->id,
            'quantity' => 10,
            'type' => InventoryMovement::TYPE_STOCK_TAKE,
        ]);

        $this->assertDatabaseHas(InventoryMovement::class, [
            'product_id' => $product2->id,
            'warehouse_id' => $warehouse->id,
            'quantity' => 20,
            'type' => InventoryMovement::TYPE_STOCK_TAKE,
        ]);

        $this->assertDatabaseHas(FifoLayer::class, [
            'product_id' => $product1->id,
            'warehouse_id' => $warehouse->id,
            'original_quantity' => 10,
            'total_cost' => 1000,
            'link_type' => StockTakeItem::class,
        ]);

        $this->assertDatabaseHas(FifoLayer::class, [
            'product_id' => $product2->id,
            'warehouse_id' => $warehouse->id,
            'original_quantity' => 20,
            'total_cost' => 4000,
            'link_type' => StockTakeItem::class,
        ]);

        // Update stock take

        $data = [
            ['sku', 'quantity', 'unit_cost'],
            ['sku1', 10, 110],
            ['sku2', 21, 200],
        ];

        $csv = Writer::createFromString('');
        $csv->insertAll($data);
        $csvContent = $csv->toString();

        Storage::disk('model-imports')->put('initial_inventory.csv', $csvContent);

        $response = $this->postJson(route('initial-inventory.import'), [
            'original_name' => 'initial_inventory.csv',
            'stored_name' => 'initial_inventory.csv',
            'meta' => [
                'warehouse_id' => $warehouse->id,
            ]
        ]);

        $response->assertOk();

        $this->assertDatabaseHas(StockTake::class, [
            'warehouse_id' => $warehouse->id,
            'is_initial_count' => true,
            'date_count' => Helpers::setting(Setting::KEY_INVENTORY_START_DATE),
            'value_change' => 5300,
        ]);

        $this->assertDatabaseHas(StockTakeItem::class, [
            'product_id' => $product1->id,
            'qty_counted' => 10,
            'unit_cost' => 110,
            'snapshot_inventory' => 0,
        ]);

        $this->assertDatabaseHas(StockTakeItem::class, [
            'product_id' => $product2->id,
            'qty_counted' => 21,
            'unit_cost' => 200,
            'snapshot_inventory' => 0,
        ]);

        $this->assertDatabaseHas(FifoLayer::class, [
            'product_id' => $product1->id,
            'warehouse_id' => $warehouse->id,
            'original_quantity' => 10,
            'total_cost' => 1100,
            'link_type' => StockTakeItem::class,
        ]);

        $this->assertDatabaseHas(FifoLayer::class, [
            'product_id' => $product2->id,
            'warehouse_id' => $warehouse->id,
            'original_quantity' => 21,
            'total_cost' => 4200,
            'link_type' => StockTakeItem::class,
        ]);

        // Update stock take with impossible decrease

        $salesOrder = app(SalesOrderManager::class)->createOrder([
            'order_status' => SalesOrder::STATUS_OPEN,
            'sales_order_lines' => [
                [
                    'product_id' => $product1->id,
                    'description' => 'Test',
                    'quantity' => 5,
                ],
            ],
        ]);

        $data = [
            ['sku', 'quantity', 'unit_cost'],
            ['sku1', 1, 100],
        ];

        $csv = Writer::createFromString('');
        $csv->insertAll($data);
        $csvContent = $csv->toString();

        Storage::disk('model-imports')->put('initial_inventory.csv', $csvContent);

        $response = $this->postJson(route('initial-inventory.import'), [
            'original_name' => 'initial_inventory.csv',
            'stored_name' => 'initial_inventory.csv',
            'meta' => [
                'warehouse_id' => $warehouse->id,
            ]
        ]);

        $response->assertInternalServerError();

        $salesOrder->delete();

        // Update stock take with decreases

        $data = [
            ['sku', 'quantity', 'unit_cost'],
            ['sku1', 1, 100],
            ['sku2', 1, 200],
        ];

        $csv = Writer::createFromString('');
        $csv->insertAll($data);
        $csvContent = $csv->toString();

        Storage::disk('model-imports')->put('initial_inventory.csv', $csvContent);

        $response = $this->postJson(route('initial-inventory.import'), [
            'original_name' => 'initial_inventory.csv',
            'stored_name' => 'initial_inventory.csv',
            'meta' => [
                'warehouse_id' => $warehouse->id,
            ]
        ]);

        $response->assertOk();

        $this->assertDatabaseHas(StockTake::class, [
            'warehouse_id' => $warehouse->id,
            'is_initial_count' => true,
            'date_count' => Helpers::setting(Setting::KEY_INVENTORY_START_DATE),
            'value_change' => 300,
        ]);

        $this->assertDatabaseHas(StockTakeItem::class, [
            'product_id' => $product1->id,
            'qty_counted' => 1,
            'unit_cost' => 100,
            'snapshot_inventory' => 0,
        ]);

        $this->assertDatabaseHas(StockTakeItem::class, [
            'product_id' => $product2->id,
            'qty_counted' => 1,
            'unit_cost' => 200,
            'snapshot_inventory' => 0,
        ]);

        $this->assertDatabaseHas(FifoLayer::class, [
            'product_id' => $product1->id,
            'warehouse_id' => $warehouse->id,
            'original_quantity' => 1,
            'total_cost' => 100,
            'link_type' => StockTakeItem::class,
        ]);

        $this->assertDatabaseHas(FifoLayer::class, [
            'product_id' => $product2->id,
            'warehouse_id' => $warehouse->id,
            'original_quantity' => 1,
            'total_cost' => 200,
            'link_type' => StockTakeItem::class,
        ]);
    }
}
