<?php

namespace Tests\Feature;

use App\Http\Requests\StoreInventoryAdjustment;
use App\Jobs\SubmitFulfillmentToThirdPartiesJob;
use App\Models\InventoryMovement;
use App\Models\Product;
use App\Models\SalesOrder;
use App\Models\SalesOrderFulfillment;
use App\Models\SalesOrderFulfillmentLine;
use App\Models\SalesOrderLine;
use App\Models\Supplier;
use App\Models\User;
use App\Models\Warehouse;
use Illuminate\Support\Facades\Queue;
use JetBrains\PhpStorm\NoReturn;
use Laravel\Sanctum\Sanctum;
use Plannr\Laravel\FastRefreshDatabase\Traits\FastRefreshDatabase;
use Tests\TestCase;

class BulkSalesOrderFulfillmentTest extends TestCase
{
    use FastRefreshDatabase;

    private function createSalesOrders(): array{
        $warehouse = Warehouse::factory()->create()->withDefaultLocation();
        $product = Product::factory()->create();

        $this->postJson('/api/inventory-adjustments', [
            'warehouse_id' => $warehouse->id,
            'product_id' => $product->id,
            'quantity' => 10,
            'adjustment_date' => now(),
            'adjustment_type' => StoreInventoryAdjustment::ADJUSTMENT_TYPE_INCREASE,
            'unit_cost' => 5
        ])->assertSuccessful();


        $payload = [
            'currency_code' => 'USD',
            'order_date' => now(),
            'order_status' => SalesOrder::STATUS_OPEN,
        ];


        $order1Response = $this->postJson('/api/sales-orders', array_merge($payload, [
            'sales_order_number' => 'SO-0001',
            'sales_order_lines' => [
                [
                    'product_id' => $product->id,
                    'quantity' => 2,
                    'description' => $product->name,
                    'amount' => 5,
                    'warehouse_id' => $warehouse->id,
                ]
            ]
        ]))->assertSuccessful()->json('data');


        $order2Response = $this->postJson('/api/sales-orders', array_merge($payload, [
            'sales_order_number' => 'SO-0002',
            'sales_order_lines' => [
                [
                    'product_id' => $product->id,
                    'quantity' => 3,
                    'description' => $product->name,
                    'amount' => 10,
                    'warehouse_id' => $warehouse->id,
                ]
            ]
        ]))->assertSuccessful()->json('data');

        return [$order1Response, $order2Response];
    }

    #[NoReturn] public function test_it_can_bulk_fulfill_sales_orders_by_ids(): void{

        Sanctum::actingAs(User::factory()->create());

        Queue::fake();

        [$order1Response, $order2Response] = $this->createSalesOrders();

        $response = $this->postJson('/api/sales-orders/fulfill', [
            'ids' => [
                $order1Response['id'],
                $order2Response['id']
            ]
        ])->assertSuccessful();

        $this->assertDatabaseCount('sales_order_fulfillments', 2);
        $this->assertDatabaseHas('sales_order_fulfillment_lines', [
            'sales_order_line_id' => $order1Response['item_info'][0]['sales_order_line_id'],
            'quantity' => 2
        ]);

        $this->assertDatabaseHas('sales_order_fulfillment_lines', [
            'sales_order_line_id' => $order2Response['item_info'][0]['sales_order_line_id'],
            'quantity' => 3
        ]);

        $this->assertDatabaseHas((new SalesOrderLine())->getTable(), [
            'id' => $order1Response['item_info'][0]['sales_order_line_id'],
            'fulfilled_quantity' => 2
        ]);

        $this->assertDatabaseHas('sales_order_lines', [
            'id' => $order2Response['item_info'][0]['sales_order_line_id'],
            'fulfilled_quantity' => 3
        ]);

        $this->assertDatabaseCount('inventory_movements', 7);

        Queue::assertPushed(SubmitFulfillmentToThirdPartiesJob::class);
    }

    public function test_it_can_fulfill_bulk_orders_using_filters(): void{
        Sanctum::actingAs(User::factory()->create());

        Queue::fake();

        [$order1Response] = $this->createSalesOrders();

        // Only fulfill order 1 via filters
        $this->postJson('/api/sales-orders/fulfill', [
            'filters' => [
                'conjunction' => 'and',
                'filterSet' => [
                    [
                        'column' => 'id',
                        'operator' => '=',
                        'value' => [
                            $order1Response['id']
                        ],
                    ],
                ],
            ]
        ])->assertSuccessful();

        $this->assertDatabaseCount('sales_order_fulfillments', 1);
        $this->assertDatabaseHas('sales_order_fulfillment_lines', [
            'sales_order_line_id' => $order1Response['item_info'][0]['sales_order_line_id'],
            'quantity' => 2
        ]);

        $this->assertDatabaseHas('sales_order_lines', [
            'id' => $order1Response['item_info'][0]['sales_order_line_id'],
            'fulfilled_quantity' => 2
        ]);

        $this->assertDatabaseCount('inventory_movements', 6);

        Queue::assertPushed(SubmitFulfillmentToThirdPartiesJob::class);
    }

    public function test_it_can_fulfill_bulk_orders_with_dropship_lines(): void
    {
        Sanctum::actingAs(User::factory()->create());

        Queue::fake();

        $supplier = Supplier::factory()->create([
            'auto_fulfill_dropship' => true, // Ensures that dropship PO gets created.
        ]);
        $warehouse = Warehouse::factory()->create([
            'supplier_id' => $supplier->id,
            'dropship_enabled' => true
        ])->withDefaultLocation();
        $product   = Product::factory()->create();
        $product2 = Product::factory()->create();
        $warehouse2 = Warehouse::factory()->create()->withDefaultLocation();

        $response = $this->postJson('/api/sales-orders', [
            'currency_code' => 'USD',
            'order_date' => now(),
            'order_status' => SalesOrder::STATUS_OPEN,
            'sales_order_number' => 'SO-0001',
            'sales_order_lines' => [
                [
                    'product_id' => $product->id,
                    'quantity' => 2,
                    'description' => $product->name,
                    'amount' => 5,
                    'warehouse_id' => $warehouse->id,
                    'warehouse_routing_method' => 'dropship'
                ],
                [
                    'product_id' => $product2->id,
                    'quantity' => 4,
                    'description' => $product2->name,
                    'amount' => 5,
                    'warehouse_id' => $warehouse2->id,
                    'warehouse_routing_method' => 'warehouse'
                ]
            ]
        ])->assertSuccessful();

        $this->postJson('/api/sales-orders/fulfill', [
            'ids' => [$response->json('data.id')]
        ])->assertSuccessful();

        $this->assertDatabaseCount('sales_order_fulfillments', 1);
        $this->assertDatabaseCount('sales_order_fulfillment_lines', 1);
        $this->assertDatabaseHas('sales_order_fulfillment_lines', [
            'sales_order_line_id' => $response->json('data.item_info')[0]['sales_order_line_id'],
            'quantity' => 2
        ]);

        $this->assertDatabaseHas('sales_order_lines', [
            'id' => $response->json('data.item_info')[0]['sales_order_line_id'],
            'fulfilled_quantity' => 2
        ]);

    }

    public function test_it_can_bulk_fulfill_sales_order_with_lines__linked_to_multiple_fifo_layers(): void{
        Sanctum::actingAs(User::factory()->create());

        Queue::fake();

        $warehouse = Warehouse::factory()->create()->withDefaultLocation();
        $product = Product::factory()->create();

        $this->postJson(route('inventory-adjustments.store'), [
            'warehouse_id' => $warehouse->id,
            'product_id' => $product->id,
            'quantity' => 1,
            'adjustment_date' => now(),
            'adjustment_type' => StoreInventoryAdjustment::ADJUSTMENT_TYPE_INCREASE,
            'unit_cost' => 5
        ])->assertSuccessful();

        $this->postJson(route('inventory-adjustments.store'), [
            'warehouse_id' => $warehouse->id,
            'product_id' => $product->id,
            'quantity' => 1,
            'adjustment_date' => now(),
            'adjustment_type' => StoreInventoryAdjustment::ADJUSTMENT_TYPE_INCREASE,
            'unit_cost' => 10
        ])->assertSuccessful();

        $payload = [
            'currency_code' => 'USD',
            'order_date' => now(),
            'order_status' => SalesOrder::STATUS_OPEN,
        ];

        $response = $this->postJson(route('sales-orders.store'), array_merge($payload, [
            'sales_order_number' => 'SO-0001',
            'sales_order_lines' => [
                [
                    'product_id' => $product->id,
                    'quantity' => 2,
                    'description' => $product->name,
                    'amount' => 5,
                    'warehouse_id' => $warehouse->id,
                ]
            ]
        ]))->assertSuccessful()->json('data');

        $this->postJson(route('sales-orders.bulk-fulfill'), [
            'ids' => [$response['id']]
        ])->assertSuccessful();

        $this->assertDatabaseCount((new SalesOrderFulfillment())->getTable(), 1);

        $this->assertDatabaseCount((new SalesOrderFulfillmentLine())->getTable(), 1);
        $this->assertDatabaseHas((new SalesOrderFulfillmentLine())->getTable(), [
            'sales_order_line_id' => $response['item_info'][0]['sales_order_line_id'],
            'quantity' => 2
        ]);

        $this->assertDatabaseHas((new SalesOrderLine())->getTable(), [
            'id' => $response['item_info'][0]['sales_order_line_id'],
            'fulfilled_quantity' => 2
        ]);

        $this->assertDatabaseCount((new InventoryMovement())->getTable(), 8);
        $this->assertDatabaseHas((new InventoryMovement())->getTable(), [
            'reference' => 'SO-0001.1', // Reference
            'link_type' => SalesOrderFulfillmentLine::class,
        ]);
    }

}
