<?php

namespace Tests\Feature;

use App\Exceptions\PurchaseOrder\NotOpenPurchaseOrderException;
use App\Exceptions\PurchaseOrder\ReceivePurchaseOrderLineException;
use App\Helpers;
use App\Http\Requests\StoreInventoryAdjustment;
use App\Http\Resources\SalesOrderLineDetailResource;
use App\Jobs\SyncBackorderQueueCoveragesJob;
use App\Models\BackorderQueue;
use App\Models\BackorderQueueCoverage;
use App\Models\BackorderQueueRelease;
use App\Models\InventoryAdjustment;
use App\Models\InventoryMovement;
use App\Models\Product;
use App\Models\PurchaseOrder;
use App\Models\PurchaseOrderLine;
use App\Models\PurchaseOrderShipmentReceiptLine;
use App\Models\SalesChannel;
use App\Models\SalesOrder;
use App\Models\SalesOrderLine;
use App\Models\Supplier;
use App\Models\SupplierProduct;
use App\Models\User;
use App\Models\Warehouse;
use App\Services\InventoryManagement\BackorderManager;
use App\Services\InventoryManagement\BulkInventoryManager;
use App\Services\SalesOrder\LineReservationAction;
use Carbon\Carbon;
use Database\Factories\FactoryDataRecycler;
use Illuminate\Contracts\Container\BindingResolutionException;
use Illuminate\Foundation\Testing\DatabaseTransactions;
use Illuminate\Foundation\Testing\WithFaker;
use Illuminate\Support\Arr;
use Plannr\Laravel\FastRefreshDatabase\Traits\FastRefreshDatabase;
use Laravel\Sanctum\Sanctum;
use Queue;
use Tests\Concerns\UseSimpleSkuFactories;
use Tests\TestCase;

class BackorderManagerTest extends TestCase
{
    use FastRefreshDatabase;
    use UseSimpleSkuFactories;
    use WithFaker;


    public function test_it_removes_backorders_and_releases_when_fifo_is_taken_from_line(){

        Sanctum::actingAs(User::factory()->create());

        $product = Product::factory()->create();
        $warehouse = Warehouse::factory()->create()->withDefaultLocation();

        // Create sales order
        $response = $this->postJson('/api/sales-orders', [
            'order_date' => now(),
            'order_status' => SalesOrder::STATUS_OPEN,
            'currency_code' => 'USD',
            'sales_order_lines' => [
                [
                    'product_id' => $product->id,
                    'amount' => 10,
                    'description' => $product->name,
                    'quantity' => 1,
                    'warehouse_id' => $warehouse->id,
                ]
            ]
        ])->assertSuccessful();

        // Unreleased backorder queue must exist.
        $this->assertDatabaseHas('backorder_queues', [
            'sales_order_line_id' => $response->json('data.item_info.0.sales_order_line_id'),
            'backordered_quantity' => 1,
            'released_quantity' => 0
        ]);

        // Release the backorder with an inventory adjustment.
        $this->postJson('/api/inventory-adjustments', [
            'adjustment_date' => now(),
            'product_id' => $product->id,
            'quantity' => 1,
            'unit_cost' => 5,
            'adjustment_type' => StoreInventoryAdjustment::ADJUSTMENT_TYPE_INCREASE,
            'warehouse_id' => $warehouse->id,
        ])->assertSuccessful();
        // Backorder queue must be released
        $this->assertDatabaseHas('backorder_queues', [
            'sales_order_line_id' => $response->json('data.item_info.0.sales_order_line_id'),
            'backordered_quantity' => 1,
            'released_quantity' => 1
        ]);
        $this->assertDatabaseHas('backorder_queue_releases', [
            'released_quantity' => 1,
            'link_type' => InventoryAdjustment::class,
        ]);

        // Create negative adjustment to take stock from sales order line
        $this->postJson('/api/inventory-adjustments', [
            'adjustment_date' => now(),
            'product_id' => $product->id,
            'quantity' => 1,
            'adjustment_type' => StoreInventoryAdjustment::ADJUSTMENT_TYPE_DECREASE,
            'warehouse_id' => $warehouse->id,
        ])->assertSuccessful();

        // Backorder queue must exist
        $this->assertDatabaseCount('backorder_queues', 1);
        $this->assertDatabaseHas('backorder_queues', [
            'sales_order_line_id' => $response->json('data.item_info.0.sales_order_line_id'),
            'backordered_quantity' => 1,
            'released_quantity' => 0
        ]);

        // Release must be deleted
        $this->assertDatabaseEmpty('backorder_queue_releases');

    }


    public function test_it_removes_backorders_and_releases_when_line_is_unreserved(){

        Sanctum::actingAs(User::factory()->create());

        $product = Product::factory()->create();
        $warehouse = Warehouse::factory()->create()->withDefaultLocation();

        // Create sales order
        $response = $this->postJson('/api/sales-orders', [
            'order_date' => now(),
            'order_status' => SalesOrder::STATUS_OPEN,
            'currency_code' => 'USD',
            'sales_order_lines' => [
                [
                    'product_id' => $product->id,
                    'amount' => 10,
                    'description' => $product->name,
                    'quantity' => 1,
                    'warehouse_id' => $warehouse->id,
                ]
            ]
        ])->assertSuccessful();

        // Unreleased backorder queue must exist.
        $this->assertDatabaseHas('backorder_queues', [
           'sales_order_line_id' => $response->json('data.item_info.0.sales_order_line_id'),
           'backordered_quantity' => 1,
           'released_quantity' => 0
        ]);

        // Release the backorder with an inventory adjustment.
        $this->postJson('/api/inventory-adjustments', [
            'adjustment_date' => now(),
            'product_id' => $product->id,
            'quantity' => 1,
            'unit_cost' => 5,
            'adjustment_type' => StoreInventoryAdjustment::ADJUSTMENT_TYPE_INCREASE,
            'warehouse_id' => $warehouse->id,
        ])->assertSuccessful();
        // Backorder queue must be released
        $this->assertDatabaseHas('backorder_queues', [
            'sales_order_line_id' => $response->json('data.item_info.0.sales_order_line_id'),
            'backordered_quantity' => 1,
            'released_quantity' => 1
        ]);
        $this->assertDatabaseHas('backorder_queue_releases', [
            'released_quantity' => 1,
            'link_type' => InventoryAdjustment::class,
        ]);

        // Un-reserve the sales order line.
        $this->postJson('/api/sales-order-lines/reservations', [
            'sales_order_id' => $response->json('data.id'),
            'sales_order_lines' => [
                [
                    'id' => $response->json('data.item_info.0.sales_order_line_id'),
                    'quantity' => 1,
                    'action' => LineReservationAction::REVERSE_RESERVATION
                ]
            ]
        ])->assertSuccessful();

        // Backorder queue and releases must be updated/deleted.
        $this->assertDatabaseEmpty('backorder_queues');
        $this->assertDatabaseEmpty('backorder_queue_releases');


    }

    /**
     * @throws BindingResolutionException
     */
    public function test_it_can_cover_backorders(): void
    {

        Queue::fake();
        // make backorders exist by creating sales orders out of stock
        // make purchase orders exist that are unreceived
        // run coverBackorderQueues

        // make assertions

        // Create direct warehouses
        $warehouse = Warehouse::factory(1)->direct()->create();

        // Create suppliers
        $supplier = Supplier::factory(1)->create();

        // Create products, all associated with suppliers
        $products = Product::factory(1)
            ->withProductPricing()
            ->withSupplierProduct(
                (new FactoryDataRecycler([
                    $supplier,
                ]))
            )
            ->create();

        $purchaseOrder = PurchaseOrder::factory(3)
            ->withLines(
                1,
                (new FactoryDataRecycler())->addRecycledData($products, null, false)
            )
            /*
             * This is Laravel's built-in recycle method.  We need to explore the differences in functionality between
             * my FactoryDataRecycler class and Laravel's recycle method.
             */
            //->recycle($suppliers)
            ->factoryDataRecycler(
                (new FactoryDataRecycler())
                    ->addRecycledData($warehouse, 'destination_warehouse_id')
                    ->addRecycledData($supplier)
            )
            ->create();

        // Create sales orders
        SalesOrder::factory(10)->withLines(
            1,
            (new FactoryDataRecycler())
                ->addRecycledData($products, null, true)
                ->addRecycledData($warehouse)
        )->open()
            ->create([
                'sales_channel_id' => SalesChannel::query()->first()->id,
                'order_date' => $this->faker->dateTimeThisYear(),
            ]);

        (new BulkInventoryManager())->bulkAllocateNegativeInventoryEvents(SalesOrderLine::all());

        $this->assertDatabaseHas((new InventoryMovement())->getTable(), [
            'link_type' => SalesOrderLine::class,
            'type' => InventoryMovement::TYPE_SALE,
            'inventory_status' => InventoryMovement::INVENTORY_STATUS_ACTIVE,
        ]);

        /** @var PurchaseOrder $lastPurchaseOrder */
        $latestPurchaseOrderWithSoonestETD = PurchaseOrder::query()
            ->orderBy('purchase_order_date', 'desc')
            ->first();
        $latestPurchaseOrderWithSoonestETD->estimated_delivery_date = now()->subDays(300);
        $latestPurchaseOrderWithSoonestETD->save();
        $latestPurchaseOrderWithSoonestETDLine = $latestPurchaseOrderWithSoonestETD->purchaseOrderLines->first();

        $firstPriorityBackorder = BackorderQueue::query()
            ->orderBy('priority')
            ->first();

        $this->assertDatabaseEmpty((new BackorderQueueCoverage())->getTable());

        (new BackorderManager())->withGlobalCoverage()->coverBackorderQueues();

        $this->assertDatabaseHas((new BackorderQueueCoverage())->getTable(), [
            'backorder_queue_id' => $firstPriorityBackorder->id,
            'purchase_order_line_id' => $latestPurchaseOrderWithSoonestETDLine->id,
        ]);
    }

    public function test_it_wont_overcover_backorders(): void
    {

        Queue::fake();
        // make backorders exist by creating sales orders out of stock
        // make purchase orders exist that are unreceived
        // run coverBackorderQueues

        // make assertions

        // Create direct warehouses
        $warehouse = Warehouse::factory()->create();

        // Create suppliers
        $supplier = Supplier::factory()->create();

        $product = Product::factory()->create();

        // Create sales orders
        SalesOrder::factory(1)->has(
            SalesOrderLine::factory(1, [
                'product_id' => $product->id,
                'quantity' => 3,
                'warehouse_id' => $warehouse->id,
            ])
        )->open()
            ->create([
                'sales_channel_id' => SalesChannel::query()->first()->id,
                'order_date' => $this->faker->dateTimeThisYear(),
            ]);

        (new BulkInventoryManager())->bulkAllocateNegativeInventoryEvents(SalesOrderLine::all());

        PurchaseOrder::factory()
            ->has(PurchaseOrderLine::factory(1, [
                'product_id' => $product->id,
                'quantity' => 3,
            ]))->create([
                'supplier_id' => $supplier->id,
                'destination_warehouse_id' => $warehouse->id,
            ]);

        $firstPurchaseOrderLine = PurchaseOrderLine::first();

        $firstPriorityBackorder = BackorderQueue::query()
            ->orderBy('priority')
            ->first();

        $this->assertDatabaseEmpty((new BackorderQueueCoverage())->getTable());

        (new BackorderManager())->withGlobalCoverage()->coverBackorderQueues();

        $this->assertEquals(1, BackorderQueueCoverage::query()->count());

        PurchaseOrder::factory()
            ->has(PurchaseOrderLine::factory(1, [
                'product_id' => $product->id,
                'quantity' => 1,
            ]))->create([
                'supplier_id' => $supplier->id,
                'destination_warehouse_id' => $warehouse->id,
            ]);

        $secondPurchaseOrderLine = PurchaseOrderLine::query()->orderBy('id', 'desc')->first();

        (new BackorderManager())->coverBackorderQueues(Arr::wrap($secondPurchaseOrderLine->id));

        $this->assertEquals(1, BackorderQueueCoverage::query()->count());
    }

    public function test_sync_backorder_queue_coverages_job_only_dispatched_once_for_po_approval(): void
    {
        Queue::fake();

        $purchaseOrder = PurchaseOrder::factory()
            ->has(
                PurchaseOrderLine::factory()
            )
            ->create([
                'order_status' => PurchaseOrder::STATUS_DRAFT,
            ]);

        $purchaseOrder->approve();

        Queue::assertPushed(SyncBackorderQueueCoveragesJob::class, 1);
    }

    /**
     * @throws BindingResolutionException
     * @throws NotOpenPurchaseOrderException
     * @throws ReceivePurchaseOrderLineException
     */
    public function test_it_can_get_backorder_history(): void
    {
        $backorderManager = app(BackorderManager::class)->withGlobalCoverage();

        // Create backorder that is partially covered, partially released

        $this->setUpWarehouse();
        $this->setUpProduct();
        $this->setUpSalesOrders(10);
        $purchaseOrder = $this->setUpPurchaseOrders(8)->first();

        $supplier = $purchaseOrder->supplier;

        // Set up supplier product
        SupplierProduct::factory()->create([
            'product_id' => $this->product->id,
            'supplier_id' => $supplier->id,
        ]);

        // Set backorder po schedule for supplier

        $supplier->auto_generate_backorder_po = true;
        $supplier->backorder_po_schedule = [
            [
                'time' => '11:00',
                'days' => ['wednesday'],
            ],
        ];
        $supplier->save();

        // Reserve lines
        app(BulkInventoryManager::class)->bulkAllocateNegativeInventoryEvents(SalesOrderLine::all());

        $backorderQueue = BackorderQueue::query()->first();

        // Needs coverage
        $backorderManager->coverBackorderQueues();

        $this->assertDatabaseHas((new BackorderQueueCoverage())->getTable(), [
            'backorder_queue_id' => $backorderQueue->id,
            'purchase_order_line_id' => PurchaseOrderLine::query()->first()->id,
            'covered_quantity' => 8,
        ]);

        // Partially receive
        $this->receivePurchaseOrder($purchaseOrder, 3);

        $this->assertDatabaseHas((new BackorderQueueRelease())->getTable(), [
            'backorder_queue_id' => $backorderQueue->id,
            'link_type' => PurchaseOrderShipmentReceiptLine::class,
            'link_id' => PurchaseOrderShipmentReceiptLine::query()->first()->id,
            'released_quantity' => 3,
        ]);

        $salesOrderLine = SalesOrderLine::with(
            'backorderQueue.backorderQueueCoverages.purchaseOrderLine.purchaseOrder',
            'backorderQueue.backorderQueueReleases',
            'backorderQueue.supplier'
        )->first();

        $history = SalesOrderLineDetailResource::make($salesOrderLine);

        $this->assertEquals([
                'covered_quantity' => 8,
                'released_quantity' => 3,
                'unreleased_quantity' => 5,
            ],
            $history->backorderQueue->backorderQueueCoverages
                ->first()
                ->only(['covered_quantity', 'released_quantity', 'unreleased_quantity'])
        );

        $this->assertEquals([
            'released_quantity' => 3,
        ],
            $history->backorderQueue->backorderQueueCoverages
                ->first()
                ->only(['released_quantity'])
        );

        $schedule = $history->backorderQueue->getSchedule();
        unset($schedule['scheduled_at']);

        $this->assertEquals([
            'quantity' => 2,
            'supplier' => [
                'id' => $supplier->id,
                'name' => $supplier->name,
            ],
// TODO: This is causing problems with test because sometimes you need to add a week and sometimes you don't
//            'scheduled_at' => Carbon::parse('wednesday 11:00', Helpers::getAppTimezone())
//                ->setTimezone('UTC')
//                ->addWeek(),
        ], $schedule);
    }
}
