<?php

namespace Tests\Feature;

use App\Models\Product;
use App\Models\ProductListing;
use App\Models\PurchaseOrder;
use App\Models\PurchaseOrderLine;
use App\Models\User;
use App\Models\Warehouse;
use App\Models\WarehouseTransfer;
use App\Models\WarehouseTransferLine;
use Carbon\Carbon;
use Exception;
use Illuminate\Foundation\Testing\WithFaker;
use Laravel\Sanctum\Sanctum;
use Modules\Amazon\Actions\InitializeFbaWarehouse;
use Modules\Amazon\Entities\AmazonFbaReportRemovalOrder;
use Modules\Amazon\Entities\AmazonFbaReportRemovalShipment;
use Modules\Amazon\Entities\AmazonIntegrationInstance;
use Modules\Amazon\Entities\AmazonProduct;
use Modules\Amazon\Entities\AmazonReport;
use Modules\Amazon\Enums\Entities\FbaRemovalOrderTypeEnum;
use Modules\Amazon\Enums\Entities\AmazonReportTypeEnum;
use Modules\Amazon\Managers\AmazonRemovalOrderManager;
use Plannr\Laravel\FastRefreshDatabase\Traits\FastRefreshDatabase;
use Tests\TestCase;

class AdvancedShipmentNoticeControllerTest extends TestCase
{
    use FastRefreshDatabase;
    use WithFaker;

    protected function setUp(): void
    {
        parent::setUp();
        Sanctum::actingAs(User::first());
    }

    public function test_user_can_send_asn_for_purchase_order(): void
    {
        $purchaseOrder = PurchaseOrder::factory()->has(
            PurchaseOrderLine::factory()->withSupplierProduct()->count(3)->state(['quantity' => 2, 'amount' => 55.00])
        )->state(
            ['destination_warehouse_id' => Warehouse::factory()->state(['type' => Warehouse::TYPE_3PL])]
        )->create();

        $purchaseOrder->load('purchaseOrderLines.product.defaultSupplierProduct.supplier.address',
            'destinationWarehouse.address');

        $this->assertEquals(Warehouse::TYPE_3PL, $purchaseOrder->destinationWarehouse->type);

        $this->assertDatabaseCount(PurchaseOrder::class, 1);
        $this->assertDatabaseCount(PurchaseOrderLine::class, 3);

        $this->assertNotNull($purchaseOrder->destinationWarehouse->address->email);

        $this->post(route('purchase-orders.asn', $purchaseOrder->id))->assertOk();
    }

    public function test_user_can_send_asn_for_warehouse_transfer(): void
    {
        $warehouseTransfer = WarehouseTransfer::factory()->has(
            WarehouseTransferLine::factory()->withSupplierProduct()->count(3)
        )->state([
            'to_warehouse_id' => Warehouse::factory()->state(['type' => Warehouse::TYPE_3PL])
        ])->create();

        $warehouseTransfer->load('warehouseTransferLines.product.defaultSupplierProduct.supplier.address',
            'toWarehouse.address');

        $this->assertEquals(Warehouse::TYPE_3PL, $warehouseTransfer->toWarehouse->type);

        $this->assertDatabaseCount(WarehouseTransfer::class, 1);
        $this->assertDatabaseCount(WarehouseTransferLine::class, 3);

        $this->assertNotNull($warehouseTransfer->toWarehouse->address->email);

        $this->post(route('warehouses.transfers.asn', $warehouseTransfer->id))->assertOk();
    }

    public function test_user_can_send_asn_for_removal_order(): void
    {
        $warehouseTransfer = $this->setUpRemovalOrdersReport();

        $warehouseTransfer->load('warehouseTransferLines.product.defaultSupplierProduct.supplier.address',
            'toWarehouse.address');

        $this->assertEquals(Warehouse::TYPE_AMAZON_FBA, $warehouseTransfer->fromWarehouse->type);
        $this->assertEquals(Warehouse::TYPE_3PL, $warehouseTransfer->toWarehouse->type);

        $this->assertDatabaseCount(WarehouseTransfer::class, 1);

        $this->assertNotNull($warehouseTransfer->toWarehouse->address->email);

        $this->post(route('warehouses.transfers.asn', $warehouseTransfer->id))->assertOk();
    }

    /**
     * @throws Exception
     */
    private function setUpRemovalOrdersReport(): WarehouseTransfer
    {
        $amazonIntegrationInstance = AmazonIntegrationInstance::factory()->hasSalesChannel()->create();
        (new InitializeFbaWarehouse($amazonIntegrationInstance))->handle();
        $integrationSettings = $amazonIntegrationInstance->integration_settings;
        $integrationSettings['automatically_create_warehouse_transfers_from_removal_orders'] = true;
        $integrationSettings['removal_order_warehouse_id'] = Warehouse::factory()->create(['type' => Warehouse::TYPE_3PL])->id;
        $amazonIntegrationInstance->integration_settings = $integrationSettings;
        $amazonIntegrationInstance->save();
        $amazonIntegrationInstance->refresh();

        /** @var AmazonReport $amazonReportForRemovalOrders */
        $amazonReportForRemovalOrders = AmazonReport::factory()->create([
            'reportType' => AmazonReportTypeEnum::FBA_REPORT_REMOVAL_ORDERS,
        ]);


        $date = Carbon::parse($this->faker->date());
        AmazonProduct::factory()->create()->each(function (AmazonProduct $amazonProduct) use ($amazonIntegrationInstance, $amazonReportForRemovalOrders, $date) {
            $jsonObject = $amazonProduct->json_object;
            $jsonObject['fulfillment_channel'] = $amazonIntegrationInstance->getFbaFulfillmentChannel();
            $amazonProduct->json_object = $jsonObject;
            $amazonProduct->save();

            $amazonProduct->refresh();

            $product = Product::factory()->create([
                'sku' => $amazonProduct->seller_sku,
            ]);

            ProductListing::factory()->create([
                'product_id' => $product->id,
                'document_id' => $amazonProduct->id,
                'document_type' => AmazonProduct::class,
            ]);
            $orderId = $this->faker->word();

            AmazonFbaReportRemovalOrder::factory()->create([
                'amazon_report_id' => $amazonReportForRemovalOrders->id,
                'json_object' => [
                    'request_date' => $date->toIso8601String(),
                    'order_id' => $orderId,
                    'sku' => $product->sku,
                    'fnsku' => $this->faker->unique()->word(),
                    'order_type' => FbaRemovalOrderTypeEnum::Return(),
                    'order_status' => 'Completed',
                    'last_updated_date' => $date->toIso8601String(),
                    'requested_quantity' => 1,
                    'disposition' => $this->faker->word(),
                ],
            ]);

            AmazonFbaReportRemovalShipment::factory()->create([
                'amazon_report_id' => $amazonReportForRemovalOrders->id,
                'json_object' => [
                    'shipment_date' => $date->toIso8601String(),
                    'order_id' => $orderId,
                    'sku' => $product->sku,
                    'fnsku' => $this->faker->word(),
                    'disposition' => $this->faker->word(),
                    'shipped_quantity' => $this->faker->randomNumber(),
                    'tracking_number' => $this->faker->word(),
                ]
            ]);
        });

        $removalOrders = AmazonFbaReportRemovalOrder::all();

        (new AmazonRemovalOrderManager($amazonIntegrationInstance))->process($removalOrders->pluck('id')->toArray());

        return WarehouseTransfer::first();
    }
}