<?php

namespace Tests\Feature;

use App\Data\FinancialLineData;
use App\Data\WarehouseTransferReceiptData;
use App\Data\WarehouseTransferReceiptProductData;
use App\Enums\AccountingTransactionTypeEnum;
use App\Enums\FinancialLineClassificationEnum;
use App\Enums\FinancialLineProrationStrategyEnum;
use App\Exceptions\InsufficientStockException;
use App\Exceptions\WarehouseTransfers\WarehouseTransferHasNoProductsException;
use App\Exceptions\WarehouseTransfers\WarehouseTransferOpenException;
use App\Helpers;
use App\Jobs\Accounting\SyncAccountingTransactionsJob;
use App\Managers\WarehouseTransferManager;
use App\Models\AccountingTransaction;
use App\Models\AccountingTransactionLine;
use App\Models\Customer;
use App\Models\FinancialLine;
use App\Models\FinancialLineType;
use App\Models\InventoryAdjustment;
use App\Models\InventoryMovement;
use App\Models\NominalCode;
use App\Models\Product;
use App\Models\PurchaseInvoice;
use App\Models\PurchaseInvoiceLine;
use App\Models\PurchaseOrder;
use App\Models\PurchaseOrderLine;
use App\Models\PurchaseOrderShipmentReceipt;
use App\Models\PurchaseOrderShipmentReceiptLine;
use App\Models\SalesChannel;
use App\Models\SalesCredit;
use App\Models\SalesCreditLine;
use App\Models\SalesCreditReturn;
use App\Models\SalesCreditReturnLine;
use App\Models\SalesOrder;
use App\Models\SalesOrderFulfillment;
use App\Models\SalesOrderFulfillmentLine;
use App\Models\SalesOrderLine;
use App\Models\Setting;
use App\Models\StockTake;
use App\Models\StockTakeItem;
use App\Models\Supplier;
use App\Models\Warehouse;
use App\Models\WarehouseTransfer;
use App\Models\WarehouseTransferShipment;
use App\Models\WarehouseTransferShipmentLine;
use App\Models\WarehouseTransferShipmentReceipt;
use App\Models\WarehouseTransferShipmentReceiptLine;
use App\Repositories\Accounting\AccountingTransactionRepository;
use App\Repositories\SalesOrder\SalesOrderRepository;
use App\Repositories\SettingRepository;
use App\Services\Accounting\AccountingTransactionManager;
use App\Services\FinancialManagement\SalesOrderLineFinancialManager;
use Carbon\Carbon;
use Carbon\CarbonImmutable;
use DB;
use Exception;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Support\Facades\Queue;
use Modules\Amazon\Entities\AmazonIntegrationInstance;
use Plannr\Laravel\FastRefreshDatabase\Traits\FastRefreshDatabase;
use Tests\TestCase;
use Throwable;
use function PHPUnit\Framework\assertEquals;

class AccountingTransactionManagerTest extends TestCase
{
    use FastRefreshDatabase;

    private AccountingTransactionManager $manager;
    
    public function setUp(): void
    {
        $this->manager = app(AccountingTransactionManager::class);
        parent::setUp(); // TODO: Change the autogenerated stub
    }

    private function resetUpdatedAt(AccountingTransaction $model): void
    {
        $model->refresh();
        $model->updated_at = Carbon::now()->subDay();
        $model->save();
    }

    public function test_it_can_dispatch_sync_accounting_transactions_from_command(): void
    {
        Queue::fake();

        $this->artisan('sku:accounting:transactions:sync');

        Queue::assertPushed(SyncAccountingTransactionsJob::class);
    }

    /**
     * @throws Exception
     * @throws Throwable
     */
    public function test_accounting_transactions_when_link_changes(): void
    {
        Queue::fake();

        $this->testSalesOrders();
        $this->testSalesCredits();
        $this->testSalesOrderFulfillments();
        $this->testPurchaseOrders();
        $this->testPurchaseOrderInvoices();
        $this->testPurchaseReceipt();
        $this->testAdjustments();
        $this->testCustomerReturns();
        $this->testStockTakes();
        $this->testWarehouseTransferShipments();
        $this->testWarehouseTransferShipmentReceipts();
    }

    /**
     * @throws Exception
     * @throws Throwable
     */
    public function testSalesOrders(): void
    {
        $salesOrder = SalesOrder::factory()->hasSalesOrderLines(3)
            ->create([
                'sales_order_number' => 'pre_update',
                'is_tax_included' => true,
            ]);

        $this->manager->sync();

        $this->assertEquals(
            1,
            AccountingTransaction::withSalesOrders($salesOrder->id)
                ->where('type', AccountingTransaction::TYPE_SALES_ORDER_INVOICE)
                ->count()
        );

        $this->assertEquals(
            3,
            AccountingTransactionLine::withSalesOrderLines(
                $salesOrder->salesOrderLines()
                    ->pluck('id')
                    ->toArray()
            )
                ->where('type', AccountingTransactionLine::TYPE_SALES_INVOICE_LINE)
                ->count()
        );

        $salesOrder->sales_order_number = 'post_update';
        $salesOrder->update();
        $this->resetUpdatedAt($salesOrder->accountingTransaction);

        $this->manager->sync();

        $this->assertDatabaseHas((new AccountingTransaction())->getTable(), [
            'reference' => 'post_update',
            'link_id' => $salesOrder->id,
            'link_type' => SalesOrder::class,
        ]);

        $this->assertDatabaseCount((new AccountingTransaction())->getTable(), 1);

        $salesOrderLines = [];

        /** @var SalesOrderLine $salesOrderLine */
        foreach ($salesOrder->salesOrderLines as $salesOrderLine) {
            $salesOrderLines[] = [
                'id' => $salesOrderLine->id,
                'quantity' => 2,
                'amount' => 55,
                'updated_at' => Carbon::now()->copy()->addMinute(),
            ];
        }

        $salesOrder->setSalesOrderLines($salesOrderLines);
        $this->resetUpdatedAt($salesOrder->accountingTransaction);

        $financialLineType = FinancialLineType::factory()->create([
            'name' => 'Shipping',
            'classification' => FinancialLineClassificationEnum::REVENUE,
            'allocate_to_products' => false,
            'proration_strategy' => FinancialLineProrationStrategyEnum::REVENUE_BASED,
        ]);

        app(SalesOrderRepository::class)->setFinancialLines($salesOrder, [
            FinancialLineData::from([
                'sales_order_id' => $salesOrder->id,
                'financial_line_type_id' => $financialLineType->id,
                'description' => 'Shipping',
                'quantity' => 1,
                'amount' => 9.90,
                'tax_allocation' => 0.90,
                'nominal_code_id' => 1,
            ])->toArray(),
        ]);

        (new SalesOrderLineFinancialManager())->calculate();
        $this->manager->sync([AccountingTransaction::first()->id]);

        $this->assertDatabaseHas((new AccountingTransaction())->getTable(), [
            'link_id' => $salesOrder->id,
            'link_type' => SalesOrder::class,
        ]);
        $this->assertDatabaseCount((new AccountingTransaction())->getTable(), 1);

        $this->assertDatabaseHas((new AccountingTransactionLine())->getTable(), [
            'link_type' => SalesOrderLine::class,
            'quantity' => 2,
            'amount' => 55.00,
        ]);

        $this->assertDatabaseHas((new AccountingTransactionLine())->getTable(), [
            'link_type' => FinancialLine::class,
            'quantity' => 1,
            'amount' => 9.00,
            'tax_amount' => 0.90,
        ]);

        $salesOrder->delete();

        $this->manager->sync();

        $this->assertDatabaseCount((new AccountingTransaction())->getTable(), 0);
        $this->assertDatabaseCount((new AccountingTransactionLine())->getTable(), 0);
    }

    /**
     * @throws Exception
     * @throws Throwable
     */
    private function testSalesCredits(): void
    {
        $salesOrder = SalesOrder::factory()->createWithFullSalesCreditFullyReturned(3);

        /** @var SalesCredit $salesCredit */
        $salesCredit = $salesOrder->salesCredits->first();

        $salesCredit->sales_credit_number = 'pre_update';
        $salesCredit->save();

        $this->manager->sync();

        // Test sales credit links to parent
        $this->assertDatabaseHas((new AccountingTransaction())->getTable(), [
            'link_id' => $salesCredit->id,
            'link_type' => SalesCredit::class,
            'parent_id' => $salesOrder->accountingTransaction->id,
        ]);

        $this->assertEquals(
            1,
            AccountingTransaction::withSalesCredits($salesOrder->salesCredits->first()->id)
                ->where('type', AccountingTransaction::TYPE_SALES_CREDIT)
                ->count()
        );

        $this->assertEquals(
            3,
            AccountingTransactionLine::withSalesCreditLines(
                $salesOrder->salesCredits->first()->salesCreditLines()
                    ->pluck('id')
                    ->toArray()
            )
                ->where('type', AccountingTransactionLine::TYPE_SALES_CREDIT_LINE)
                ->count()
        );

        $salesCreditLines = [];

        /** @var SalesCreditLine $salesCreditLine */
        foreach ($salesCredit->salesCreditLines as $salesCreditLine) {
            $salesCreditLines[] = [
                'id' => $salesCreditLine->id,
                'quantity' => 2,
                'amount' => 55,
                'updated_at' => Carbon::now()->copy()->addMinute(),
            ];
        }

        $salesCredit->setSalesCreditLines($salesCreditLines);
        $this->resetUpdatedAt($salesCredit->accountingTransaction);

        (new SalesOrderLineFinancialManager())->calculate();
        $this->manager->sync();

        $this->assertDatabaseHas((new AccountingTransaction())->getTable(), [
            'link_id' => $salesCredit->id,
            'link_type' => SalesCredit::class,
            'total' => 330.00,
        ]);
        $this->assertDatabaseCount((new AccountingTransaction())->getTable(), 3);

        $this->assertDatabaseHas((new AccountingTransactionLine())->getTable(), [
            'link_type' => SalesCreditLine::class,
            'quantity' => 2,
            'amount' => 55,
        ]);

        $salesOrder->delete();

        $this->manager->sync();

        $this->assertDatabaseCount((new AccountingTransaction())->getTable(), 0);
        $this->assertDatabaseCount((new AccountingTransactionLine())->getTable(), 0);
    }

    /**
     * @throws Throwable
     */
    private function testSalesOrderFulfillments(): void
    {
        /** @var SalesOrder $salesOrder */
        $salesOrder = SalesOrder::factory()
            ->has(
                SalesOrderLine::factory()
                    ->state(['quantity' => 2])
            )
            ->has(
                SalesOrderFulfillment::factory()
                    ->has(
                        SalesOrderFulfillmentLine::factory()
                            ->state(function ($attributes, SalesOrderFulfillment $salesOrderFulfillment) {
                                return [
                                    'sales_order_line_id' => $salesOrderFulfillment->salesOrder->salesOrderLines()->first()->id,
                                    'quantity' => 1,
                                ];
                            })
                    )
            )
            ->create();

        $this->manager->sync();

        $this->assertEquals(
            1,
            AccountingTransaction::withSalesOrderFulfillments(
                $salesOrder->salesOrderFulfillments()->first()->id
            )
                ->where('type', AccountingTransaction::TYPE_SALES_ORDER_FULFILLMENT)
                ->count()
        );

        $this->assertEquals(
            1,
            AccountingTransactionLine::withSalesOrderFulfillmentLines(
                $salesOrder->salesOrderFulfillments()->first()->salesOrderFulfillmentLines()
                    ->pluck('id')
                    ->toArray()
            )
                ->where('type', AccountingTransactionLine::TYPE_DEBIT)
                ->count()
        );

        $this->assertEquals(
            1,
            AccountingTransactionLine::withSalesOrderFulfillmentLines(
                $salesOrder->salesOrderFulfillments()->first()->salesOrderFulfillmentLines()
                    ->pluck('id')
                    ->toArray()
            )
                ->where('type', AccountingTransactionLine::TYPE_CREDIT)
                ->count()
        );

        /** @var SalesOrderFulfillment $salesOrderFulfillment */
        $salesOrderFulfillment = $salesOrder->salesOrderFulfillments()->first();

        $salesOrderFulfillment->fulfillment_sequence = 2;
        $salesOrderFulfillment->update();
        $this->resetUpdatedAt($salesOrderFulfillment->accountingTransaction);

        (new SalesOrderLineFinancialManager())->calculate();
        $this->manager->sync();

        // Test sales order fulfillment links to parent
        $this->assertDatabaseHas((new AccountingTransaction())->getTable(), [
            'link_id' => $salesOrderFulfillment->id,
            'link_type' => SalesOrderFulfillment::class,
            'parent_id' => $salesOrder->accountingTransaction->id,
        ]);

        $this->assertDatabaseHas((new AccountingTransaction())->getTable(), [
            'reference' => $salesOrder->customer->name . ': Fulfillment '.$salesOrderFulfillment->fulfillment_number,
            'link_id' => $salesOrderFulfillment->id,
            'link_type' => SalesOrderFulfillment::class,
        ]);

        $this->assertDatabaseCount((new AccountingTransaction())->getTable(), 2);

        /** @var SalesOrderLine $salesOrderLine */
        $salesOrderLine = $salesOrder->salesOrderLines->first();

        $salesOrderLine->product->unit_cost = 55;
        $salesOrderLine->product->save();

        /** @var SalesOrderFulfillment $salesOrderFulfillment */
        $salesOrderFulfillment = $salesOrder->salesOrderFulfillments()->first();

        $salesOrderFulfillmentLine = $salesOrderFulfillment->salesOrderFulfillmentLines()->first();

        (new SalesOrderLineFinancialManager())->calculate();
        $this->resetUpdatedAt($salesOrderFulfillment->accountingTransaction);
        $this->manager->sync();

        $this->assertDatabaseHas((new AccountingTransaction())->getTable(), [
            'link_id' => $salesOrderFulfillment->id,
            'link_type' => SalesOrderFulfillment::class,
            'total' => 55.00,
        ]);
        $this->assertDatabaseCount((new AccountingTransaction())->getTable(), 2);

        $salesOrderLines = [[
            'id' => $salesOrderFulfillmentLine->salesOrderLine->id,
            'quantity' => 2,
            'updated_at' => CarbonImmutable::now()->addMinute(2),
        ]];

        $salesOrder->setSalesOrderLines($salesOrderLines);

        /** @var SalesOrderLine $salesOrderLine */
        $salesOrderLine = $salesOrder->salesOrderLines()->first();

        $salesOrderLine->product->update(['unit_cost' => 60]);

        // TODO: sales order line updates happen too often... need to reconsider this.
        //        (new SalesOrderLineFinancialManager())->calculate();
        //        $this->manager->sync();
        //
        //        $this->assertDatabaseHas((new AccountingTransaction())->getTable(), [
        //            'link_id' => $salesOrderFulfillment->id,
        //            'link_type' => SalesOrderFulfillment::class
        //        ]);
        //        $this->assertDatabaseCount((new AccountingTransaction())->getTable(), 2);
        //
        //        $this->assertDatabaseHas((new AccountingTransactionLine())->getTable(), [
        //            'link_type' => SalesOrderFulfillmentLine::class,
        //            'quantity' => 1,
        //            'amount' => 60,
        //        ]);
        //
        //        $salesOrderFulfillmentLine->update(['quantity' => 2, 'updated_at' => CarbonImmutable::now()->addMinute(2)]);
        //
        //        (new SalesOrderLineFinancialManager())->calculate();
        //        $this->manager->sync();
        //
        //        $this->assertDatabaseHas((new AccountingTransaction())->getTable(), [
        //            'link_id' => $salesOrderFulfillment->id,
        //            'link_type' => SalesOrderFulfillment::class,
        //            'total' => 120,
        //        ]);
        //
        //        $this->assertDatabaseHas((new AccountingTransactionLine())->getTable(), [
        //            'link_type' => SalesOrderFulfillmentLine::class,
        //            'quantity' => 2,
        //            'amount' => 60,
        //        ]);
        $salesOrder->delete();

        $this->manager->sync();

        $this->assertDatabaseCount((new AccountingTransaction())->getTable(), 0);
        $this->assertDatabaseCount((new AccountingTransactionLine())->getTable(), 0);
    }

    /**
     * @throws Exception
     * @throws Throwable
     */
    private function testPurchaseOrders(): void
    {
        /** @var PurchaseOrder $purchaseOrder */
        $purchaseOrder = PurchaseOrder::factory()
            ->has(
                PurchaseOrderLine::factory()
                    ->state(['quantity' => 2, 'amount' => 55.00])
            )
            ->create(['purchase_order_number' => 'pre_update']);

        $purchaseOrderLine = $purchaseOrder->purchaseOrderLines()->first();

        $this->manager->sync();

        $this->assertEquals(
            1,
            AccountingTransaction::withPurchaseOrders($purchaseOrder->id)
                ->where('type', AccountingTransaction::TYPE_PURCHASE_ORDER)
                ->count()
        );

        $this->assertEquals(
            1,
            AccountingTransactionLine::withPurchaseOrderLines(
                $purchaseOrder->purchaseOrderLines()->first()
                    ->pluck('id')
                    ->toArray()
            )
                ->where('type', AccountingTransactionLine::TYPE_PURCHASE_ORDER_LINE)
                ->count()
        );

        $purchaseOrder->purchase_order_number = 'post_update';
        $purchaseOrder->save();
        $this->resetUpdatedAt($purchaseOrder->accountingTransaction);

        (new SalesOrderLineFinancialManager())->calculate();
        $this->manager->sync();

        $this->assertDatabaseHas((new AccountingTransaction())->getTable(), [
            'reference' => 'post_update',
            'link_id' => $purchaseOrder->id,
            'link_type' => $purchaseOrder::class,
        ]);

        $this->assertDatabaseCount((new AccountingTransaction())->getTable(), 1);

        $purchaseOrderLines = [[
            'id' => $purchaseOrderLine->id,
            'quantity' => 2,
            'amount' => 60,
        ]];

        $purchaseOrder->setPurchaseOrderLines($purchaseOrderLines);
        $this->resetUpdatedAt($purchaseOrder->accountingTransaction);

        (new SalesOrderLineFinancialManager())->calculate();
        $this->manager->sync();

        $this->assertDatabaseHas((new AccountingTransaction())->getTable(), [
            'link_id' => $purchaseOrder->id,
            'link_type' => PurchaseOrder::class,
        ]);
        $this->assertDatabaseCount((new AccountingTransaction())->getTable(), 1);

        $this->assertDatabaseHas((new AccountingTransactionLine())->getTable(), [
            'link_type' => PurchaseOrderLine::class,
            'quantity' => 2,
            'amount' => 60,
        ]);

        $purchaseOrder->delete();

        $this->manager->sync();

        $this->assertDatabaseCount((new AccountingTransaction())->getTable(), 0);
        $this->assertDatabaseCount((new AccountingTransactionLine())->getTable(), 0);
    }

    /**
     * @throws Exception|Throwable
     */
    private function testPurchaseOrderInvoices(): void
    {
        /** @var PurchaseOrder $purchaseOrder */
        $purchaseOrder = PurchaseOrder::factory()
            ->has(
                PurchaseOrderLine::factory()
                    ->state(['quantity' => 2, 'amount' => 55.00])
            )
            ->has(
                PurchaseInvoice::factory()
                    ->has(
                        PurchaseInvoiceLine::factory()
                            ->state(function ($attributes, PurchaseInvoice $purchaseInvoice) {
                                return [
                                    'purchase_order_line_id' => $purchaseInvoice->purchaseOrder->purchaseOrderLines()->first()->id,
                                    'quantity_invoiced' => 1,
                                ];
                            })
                    )
            )
            ->create();

        $purchaseInvoice = $purchaseOrder->purchaseInvoices()->first();
        $purchaseInvoiceLine = $purchaseInvoice->purchaseInvoiceLines()->first();

        $this->manager->sync();

        // Test purchase invoice links to parent
        $this->assertDatabaseHas((new AccountingTransaction())->getTable(), [
            'link_id' => $purchaseInvoice->id,
            'link_type' => PurchaseInvoice::class,
            'parent_id' => $purchaseOrder->accountingTransaction->id,
        ]);

        $this->assertEquals(
            1,
            AccountingTransaction::withPurchaseInvoices($purchaseOrder->purchaseInvoices()->first()->id)
                ->where('type', AccountingTransaction::TYPE_PURCHASE_ORDER_INVOICE)
                ->count()
        );

        $this->assertEquals(
            1,
            AccountingTransactionLine::withPurchaseInvoiceLines(
                $purchaseOrder->purchaseInvoices()->first()->purchaseInvoiceLines()
                    ->pluck('id')
                    ->toArray()
            )
                ->where('type', AccountingTransactionLine::TYPE_PURCHASE_INVOICE_LINE)
                ->count()
        );

        $purchaseInvoice->supplier_invoice_number = 'post_update';
        $purchaseInvoice->update();

        $this->resetUpdatedAt($purchaseInvoice->accountingTransaction);

        (new SalesOrderLineFinancialManager())->calculate();
        $this->manager->sync();

        $this->assertDatabaseHas((new AccountingTransaction())->getTable(), [
            'reference' => $purchaseInvoice->purchaseOrder->purchase_order_number.': post_update',
            'link_id' => $purchaseInvoice->id,
            'link_type' => PurchaseInvoice::class,
        ]);

        $this->assertDatabaseCount((new AccountingTransaction())->getTable(), 2);

        $purchaseOrderLines = [[
            'id' => $purchaseInvoiceLine->purchaseOrderLine->id,
            'quantity' => 2,
            'amount' => 60,
            'updated_at' => Carbon::now()->copy()->addMinute(),
        ]];

        $purchaseOrder->setPurchaseOrderLines($purchaseOrderLines);
        $this->resetUpdatedAt($purchaseOrder->accountingTransaction);
        $this->resetUpdatedAt($purchaseInvoice->accountingTransaction);

        (new SalesOrderLineFinancialManager())->calculate();
        $this->manager->sync();

        $this->assertDatabaseHas((new AccountingTransaction())->getTable(), [
            'link_id' => $purchaseInvoice->id,
            'link_type' => PurchaseInvoice::class,
        ]);
        $this->assertDatabaseCount((new AccountingTransaction())->getTable(), 2);

        $this->assertDatabaseHas((new AccountingTransactionLine())->getTable(), [
            'link_type' => PurchaseInvoiceLine::class,
            'quantity' => 1,
            'amount' => 60,
        ]);

        $purchaseInvoiceLine->update(['quantity_invoiced' => 2]);
        $this->resetUpdatedAt($purchaseInvoice->accountingTransaction);

        (new SalesOrderLineFinancialManager())->calculate();
        $this->manager->sync();

        $this->assertDatabaseHas((new AccountingTransaction())->getTable(), [
            'link_id' => $purchaseInvoice->id,
            'link_type' => PurchaseInvoice::class,
            'total' => 120,
        ]);

        $this->assertDatabaseHas((new AccountingTransactionLine())->getTable(), [
            'link_type' => PurchaseInvoiceLine::class,
            'quantity' => 2,
            'amount' => 60,
        ]);

        $purchaseOrder->delete();

        $this->manager->sync();

        $this->assertDatabaseCount((new AccountingTransaction())->getTable(), 0);
        $this->assertDatabaseCount((new AccountingTransactionLine())->getTable(), 0);
    }

    /**
     * @throws Exception|Throwable
     */
    private function testPurchaseReceipt(): void
    {
        /** @var PurchaseOrder $purchaseOrder */
        $purchaseOrder = PurchaseOrder::factory()
            ->received(1)
            ->create();

        $purchaseLine = $purchaseOrder->purchaseOrderLines()->first();
        $purchaseReceipt = $purchaseOrder->purchaseOrderShipments()->first()->purchaseOrderShipmentReceipts()->first();
        $purchaseReceiptLine = $purchaseReceipt->purchaseOrderShipmentReceiptLines()->first();

        $this->manager->sync();

        // Test purchase receipt links to parent
        $this->assertDatabaseHas((new AccountingTransaction())->getTable(), [
            'link_id' => $purchaseReceipt->id,
            'link_type' => PurchaseOrderShipmentReceipt::class,
            'parent_id' => $purchaseOrder->accountingTransaction->id,
        ]);

        $this->assertEquals(
            1,
            AccountingTransaction::withPurchaseOrderReceipts(
                $purchaseOrder->purchaseOrderShipments()->first()->purchaseOrderShipmentReceipts()->first()->id
            )
                ->where('type', AccountingTransaction::TYPE_PURCHASE_ORDER_RECEIPT)
                ->count()
        );

        $this->assertEquals(
            1,
            AccountingTransactionLine::withPurchaseOrderReceiptLines(
                $purchaseOrder->purchaseOrderShipments()->first()
                    ->purchaseOrderShipmentReceipts()->first()->purchaseOrderShipmentReceiptLines()
                    ->pluck('id')
                    ->toArray()
            )
                ->where('type', AccountingTransactionLine::TYPE_DEBIT)
                ->count()
        );

        $this->assertEquals(
            1,
            AccountingTransactionLine::withPurchaseOrderReceiptLines(
                $purchaseOrder->purchaseOrderShipments()->first()
                    ->purchaseOrderShipmentReceipts()->first()->purchaseOrderShipmentReceiptLines()
                    ->pluck('id')
                    ->toArray()
            )
                ->where('type', AccountingTransactionLine::TYPE_CREDIT)
                ->count()
        );

        $purchaseOrder->purchase_order_number = 'post_update';
        $purchaseOrder->save();
        $this->resetUpdatedAt($purchaseReceipt->accountingTransaction);

        (new SalesOrderLineFinancialManager())->calculate();
        $this->manager->sync();

        $this->assertDatabaseHas((new AccountingTransaction())->getTable(), [
            'reference' => $purchaseOrder->supplier->name . ': Receipt #'.$purchaseReceipt->id.' on post_update',
            'link_id' => $purchaseReceipt->id,
            'link_type' => PurchaseOrderShipmentReceipt::class,
        ]);

        $this->assertDatabaseCount((new AccountingTransaction())->getTable(), 2);

        $newReceivedDate = Carbon::now()->copy()->addMinute();

        $purchaseReceipt->received_at = $newReceivedDate;
        $purchaseReceipt->save();
        $this->resetUpdatedAt($purchaseReceipt->accountingTransaction);

        (new SalesOrderLineFinancialManager())->calculate();
        $this->manager->sync();

        $this->assertDatabaseHas((new AccountingTransaction())->getTable(), [
            'reference' => $purchaseOrder->supplier->name . ': Receipt #'.$purchaseReceipt->id.' on post_update',
            'transaction_date' => $newReceivedDate,
            'link_id' => $purchaseReceipt->id,
            'link_type' => PurchaseOrderShipmentReceipt::class,
        ]);

        $this->assertDatabaseCount((new AccountingTransaction())->getTable(), 2);

        $purchaseLine->update(['quantity' => 2, 'amount' => 55.00]);

        $purchaseReceiptLine->update(['quantity' => 1]);

        $this->resetUpdatedAt($purchaseReceipt->accountingTransaction);

        $this->manager->sync();

        $this->assertDatabaseHas((new AccountingTransaction())->getTable(), [
            'link_id' => $purchaseReceipt->id,
            'link_type' => PurchaseOrderShipmentReceipt::class,
            'total' => 55.00,
        ]);
        $this->assertDatabaseCount((new AccountingTransaction())->getTable(), 2);

        $purchaseOrderLines = [[
            'id' => $purchaseLine->id,
            'quantity' => 2,
            'amount' => 60,
            'updated_at' => Carbon::now()->copy()->addMinute(),
        ]];

        $purchaseOrder->setPurchaseOrderLines($purchaseOrderLines);
        $this->resetUpdatedAt($purchaseReceipt->accountingTransaction);

        (new SalesOrderLineFinancialManager())->calculate();
        $this->manager->sync();

        $this->assertDatabaseHas((new AccountingTransaction())->getTable(), [
            'link_id' => $purchaseReceipt->id,
            'link_type' => PurchaseOrderShipmentReceipt::class,
        ]);
        $this->assertDatabaseCount((new AccountingTransaction())->getTable(), 2);

        $this->assertDatabaseHas((new AccountingTransactionLine())->getTable(), [
            'link_type' => PurchaseOrderShipmentReceiptLine::class,
            'quantity' => 1,
            'amount' => 60,
        ]);

        $purchaseReceiptLine->update(['quantity' => 2]);

        $this->resetUpdatedAt($purchaseReceipt->accountingTransaction);

        (new SalesOrderLineFinancialManager())->calculate();
        $this->manager->sync();

        $this->assertDatabaseHas((new AccountingTransaction())->getTable(), [
            'link_id' => $purchaseReceipt->id,
            'link_type' => PurchaseOrderShipmentReceipt::class,
            'total' => 120.00,
        ]);

        $this->assertDatabaseHas((new AccountingTransactionLine())->getTable(), [
            'link_type' => PurchaseOrderShipmentReceiptLine::class,
            'quantity' => 2,
            'amount' => 60,
        ]);

        $purchaseOrder->delete();

        $this->manager->sync();

        $this->assertDatabaseCount((new AccountingTransaction())->getTable(), 0);
        $this->assertDatabaseCount((new AccountingTransactionLine())->getTable(), 0);
    }

    /**
     * @throws Exception|Throwable
     */
    private function testAdjustments(): void
    {
        /** @var InventoryAdjustment $adjustment */
        $adjustment = InventoryAdjustment::factory()
            ->has(
                InventoryMovement::factory()
                    ->state(function (array $attributes, InventoryAdjustment $inventoryAdjustment) {
                        return [
                            'link_id' => $inventoryAdjustment->id,
                            'link_type' => InventoryAdjustment::class,
                            'quantity' => 2,
                        ];
                    })
            )->create(
                [
                    'quantity' => 2,
                ]
            );

        $this->manager->sync();

        $this->assertEquals(
            1,
            AccountingTransaction::withInventoryAdjustments($adjustment->id)
                ->where('type', AccountingTransaction::TYPE_INVENTORY_ADJUSTMENT)
                ->count()
        );

        $this->assertEquals(
            1,
            AccountingTransactionLine::withInventoryAdjustments(
                $adjustment
                    ->pluck('id')
                    ->toArray()
            )
                ->where('type', AccountingTransactionLine::TYPE_DEBIT)
                ->count()
        );

        $this->assertEquals(
            1,
            AccountingTransactionLine::withInventoryAdjustments(
                $adjustment
                    ->pluck('id')
                    ->toArray()
            )
                ->where('type', AccountingTransactionLine::TYPE_CREDIT)
                ->count()
        );

        $adjustment_date = Carbon::now()->copy()->addMinute();
        $adjustment->adjustment_date = $adjustment_date;
        $adjustment->update();

        $this->resetUpdatedAt($adjustment->accountingTransaction);

        (new SalesOrderLineFinancialManager())->calculate();
        $this->manager->sync();

        $this->assertDatabaseHas((new AccountingTransaction())->getTable(), [
            'transaction_date' => $adjustment_date,
            'link_id' => $adjustment->id,
            'link_type' => InventoryAdjustment::class,
        ]);

        $this->assertDatabaseCount((new AccountingTransaction())->getTable(), 1);

        $inventoryMovement = $adjustment->inventoryMovements()->first();
        $fifoLayer = $inventoryMovement->layer;

        $fifoLayer->update(['total_cost' => 110, 'original_quantity' => 2]);
        $this->resetUpdatedAt($adjustment->accountingTransaction);

        (new SalesOrderLineFinancialManager())->calculate();
        $this->manager->sync();

        $this->assertDatabaseHas((new AccountingTransaction())->getTable(), [
            'link_id' => $adjustment->id,
            'link_type' => InventoryAdjustment::class,
            'total' => 110.0000,
        ]);
        $this->assertDatabaseCount((new AccountingTransaction())->getTable(), 1);

        $inventoryMovement->update(['quantity' => 2, 'updated_at' => Carbon::now()->copy()->addMinute()]);
        $fifoLayer->update(['total_cost' => 120, 'original_quantity' => 2]);

        $this->resetUpdatedAt($adjustment->accountingTransaction);

        (new SalesOrderLineFinancialManager())->calculate();
        $this->manager->sync();

        $this->assertDatabaseHas((new AccountingTransaction())->getTable(), [
            'link_id' => $adjustment->id,
            'link_type' => InventoryAdjustment::class,
            'total' => 120.00,
        ]);
        $this->assertDatabaseCount((new AccountingTransaction())->getTable(), 1);

        $this->assertDatabaseHas((new AccountingTransactionLine())->getTable(), [
            'link_type' => InventoryAdjustment::class,
            'quantity' => 2,
            'amount' => 60,
        ]);

        $adjustment->delete();

        $this->manager->sync();

        $this->assertDatabaseCount((new AccountingTransaction())->getTable(), 0);
        $this->assertDatabaseCount((new AccountingTransactionLine())->getTable(), 0);
    }

    /**
     * @throws Exception|Throwable
     */
    private function testCustomerReturns(): void
    {
        /*
         * TODO: once we refactor the sales order -> sales credit relationships, we should add
         *  a factory method to create sales credit returns with non accounting situations and test
         *  that those don't create accounting transactions / transaction lines
         */
        $salesOrder = SalesOrder::factory()
            ->createWithFullSalesCreditFullyReturned(3);

        /** @var SalesCredit $salesCredit */
        $salesCredit = $salesOrder->salesCredits->first();

        /** @var SalesCreditReturn $salesCreditReturn */
        $salesCreditReturn = $salesCredit->salesCreditReturns()->first();

        /** @var SalesCreditReturnLine $salesCreditReturnLine */
        $salesCreditReturnLine = $salesCreditReturn->salesCreditReturnLines()->first();

        $inventoryMovement = $salesCreditReturnLine->inventoryMovements()->first();
        $fifoLayer = $inventoryMovement->layer;

        $this->manager->sync();

        $this->assertEquals(
            1,
            AccountingTransaction::withCustomerReturns(
                $salesOrder->salesCredits->first()
                    ->salesCreditReturns()->where('status', SalesCreditReturn::STATUS_CLOSED)->first()->id
            )
                ->where('type', AccountingTransaction::TYPE_CUSTOMER_RETURN)
                ->count()
        );

        $this->assertEquals(
            3,
            AccountingTransactionLine::withCustomerReturnLines(
                $salesOrder->salesCredits->first()
                    ->salesCreditReturns()->first()->salesCreditReturnLines()
                    ->whereIn('action', SalesCreditReturnLine::ACTION_INVENTORY_IMPACT)
                    ->pluck('id')
                    ->toArray()
            )
                ->where('type', AccountingTransactionLine::TYPE_DEBIT)
                ->count()
        );

        $this->assertEquals(
            3,
            AccountingTransactionLine::withCustomerReturnLines(
                $salesOrder->salesCredits->first()
                    ->salesCreditReturns()->first()->salesCreditReturnLines()
                    ->pluck('id')
                    ->toArray()
            )
                ->where('type', AccountingTransactionLine::TYPE_CREDIT)
                ->count()
        );

        $salesOrder->update(['sales_order_number' => 'post_update']);
        $this->resetUpdatedAt($salesCreditReturn->accountingTransaction);


        (new SalesOrderLineFinancialManager())->calculate();
        $this->manager->sync();

        $this->assertDatabaseHas((new AccountingTransaction())->getTable(), [
            'reference' => $salesOrder->customer->name . ': Customer Return #'.$salesCreditReturn->id.' through '.
                $salesCredit->sales_credit_number.' (post_update)',
            'link_id' => $salesCreditReturn->id,
            'link_type' => SalesCreditReturn::class,
        ]);

        $this->assertDatabaseCount((new AccountingTransaction())->getTable(), 3);

        $salesCredit->update(['sales_credit_number' => 'post_update']);
        $this->resetUpdatedAt($salesCreditReturn->accountingTransaction);

        (new SalesOrderLineFinancialManager())->calculate();
        $this->manager->sync();

        $this->assertDatabaseHas((new AccountingTransaction())->getTable(), [
            'reference' => $salesOrder->customer->name . ': Customer Return #'.$salesCreditReturn->id.' through post_update ('.$salesOrder->sales_order_number.')',
            'link_id' => $salesCreditReturn->id,
            'link_type' => SalesCreditReturn::class,
        ]);

        $this->assertDatabaseCount((new AccountingTransaction())->getTable(), 3);

        $newReceivedDate = CarbonImmutable::now()->addMinute()->format('Y-m-d H:i:s');
        //Log::debug('Received at date 1: ' . $newReceivedDate);

        $salesCreditReturn->update(['received_at' => $newReceivedDate]);
        $this->resetUpdatedAt($salesCreditReturn->accountingTransaction);

        (new SalesOrderLineFinancialManager())->calculate();
        $this->manager->sync();

        //Log::debug('Received at date 3: ' . $newReceivedDate);
        $this->assertDatabaseHas((new AccountingTransaction())->getTable(), [
            'reference' => $salesOrder->customer->name . ': Customer Return #'.$salesCreditReturn->id.' through post_update (post_update)',
            'transaction_date' => $newReceivedDate,
            'link_id' => $salesCreditReturn->id,
            'link_type' => SalesCreditReturn::class,
        ]);

        $this->assertDatabaseCount((new AccountingTransaction())->getTable(), 3);

        $inventoryMovement->update(['quantity' => $salesCreditReturnLine->quantity]);
        $fifoLayer->update(['total_cost' => $salesCreditReturnLine->quantity * 55, 'original_quantity' => $salesCreditReturnLine->quantity]);

        $this->resetUpdatedAt($salesCreditReturn->accountingTransaction);

        (new SalesOrderLineFinancialManager())->calculate();
        $this->manager->sync();

        $this->assertDatabaseHas((new AccountingTransaction())->getTable(), [
            'link_id' => $salesCreditReturn->id,
            'link_type' => SalesCreditReturn::class,
            'total' => $salesCreditReturnLine->quantity * 55,
        ]);
        $this->assertDatabaseCount((new AccountingTransaction())->getTable(), 3);

        $inventoryMovement->update(['quantity' => $salesCreditReturnLine->quantity]);
        $fifoLayer->update(['total_cost' => $salesCreditReturnLine->quantity * 60, 'original_quantity' => $salesCreditReturnLine->quantity]);

        $this->resetUpdatedAt($salesCreditReturn->accountingTransaction);

        (new SalesOrderLineFinancialManager())->calculate();
        $this->manager->sync();

        $this->assertDatabaseHas((new AccountingTransaction())->getTable(), [
            'link_id' => $salesCreditReturn->id,
            'link_type' => SalesCreditReturn::class,
            'total' => $salesCreditReturnLine->quantity * 60,
        ]);
        $this->assertDatabaseCount((new AccountingTransaction())->getTable(), 3);

        $this->assertDatabaseHas((new AccountingTransactionLine())->getTable(), [
            'link_type' => SalesCreditReturnLine::class,
            'quantity' => $salesCreditReturnLine->quantity,
            'amount' => 60,
        ]);

        $salesOrder->delete();

        $this->manager->sync();

        $this->assertDatabaseCount((new AccountingTransaction())->getTable(), 0);
        $this->assertDatabaseCount((new AccountingTransactionLine())->getTable(), 0);
    }

    /**
     * @throws Exception|Throwable
     */
    private function testStockTakes(): void
    {
        /** @var StockTake $stockTake */
        $stockTake = StockTake::factory()->finalized()->has(
            StockTakeItem::factory()->has(
                InventoryMovement::factory()
                    ->state(function (array $attributes, StockTakeItem $stockTakeItem) {
                        return [
                            'type' => InventoryMovement::TYPE_STOCK_TAKE,
                            'link_type' => StockTakeItem::class,
                            'link_id' => $stockTakeItem->id,
                            'quantity' => 2,
                        ];
                    })
            )
        )
            ->create();

        $this->manager->sync();

        $this->assertEquals(
            1,
            AccountingTransaction::withStockTakes(
                $stockTake->id
            )
                ->where('type', AccountingTransaction::TYPE_STOCK_TAKE)
                ->count()
        );

        $this->assertEquals(
            1,
            AccountingTransactionLine::withStockTakeLines(
                $stockTake->stockTakeItems()
                    ->pluck('id')
                    ->toArray()
            )
                ->where('type', AccountingTransactionLine::TYPE_DEBIT)
                ->count()
        );

        $this->assertEquals(
            1,
            AccountingTransactionLine::withStockTakeLines(
                $stockTake->stockTakeItems()
                    ->pluck('id')
                    ->toArray()
            )
                ->where('type', AccountingTransactionLine::TYPE_CREDIT)
                ->count()
        );

        $dateCount = Carbon::now()->format('Y-m-d');

        $stockTake->update(['date_count' => $dateCount]);
        $this->resetUpdatedAt($stockTake->accountingTransaction);

        (new SalesOrderLineFinancialManager())->calculate();
        $this->manager->sync();

        $this->assertDatabaseHas((new AccountingTransaction())->getTable(), [
            'transaction_date' => $dateCount,
            'link_id' => $stockTake->id,
            'link_type' => StockTake::class,
        ]);

        $this->assertDatabaseCount((new AccountingTransaction())->getTable(), 1);

        $stockTakeItem = $stockTake->stockTakeItems()->first();
        $inventoryMovement = $stockTakeItem->inventoryMovements()->first();
        $fifoLayer = $inventoryMovement->layer;

        $fifoLayer->update(['total_cost' => 2 * 55, 'original_quantity' => 2]);
        $stockTakeItem->update(['qty_counted' => 2, 'snapshot_inventory' => 0]);

        $this->resetUpdatedAt($stockTake->accountingTransaction);

        (new SalesOrderLineFinancialManager())->calculate();
        $this->manager->sync();

        $this->assertDatabaseHas((new AccountingTransaction())->getTable(), [
            'link_id' => $stockTake->id,
            'link_type' => StockTake::class,
            'total' => 2 * 55,
        ]);
        $this->assertDatabaseCount((new AccountingTransaction())->getTable(), 1);

        $inventoryMovement->update(['quantity' => 2]);
        $fifoLayer->update(['total_cost' => 2 * 60, 'original_quantity' => 2]);

        $this->resetUpdatedAt($stockTake->accountingTransaction);

        (new SalesOrderLineFinancialManager())->calculate();
        $this->manager->sync();

        $this->assertDatabaseHas((new AccountingTransaction())->getTable(), [
            'link_id' => $stockTake->id,
            'link_type' => StockTake::class,
            'total' => 2 * 60,
        ]);
        $this->assertDatabaseCount((new AccountingTransaction())->getTable(), 1);

        $this->assertDatabaseHas((new AccountingTransactionLine())->getTable(), [
            'link_type' => StockTakeItem::class,
            'quantity' => 2,
            'amount' => 60,
        ]);

        $stockTake->delete();

        $this->manager->sync();

        $this->assertDatabaseCount((new AccountingTransaction())->getTable(), 0);
        $this->assertDatabaseCount((new AccountingTransactionLine())->getTable(), 0);
    }

    /**
     * @throws Throwable
     * @throws InsufficientStockException
     * @throws WarehouseTransferHasNoProductsException
     * @throws WarehouseTransferOpenException
     */
    private function testWarehouseTransferShipments(): void
    {
        $fromWarehouse = Warehouse::first();
        $toWarehouse = Warehouse::factory()->create();
        $product = Product::factory()->create();

        $product->setInitialInventory($fromWarehouse->id, 50, 5.00);

        $warehouseTransfer = WarehouseTransfer::factory()
            ->hasWarehouseTransferLines(1, [
                'quantity' => 1,
                'product_id' => $product->id,
            ])
            ->create([
                'from_warehouse_id' => $fromWarehouse->id,
                'to_warehouse_id' => $toWarehouse->id,
                'transfer_status' => WarehouseTransfer::TRANSFER_STATUS_DRAFT,
            ]);

        app(WarehouseTransferManager::class)->openWarehouseTransfer($warehouseTransfer, []);

        $warehouseTransferShipment = $warehouseTransfer->shipment;

        $this->manager->sync();

        $this->assertEquals(
            1,
            AccountingTransaction::withWarehouseTransferShipments(
                $warehouseTransferShipment->id
            )
                ->where('type', AccountingTransactionTypeEnum::WAREHOUSE_TRANSFER_SHIPMENT)
                ->count()
        );

        $this->assertEquals(
            1,
            AccountingTransactionLine::withWarehouseTransferShipmentLines(
                $warehouseTransferShipment->shipmentLines()
                    ->pluck('id')
                    ->toArray()
            )
                ->where('type', AccountingTransactionLine::TYPE_DEBIT)
                ->count()
        );

        $this->assertEquals(
            1,
            AccountingTransactionLine::withWarehouseTransferShipmentLines(
                $warehouseTransferShipment->shipmentLines()
                    ->pluck('id')
                    ->toArray()
            )
                ->where('type', AccountingTransactionLine::TYPE_CREDIT)
                ->count()
        );

        $dateTransferShipment = Carbon::now()->format('Y-m-d');

        $warehouseTransferShipment->update(['shipped_at' => $dateTransferShipment]);
        $this->resetUpdatedAt($warehouseTransferShipment->accountingTransaction);

        (new SalesOrderLineFinancialManager())->calculate();
        $this->manager->sync();

        $this->assertDatabaseHas((new AccountingTransaction())->getTable(), [
            'transaction_date' => $dateTransferShipment,
            'link_id' => $warehouseTransferShipment->id,
            'link_type' => WarehouseTransferShipment::class,
        ]);

        $this->assertDatabaseCount((new AccountingTransaction())->getTable(), 1);

        $warehouseTransferShipmentItem = $warehouseTransferShipment->shipmentLines()->first();
        $inventoryMovement = $warehouseTransferShipmentItem->inventoryMovements()->first();
        $fifoLayer = $inventoryMovement->layer;

        $fifoLayer->update(['total_cost' => 2 * 55, 'original_quantity' => 2]);
        $warehouseTransferShipmentItem->update(['quantity' => 2]);

        $this->resetUpdatedAt($warehouseTransferShipment->accountingTransaction);

        (new SalesOrderLineFinancialManager())->calculate();
        $this->manager->sync();

        $this->assertDatabaseHas((new AccountingTransaction())->getTable(), [
            'link_id' => $warehouseTransferShipment->id,
            'link_type' => WarehouseTransferShipment::class,
            'total' => 2 * 55.00,
        ]);
        $this->assertDatabaseCount((new AccountingTransaction())->getTable(), 1);

        $inventoryMovement->update(['quantity' => 2]);
        $fifoLayer->update(['total_cost' => 2 * 60, 'original_quantity' => 2]);

        $this->resetUpdatedAt($warehouseTransferShipment->accountingTransaction);

        (new SalesOrderLineFinancialManager())->calculate();
        $this->manager->sync();

        $this->assertDatabaseHas((new AccountingTransaction())->getTable(), [
            'link_id' => $warehouseTransferShipment->id,
            'link_type' => WarehouseTransferShipment::class,
            'total' => 2 * 60,
        ]);
        $this->assertDatabaseCount((new AccountingTransaction())->getTable(), 1);

        $this->assertDatabaseHas((new AccountingTransactionLine())->getTable(), [
            'link_type' => WarehouseTransferShipmentLine::class,
            'quantity' => 2,
            'amount' => 60,
        ]);

        $warehouseTransferShipment->delete();

        $this->manager->sync();

        $this->assertDatabaseCount((new AccountingTransaction())->getTable(), 0);
        $this->assertDatabaseCount((new AccountingTransactionLine())->getTable(), 0);
    }

    /**
     * @throws Throwable
     * @throws InsufficientStockException
     * @throws WarehouseTransferHasNoProductsException
     * @throws WarehouseTransferOpenException
     */
    private function testWarehouseTransferShipmentReceipts(): void
    {
        $fromWarehouse = Warehouse::first();
        $toWarehouse = Warehouse::factory()->create();
        $product = Product::factory()->create();

        $product->setInitialInventory($fromWarehouse->id, 50, 5.00);

        $warehouseTransfer = WarehouseTransfer::factory()
            ->hasWarehouseTransferLines(1, [
                'quantity' => 1,
                'product_id' => $product->id,
            ])
            ->create([
                'from_warehouse_id' => $fromWarehouse->id,
                'to_warehouse_id' => $toWarehouse->id,
                'transfer_status' => WarehouseTransfer::TRANSFER_STATUS_DRAFT,
            ]);

        app(WarehouseTransferManager::class)->openWarehouseTransfer($warehouseTransfer, []);
        app(WarehouseTransferManager::class)->receiveShipment($warehouseTransfer, WarehouseTransferReceiptData::from([
            'receipt_date' => Carbon::now(),
            'products' => WarehouseTransferReceiptProductData::collection([
                WarehouseTransferReceiptProductData::from([
                    'id' => $product->id,
                    'quantity' => 1,
                ])
            ])
        ]));

        $warehouseTransferShipment = $warehouseTransfer->shipment;
        $warehouseTransferReceipt = $warehouseTransferShipment->receipts()->first();

        $this->manager->sync();

        $this->assertEquals(
            1,
            AccountingTransaction::withWarehouseTransferReceipts(
                $warehouseTransferReceipt->id
            )
                ->where('type', AccountingTransactionTypeEnum::WAREHOUSE_TRANSFER_RECEIPT)
                ->count()
        );

        $this->assertEquals(
            1,
            AccountingTransactionLine::withWarehouseTransferReceiptLines(
                $warehouseTransferReceipt->receiptLines()
                    ->pluck('id')
                    ->toArray()
            )
                ->where('type', AccountingTransactionLine::TYPE_DEBIT)
                ->count()
        );

        $this->assertEquals(
            1,
            AccountingTransactionLine::withWarehouseTransferReceiptLines(
                $warehouseTransferReceipt->receiptLines()
                    ->pluck('id')
                    ->toArray()
            )
                ->where('type', AccountingTransactionLine::TYPE_CREDIT)
                ->count()
        );

        $dateTransferReceipt = Carbon::now()->format('Y-m-d');

        $warehouseTransferReceipt->update(['received_at' => $dateTransferReceipt]);
        $this->resetUpdatedAt($warehouseTransferReceipt->accountingTransaction);

        (new SalesOrderLineFinancialManager())->calculate();
        $this->manager->sync();

        $this->assertDatabaseHas((new AccountingTransaction())->getTable(), [
            'transaction_date' => $dateTransferReceipt,
            'link_id' => $warehouseTransferReceipt->id,
            'link_type' => WarehouseTransferShipmentReceipt::class,
        ]);

        $this->assertDatabaseCount((new AccountingTransaction())->getTable(), 2);

        $warehouseTransferReceiptItem = $warehouseTransferReceipt->receiptLines()->first();
        $inventoryMovement = $warehouseTransferReceiptItem->inventoryMovements()->first();
        $fifoLayer = $warehouseTransferReceiptItem->getFifoLayer();

        $fifoLayer->update(['total_cost' => 2 * 55, 'original_quantity' => 2]);
        $warehouseTransferReceiptItem->update(['quantity' => 2]);

        $this->resetUpdatedAt($warehouseTransferReceipt->accountingTransaction);

        (new SalesOrderLineFinancialManager())->calculate();
        $this->manager->sync();

        $this->assertDatabaseHas((new AccountingTransaction())->getTable(), [
            'link_id' => $warehouseTransferReceipt->id,
            'link_type' => WarehouseTransferShipmentReceipt::class,
            'total' => 2 * 55.00,
        ]);
        $this->assertDatabaseCount((new AccountingTransaction())->getTable(), 2);

        $inventoryMovement->update(['quantity' => 2]);
        $fifoLayer->update(['total_cost' => 2 * 60, 'original_quantity' => 2]);

        $this->resetUpdatedAt($warehouseTransferReceipt->accountingTransaction);

        (new SalesOrderLineFinancialManager())->calculate();
        $this->manager->sync();

        $this->assertDatabaseHas((new AccountingTransaction())->getTable(), [
            'link_id' => $warehouseTransferReceipt->id,
            'link_type' => WarehouseTransferShipmentReceipt::class,
            'total' => 2 * 60,
        ]);
        $this->assertDatabaseCount((new AccountingTransaction())->getTable(), 2);

        $this->assertDatabaseHas((new AccountingTransactionLine())->getTable(), [
            'link_type' => WarehouseTransferShipmentReceiptLine::class,
            'quantity' => 2,
            'amount' => 60,
        ]);

        $warehouseTransferReceipt->delete();

        $this->manager->sync();

        $this->assertDatabaseCount((new AccountingTransaction())->getTable(), 1);
        $this->assertDatabaseCount((new AccountingTransactionLine())->getTable(), 2);
    }

    /**
     * @throws Exception|Throwable
     */
    public function test_customer_returns_link_to_parent(): void
    {
        $salesOrder = SalesOrder::factory()->createWithSalesCredit();

        /** @var SalesCredit $salesCredit */
        $salesCredit = $salesOrder->salesCredits->first();

        (new SalesOrderLineFinancialManager())->calculate();
        $this->manager->sync();

        SalesCreditReturn::factory()
            ->returned($salesCredit, 1);
        /** @var SalesCreditReturn $salesCreditReturn */
        $salesCreditReturn = $salesCredit->salesCreditReturns()->first();

        (new SalesOrderLineFinancialManager())->calculate();
        $this->manager->sync();

        $this->assertDatabaseHas((new AccountingTransaction())->getTable(), [
            'link_id' => $salesCreditReturn->id,
            'link_type' => SalesCreditReturn::class,
            'parent_id' => $salesCredit->accountingTransaction->id,
        ]);
    }

    /**
     * @throws Exception|Throwable
     */
    public function test_it_can_create_batchable_accounting_transactions(): void
    {
        /** @var AmazonIntegrationInstance $amazonIntegrationInstance */
        $amazonIntegrationInstance = AmazonIntegrationInstance::factory([
            'integration_settings' => [
                'batch_sales_order_invoices' => true,
            ],
        ])->hasSalesChannel()->create();

        /** @var SalesChannel $salesChannel */
        $salesChannel = $amazonIntegrationInstance->salesChannel;

        assertEquals(true, $amazonIntegrationInstance->integration_settings['batch_sales_order_invoices']);

        $salesOrder = SalesOrder::factory([
            'sales_channel_id' => $salesChannel->id,
        ])->hasSalesOrderLines(3)->create();

        (new SalesOrderLineFinancialManager())->calculate();
        $this->manager->sync();

        $this->assertEquals(
            1,
            AccountingTransaction::withSalesOrders($salesOrder->id)
                ->where('type', AccountingTransaction::TYPE_SALES_ORDER_INVOICE)
                ->where('is_batchable', true)
                ->count()
        );

        $this->assertEquals(
            3,
            AccountingTransactionLine::withSalesOrderLines(
                $salesOrder->salesOrderLines()
                    ->pluck('id')
                    ->toArray()
            )
                ->where('type', AccountingTransactionLine::TYPE_SALES_INVOICE_LINE)
                ->count()
        );
    }

    /**
     * @throws Exception
     * @throws Throwable
     */
    public function testDropshipPurchaseOrders(): void
    {
        $cogsAccount = NominalCode::find(Helpers::setting(Setting::KEY_NC_MAPPING_COGS));

        $customer = Customer::factory()
            ->hasAddresses()
            ->create();

        $salesOrder = SalesOrder::factory()
            ->withLines()
            ->create([
                'customer_id' => $customer->id,
            ]);

        /** @var PurchaseOrder $purchaseOrder */
        $purchaseOrder = PurchaseOrder::factory()
            ->hasPurchaseOrderLines()
            ->create([
                'sales_order_id' => $salesOrder->id,
                'destination_warehouse_id' => null,
                'destination_address_id' => $customer->addresses()->first()
            ]);

        $this->manager->sync();

        $this->assertEquals(
            1,
            AccountingTransaction::withPurchaseOrders($purchaseOrder->id)
                ->where('type', AccountingTransaction::TYPE_PURCHASE_ORDER)
                ->count()
        );

        $this->assertEquals(
            1,
            AccountingTransactionLine::withPurchaseOrderLines(
                $purchaseOrder->purchaseOrderLines()->first()
                    ->pluck('id')
                    ->toArray()
            )
                ->where('type', AccountingTransactionLine::TYPE_PURCHASE_ORDER_LINE)
                ->count()
        );

        $this->assertDatabaseHas(AccountingTransactionLine::class, [
            'link_type' => PurchaseOrderLine::class,
            'type' => AccountingTransactionLine::TYPE_PURCHASE_ORDER_LINE,
            'nominal_code_id' => $cogsAccount->id,
        ]);
    }

    /**
     * @throws Exception|Throwable
     */
    public function testDropshipPurchaseOrdersInvoices(): void
    {
        $cogsAccount = NominalCode::find(Helpers::setting(Setting::KEY_NC_MAPPING_COGS));

        $customer = Customer::factory()
            ->hasAddresses()
            ->create();

        $supplier = Supplier::factory()
            ->hasWarehouses(1, [
                'type' => Warehouse::TYPE_SUPPLIER,
            ])
            ->create();

        $warehouse = $supplier->warehouses()->first();

        $salesOrder = SalesOrder::factory()
            ->has(
                SalesOrderLine::factory([
                    'warehouse_id' => $warehouse->id,
                ])
            )
            ->has(
                SalesOrderFulfillment::factory([
                    'warehouse_id' => $warehouse->id,
                ])
                    ->has(
                        SalesOrderFulfillmentLine::factory()
                            ->state(function ($attributes, SalesOrderFulfillment $salesOrderFulfillment) {
                                return [
                                    'sales_order_line_id' => $salesOrderFulfillment->salesOrder->salesOrderLines()->first()->id,
                                    'quantity' => 1,
                                ];
                            })
                    )
            )
            ->create([
                'customer_id' => $customer->id,
            ]);

        /** @var PurchaseOrder $purchaseOrder */
        $purchaseOrder = PurchaseOrder::factory()
            ->hasPurchaseOrderLines()
            ->received(1)
            ->has(
                PurchaseInvoice::factory()
                    ->has(
                        PurchaseInvoiceLine::factory()
                            ->state(function ($attributes, PurchaseInvoice $purchaseInvoice) {
                                return [
                                    'purchase_order_line_id' => $purchaseInvoice->purchaseOrder->purchaseOrderLines()->first()->id,
                                    'quantity_invoiced' => 1,
                                ];
                            })
                    )
            )
            ->create([
                'sales_order_id' => $salesOrder->id,
                'destination_warehouse_id' => null,
                'destination_address_id' => $customer->addresses()->first()
            ]);

        $this->manager->sync();

        $this->assertEquals(
            1,
            AccountingTransaction::withPurchaseInvoices($purchaseOrder->purchaseInvoices()->first()->id)
                ->where('type', AccountingTransaction::TYPE_PURCHASE_ORDER_INVOICE)
                ->count()
        );

        $this->assertEquals(
            1,
            AccountingTransactionLine::withPurchaseInvoiceLines(
                $purchaseOrder->purchaseInvoices()->first()->purchaseInvoiceLines()
                    ->pluck('id')
                    ->toArray()
            )
                ->where('type', AccountingTransactionLine::TYPE_PURCHASE_INVOICE_LINE)
                ->count()
        );

        $this->assertDatabaseHas(AccountingTransactionLine::class, [
            'link_type' => PurchaseInvoiceLine::class,
            'type' => AccountingTransactionLine::TYPE_PURCHASE_INVOICE_LINE,
            'nominal_code_id' => $cogsAccount->id,
        ]);

        // Assert receiving does not generate accounting transaction

        $this->assertDatabaseMissing(AccountingTransaction::class, [
            'link_type' => PurchaseOrderShipmentReceipt::class,
        ]);

        $this->assertDatabaseMissing(AccountingTransactionLine::class, [
            'link_type' => PurchaseOrderShipmentReceiptLine::class,
        ]);

        // Assert fulfilling does not generate accounting transaction

        $this->assertDatabaseMissing(AccountingTransaction::class, [
            'link_type' => SalesOrderFulfillment::class,
        ]);

        $this->assertDatabaseMissing(AccountingTransactionLine::class, [
            'link_type' => SalesOrderFulfillmentLine::class,
        ]);
    }

    /**
     * @throws Throwable
     */
    public function test_it_cant_update_locked_accounting_transaction(): void
    {
        $salesOrder = SalesOrder::factory([
            'order_date' => Carbon::now()->subDays(1),
        ])->create();

        $accountingTransaction = AccountingTransaction::factory([
            'type' => AccountingTransaction::TYPE_SALES_ORDER_INVOICE,
            'link_id' => $salesOrder->id,
            'link_type' => SalesOrder::class,
            'updated_at' => Carbon::now()->subDays(3),
        ])
            ->create();

        $salesOrder->customer->name = 'newCustomer';
        $salesOrder->customer->save();
        $salesOrder->update();

        $this->manager->sync();

        $this->assertDatabaseHas(AccountingTransaction::class, [
            'id' => $accountingTransaction->id,
            'name' => 'newCustomer'
        ]);

        $salesOrder->customer->name = 'newCustomer2';
        $salesOrder->customer->save();
        $salesOrder->updated_at = Carbon::now()->addDay();
        $salesOrder->update();

        $accountingTransaction->is_locked = true;
        $accountingTransaction->save();

        $this->manager->sync();

        // Update of transaction did not happen
        $this->assertDatabaseHas(AccountingTransaction::class, [
            'id' => $accountingTransaction->id,
            'name' => 'newCustomer'
        ]);
    }

    /**
     * @throws Throwable
     */
    public function test_it_cant_create_accounting_transaction_prior_to_start_date(): void
    {
        $startDate = Carbon::parse('2021-05-01');
        $orderDate = Carbon::parse('2021-01-01');

        app(SettingRepository::class)->set(Setting::KEY_ACCOUNTING_START_DATE, $startDate->toDateTimeString());

        SalesOrder::factory([
            'order_date' => $orderDate,
        ])->create();

        $this->manager->sync();

       $this->assertDatabaseEmpty(AccountingTransaction::class);
    }

    /**
     * @throws Throwable
     */
    public function test_it_cant_create_new_transactions_or_update_existing_before_lock_date(): void
    {
        $lockDate = Carbon::parse('2021-05-01');
        $orderDate = Carbon::parse('2021-01-01');

        app(SettingRepository::class)->set(Setting::KEY_ACCOUNTING_LOCK_DATE, $lockDate->toDateTimeString());

        $salesOrder = SalesOrder::factory([
            'order_date' => $orderDate,
        ])->create();

        $this->manager->sync();

        $this->assertDatabaseEmpty(AccountingTransaction::class);

        $lockDate = Carbon::parse('2020-01-01');
        app(SettingRepository::class)->set(Setting::KEY_ACCOUNTING_LOCK_DATE, $lockDate->toDateTimeString());

        $this->manager->sync();

        $this->assertDatabaseCount(AccountingTransaction::class, 1);

        $salesOrder->customer->name = 'New customer name';
        $salesOrder->customer->save();
        $salesOrder->updated_at = Carbon::now()->addDay();
        $salesOrder->save();

        $this->manager->sync();

        $this->assertDatabaseHas(AccountingTransaction::class, [
            'name' => 'New customer name'
        ]);

        $salesOrder->customer->name = 'New customer name 2';
        $salesOrder->customer->save();
        $salesOrder->updated_at = Carbon::now()->addDay();
        $salesOrder->save();

        $lockDate = Carbon::parse('2021-05-01');
        app(SettingRepository::class)->set(Setting::KEY_ACCOUNTING_LOCK_DATE, $lockDate->toDateTimeString());

        $this->manager->sync();

        $this->assertDatabaseHas(AccountingTransaction::class, [
            'name' => 'New customer name'
        ]);

    }
}
