<?php

namespace Tests\Concerns;

use App\Exceptions\PurchaseOrder\NotOpenPurchaseOrderException;
use App\Exceptions\PurchaseOrder\ReceivePurchaseOrderLineException;
use App\Models\Product;
use App\Models\PurchaseOrder;
use App\Models\PurchaseOrderShipmentReceipt;
use App\Models\SalesOrder;
use App\Models\Warehouse;
use App\Services\PurchaseOrder\ShipmentManager;
use Carbon\Carbon;
use Illuminate\Support\Arr;
use Illuminate\Support\Collection;
use Throwable;

trait UseSimpleSkuFactories
{
    protected Product $product;

    protected Warehouse $warehouse;

    public function setUpProduct(): void
    {
        $this->product = Product::factory(1)->create()->first();
    }

    public function setUpWarehouse(): void
    {
        $this->warehouse = Warehouse::factory(1)->create()->first();
    }

    /**
     * @throws Throwable
     */
    public function setInitialStock(int $quantity): void
    {
        $this->product->setInitialInventory($this->warehouse->id, $quantity);
    }

    private function setUpPurchaseOrders(array|int $lineQuantities): Collection
    {
        $lineQuantities = Arr::wrap($lineQuantities);
        $purchaseOrders = collect();
        foreach ($lineQuantities as $lineQuantity) {
            $purchaseOrders->push(
                PurchaseOrder::factory(1)
                    ->hasPurchaseOrderLines(1, [
                        'quantity' => $lineQuantity,
                        'product_id' => $this->product->id,
                    ])
                    ->create(['destination_warehouse_id' => $this->warehouse->id])
                    ->first()
            );
        }

        return $purchaseOrders;
    }

    /**
     * @throws NotOpenPurchaseOrderException
     * @throws ReceivePurchaseOrderLineException
     */
    private function receivePurchaseOrder(PurchaseOrder $purchaseOrder, $releaseQuantity = null): PurchaseOrderShipmentReceipt
    {
        $shipmentManager = new ShipmentManager();

        $payload = [
            'received_at' => Carbon::now(),
            'purchase_order_id' => $purchaseOrder->id,
            'warehouse_id' => $purchaseOrder->destination_warehouse_id,
        ];

        if ($releaseQuantity) {
            $purchaseOrderLine = $purchaseOrder->purchaseOrderLines->first();
            $payload['receipt_lines'] = [
                [
                    'purchase_order_line_id' => $purchaseOrderLine->id,
                    'quantity' => $releaseQuantity,
                ],
            ];
        }

        return $shipmentManager->receiveShipment($payload);
    }

    private function setUpSalesOrders(array|int $lineQuantities): Collection
    {
        $lineQuantities = Arr::wrap($lineQuantities);
        $salesOrders = collect();
        foreach ($lineQuantities as $lineQuantity) {
            $salesOrders->push(
                SalesOrder::factory(1)
                    ->hasSalesOrderLines(1, [
                        'quantity' => $lineQuantity,
                        'product_id' => $this->product->id,
                        'warehouse_id' => $this->warehouse->id,
                    ])
                    ->create(['order_date' => now()])
                    ->first()
            );
        }

        return $salesOrders;
    }
}
