import moment from 'moment-timezone'

/**
 * moment.defaultZone.name is the user's timezone
 */

export const SERVER_DATE_FORMAT = 'YYYY-MM-DD'

export const SIMPLE_8601_FORMAT = 'YYYY-MM-DD HH:mm:ss'

export const SERVER_DATE_TIME_FORMAT = `${SERVER_DATE_FORMAT}THH:mm:ssZ`

export const buildTimeFromString = (time) => {
  let [hh, mm, h] = time.split(':')

  h = Number(hh)
  if (h === 0) h = 12
  const A = h >= 12 ? 'PM' : 'AM'
  if (h > 12) h -= 12

  if (isNaN(Number(mm))) mm = '00'

  return { A, h: h.toString(), mm }
}

export const buildTimeFromObject = (obj) => {
  let h

  if (obj.h === '') {
    h = '00'
  } else if (obj.A === 'PM' && obj.h !== '12') {
    h = (Number(obj.h) + 12).toString()
  } else if (obj.A === 'AM' && obj.h === '12') {
    h = '00'
  } else {
    h = obj.h
  }

  let mm = obj.mm || '00'
  if (!mm || mm === '0' || isNaN(Number(mm))) {
    mm = '00'
  }

  return `${h.padStart(2, '0')}:${mm.padStart(2, '0')}`
}

export const toUserTimezone = (value, formatOut = null) => {
  if (!value) return ''
  const dateString = checkIsDateTimeString(value) ? value : prepareDateTimeStringToFormat(value)
  const userTz = moment.utc(dateString, SERVER_DATE_TIME_FORMAT).tz(moment.defaultZone.name)

  return formatOut ? userTz.format(formatOut) : userTz
}

export const fromUserTimezoneToUtc = (value) => {
  if (!value) return ''
  return moment.tz(value, moment.defaultZone.name).utc().format(SERVER_DATE_TIME_FORMAT)
}

export const fromTimezoneToUtc = (value, format) => {
  if (!value) return ''
  return moment.tz(value, format).utc().format(SERVER_DATE_TIME_FORMAT)
}

export const fromUtcToTimezone = (value, timezone) => {
  if (!value) return ''
  return moment.tz(value, 'utc').tz(timezone).format('YYYY-MM-DDTHH:mm:ss')
}

export const convertShedulersToUserTimezone = (shedulers = [], timeFormat = 'HH:mm') => {
  if (!shedulers || !Array.isArray(shedulers) || shedulers.length === 0) return []

  return shedulers.map(({ time, days }) => {
    const [hour, minute] = time.split(':')
    const listOfDays = days.map(day => {
      const m = moment.utc().day(day).hour(hour).minute(minute).tz(moment.defaultZone.name)
      return m.format('dddd').toLowerCase()
    })
    return {
      time: moment.utc().hour(hour).minute(minute).tz(moment.defaultZone.name).format(timeFormat),
      days: listOfDays
    }
  })
}

export const dateIsBetween = (date, start, end) => {
  const [day] = date.split(' ')
  const momentDay = moment.isMoment(day) ? day.startOf('day') : moment(day).startOf('day')
  const momentStartDate = moment.isMoment(start) ? start.startOf('day') : moment(start).startOf('day')
  const momentEndDate = moment.isMoment(end) ? end.startOf('day') : moment(start).startOf('day')

  return momentDay.isSameOrAfter(momentStartDate.format(SERVER_DATE_FORMAT)) && momentDay.isSameOrBefore(momentEndDate.format(SERVER_DATE_FORMAT))
}

export const checkIsDateString = (value) => {
  if (!value || typeof value !== 'string') return false
  return value.search(/[\d]{4}-[\d]{2}-[\d]{2}/) !== -1
}

export const toUTC = (value, format) => {
  if (!value || !format) {
    console.warn('timeHelper toUTC, has wrong params')
    return value
  }
  value = moment.isMoment(value) ? value : moment(value, format)

  return value.utc()
}

export const prepareDateTimeString = (value) => {
  if (!value) return null
  const [date,timeWithGap] = value.split('T')
  const signIndex = timeWithGap.search(/[+,-]/)
  const gapSign = timeWithGap.substring(signIndex, signIndex + 1)
  const [time,gap] = timeWithGap.split(gapSign)
  const [h,m] = gap.split(':')
  if (!gapSign) return value
  let momentObj = moment(`${date}T${time}Z`)

  momentObj = gapSign === '-' ?
    momentObj.subtract(h, 'hours').subtract(m,'minutes') :
    momentObj.add(h, 'hours').add(m,'minutes')

  return momentObj.utc().format()
}

export const toDateFormat = (value, format) => moment.tz(value, SERVER_DATE_FORMAT).format(format)

export const getDayOfWeek = (value, format = 'dddd') => moment(value).format(format)

export const checkIsDateTimeString = (value) => {
  if (!value || typeof value !== 'string') return false
  return value.search(/[\d]{4}-[\d]{2}-[\d]{2}T[\d]{2}:[\d]{2}:[\d]{2}Z/) !== -1 ||
    value.search(/[\d]{4}-[\d]{2}-[\d]{2}T[\d]{2}:[\d]{2}:[\d]{2}[+,-][\d]{2}:[\d]{2}/) !== -1
}

export const prepareDateTimeStringToFormat = (value, splitBy = ' ') => {
  if (checkIsDateTimeString(value)) return value
  if (!value.includes('UTC') && value.includes('T')) return `${value}Z`
  const [date,time] = value.split(splitBy)

  return `${date}T${time}Z`
}

export const checkIsApiDateTimeFormat = (value) => {
  if (!value || typeof value !== 'string') return false

  return value.search(/[\d]{4}-[\d]{2}-[\d]{2}T[\d]{2}:[\d]{2}/) !== -1
}

export const minutesToHoursAndMinutes = (totalMinutes, format = 'string') => {
  const total = totalMinutes < 0 ? totalMinutes * -1 : totalMinutes
  let hours = Math.floor(total / 60)
  let minutes = total % 60
  let offsetSign = '+'
  if (hours < 10) hours = `0${hours}`
  if (minutes < 10) minutes = `0${minutes}`
  if (totalMinutes < 0) offsetSign = '-'

  return format === 'string' ? `${offsetSign}${hours}:${minutes}` : { hours, minutes }
}
