import exactMath from 'exact-math'

const PRODUCT_TYPES = {
  PRODUCT: 'product',
  UNMAPPED: 'unmapped',
  NON_PRODUCT: 'non-product',
  EXTERNALLY_FULFILLED: 'externally-fulfilled'
}

export const productTypeValidation = (value) => Object.values(PRODUCT_TYPES).includes(value)

export const isProductType = (type) => type === PRODUCT_TYPES.PRODUCT

export const isNonProductType = (type) => type === PRODUCT_TYPES.NON_PRODUCT

export const isUnmappedType = (type) => type === PRODUCT_TYPES.UNMAPPED

export const isExternallyFulfilledType = (type) => type === PRODUCT_TYPES.EXTERNALLY_FULFILLED

export const calculateFulfilledQty = (fulfillments, qtyProperty = 'quantity') => fulfillments.reduce((acc, item) => {
  acc = exactMath.add(acc, Number(item[qtyProperty] || 0))
  return acc
}, 0)

export const calculateUnfulfilledQty = (product, fulfilledQty, qtyProperty = 'item_reserved_quantity') => {
  const unfulfilledQty = exactMath.sub(Number(product[qtyProperty] || 0), Number(fulfilledQty))

  return unfulfilledQty >= 0 ? unfulfilledQty : 0
}

export const isWarehouseNotFulfillable = (warehouseProducts = []) => {
  // if all items are not fulfillable then warehouse is not fulfillable
  let isNotFulfillableCount = 0
  for (let i = 0; i <= warehouseProducts.length; i++) {
    let product = warehouseProducts[i]
    if (product) {
      const fulfillmentsData = product.warehouse?.fulfillments_data || []
      const fulfilledQty = calculateFulfilledQty(fulfillmentsData)
      product.unfulfilled_quantity = calculateUnfulfilledQty(product, fulfilledQty)
      if (product.backordered_quantity) {
        if (Number(product.backordered_quantity.total_released) + Number(product.unfulfilled_quantity) - Number(product.backordered_quantity.total_backordered) <= 0) {
          isNotFulfillableCount++
        }
      } else if (product.warehouse.type !== 'amazon_fba') {
        if (product.item_reserved_quantity == 0 || Number(product.unfulfilled_quantity) <= 0) {
          isNotFulfillableCount++
        }
      }
    }
  }
  return isNotFulfillableCount === warehouseProducts.length
}
