import { chain, sumBy } from 'lodash'
import { calculateAmountBasedOnRate } from '@/support/calculateProportions'
import exactMath from 'exact-math'
import { API_MODELS } from '@/assets/constants'

export const NEED_PREPARE_KEYS = ['quantity', 'sku', 'barcode', 'received', 'nominal_code_id', 'nominal_code', 'name', 'stock', 'discount_amount', 'discount_rate', 'eta', 'img_url', 'inventory']

export const getTaxRateObject = (tax, total = 0, key = null, isIncluded = false) => {
  const { id = null, name = null, rate = null } = tax
  return { id, name, rate, total, amount: calculateTaxAmount(rate, total, isIncluded), key }
}

export const groupedTaxes = (taxes) => {
  return chain(taxes.filter(tax => tax.id !== 'null')).groupBy('id')
    .map((value, key) => ({ name: value[0].rate !== undefined ? `${value[0].rate} % - ${value[0].name}`
      : value[0].name,
    amount: sumBy(value, 'amount') })
    )
    .value()
}

export const getTaxIdentifier = (prefix, value, glue = '_') => {
  return `${prefix}${glue}${value}`
}

export const calculateTaxAmount = (rate = 0, total = 0, isIncluded = false) => {
  rate = exactMath.div(Number(rate), 100)
  total = Number(total)
  const amount = !isIncluded
    ? exactMath.mul(rate, total)
    : exactMath.div(exactMath.mul(rate, total), exactMath.add(rate, 1))

  return exactMath.round(amount, -3)
}

export const rowPrice = (row) => {
  const x = Number(row.item_price) || Number(row.price) || 0
  const y = Number(row.item_quantity) || Number(row.quantity) || 0
  return exactMath.mul(x, y)
}

export const rowDiscount = (row) => {
  const x = rowPrice(row)
  const y = Number(row.item_discount) || Number(row.discount) || 0
  const xTimesY = exactMath.mul(x, y)
  return exactMath.div(xTimesY, 100)
}

export const rowPriceAfterDiscount = (row) => {
  const x = rowPrice(row)
  const y = rowDiscount(row)
  return exactMath.sub(x, y)
}

export const sumOfAllOrderDiscounts = (discounts = [], totalBeforeTax = 0) => {
  if (!discounts || !discounts.length) return 0

  const sumOfAllOrderDiscounts = discounts
    .map(i => {
      // if amount exists then return amount, else if rate exists calculate amount based on rate and return amount
      if (i.hasOwnProperty('amount')) {
        return Number(i.amount)
      } else if (i.hasOwnProperty('rate')) {
        return calculateAmountBasedOnRate(i.rate, totalBeforeTax)
      }
      return 0
    })
    .reduce((acc, value) => exactMath.add(acc, value))

  return sumOfAllOrderDiscounts || 0
}

export const orderTotalBeforeOrderTax = (products) => {
  const allItemsTotalsArray = products.map(i => rowPriceAfterDiscount(i))
  if (!allItemsTotalsArray.length) return 0
  const sumOfAll = allItemsTotalsArray.reduce((acc, value) => exactMath.add(acc, value))

  return exactMath.round(sumOfAll, -2)
}

export const friendlyNameForType = ({ link_type, type }) => {
  if (type === 'coverage') return 'Purchase Order'
  if (!link_type) return ''
  let friendlyName = ''
  switch (link_type) {
    case API_MODELS.INVENTORY_ADJUSTMENT:
      friendlyName = 'Inventory Adjustment'
      break
    case API_MODELS.WAREHOUSE_TRANSFER:
      friendlyName = 'Warehouse Transfer'
      break
    case API_MODELS.PURCHASE_ORDER:
      friendlyName = 'Purchase Order'
      break
    case API_MODELS.STOCK_TAKE:
      friendlyName = 'Stock Take'
      break
    case 'initial_count':
      friendlyName = 'Initial Count'
      break
    case API_MODELS.SALES_CREDIT:
      friendlyName = 'Sales Credit'
      break
  }

  return friendlyName
}

export const preparePOLines = (lines = [], keys = NEED_PREPARE_KEYS) => {
  return lines.map(line => {
    let item = {}
    Object.keys(line).forEach(key => {
      let name = key
      if (keys.includes(key)) name = `${key}`
      if (key === 'amount') name = 'price'
      item = { ...item, [name]: line[key] }
    })
    item.discount = exactMath.mul(parseFloat(item.discount_rate || 0), 100) || 0
    return item
  })
}

export const rowTax = (row) => {
  const x = rowPriceAfterDiscount(row)
  const y = parseFloat(row?.tax_rate || 0)
  const xTimesY = exactMath.mul(x, y)
  return exactMath.div(xTimesY, 100)
}
