import moment from 'moment-timezone'
import router from '@/router.js'
import { isNavigationFailure, NavigationFailureType } from 'vue-router/src/util/errors'
import { checkIsDateString } from '@/support/timeHelper'

const pushToRoute = (params) => {
  router.push(params).catch(error => {
    if (!isNavigationFailure(error, NavigationFailureType.duplicated)) throw error
  })
}

const getDateFormat = (date, format='YYYY-MM-DD') => {
  return moment(date).format(format)
}

export function getValue (urlValue) {
  let value = decodeURIComponent(urlValue)
  value = value.replace(/\+/g, ' ')
  value = value.replace(/\%2B/g, '+')
  return value
}

export function getRuleValue (rule) {
  let decoded = getValue(rule[3])
  if (decoded.indexOf('mode') !== -1) {
    return JSON.parse(decoded)
  }
  return rule[2] === 'in' ? JSON.parse(decoded) : decoded
}

export function getRulesetFromUrl (storeNamespace) {
  // have to use this function with .call to pass 'this'
  if (this.$route.query && this.$route.query[storeNamespace]) {
    let query = this.$route.query && this.$route.query[storeNamespace]

    const args = { filters: { conjunction: 'and', filterSet: [] }, sortObjs: [], query: '' }
    const parts = String(query).split(';')
    let viewId, archived, density

    for (let i in parts) {
      if (parts.hasOwnProperty(i)) {
        let rule = parts[i].split(':')
        switch (rule.length) {
          case 2: {
            if (rule[0] === 'q') {
              args.query = rule[1]
            }
            if (rule[0] === 'v') { // view
              viewId = rule[1]
            }
            if (rule[0] === 'c') { // conjunction
              args.filters.conjunction = rule[1].toUpperCase() === 'AND' ? 'and' : 'or'
            }
            if (rule[0] === 'a') { // archived
              archived = rule[1]
            }
            if (rule[0] === 'd') { // density
              density = rule[1]
            }
            break
          }

          case 3: {
            if (rule[0] === 's') {
              args.sortObjs.push(
                {
                  id: Math.floor(Math.random() * 100000000),
                  column: rule[1],
                  ascending: (!(rule[2] === 'd')),
                  headerValue: rule[1].split('.').length ? rule[1].split('.')[0] : rule[1]
                }
              )
            }
            break
          }

          case 4: {
            if (rule[0] === 'f') {

              let value = getRuleValue(rule)
              let column = rule[1]

              // date filter support
              if (typeof value !== 'object' && value !== null) {
                if (value.indexOf('isdate') !== -1) {
                  const date = value.slice(6)
                  value = getDateFormat(date)
                } else if (value.indexOf('isDateRange') !== -1) {
                  const date = JSON.parse(value.slice(11))
                  value = {
                    from: getDateFormat(date.from),
                    to: getDateFormat(date.to)
                  }
                }
              }

              // boolean filter support
              if (value === 'true') value = true
              if (value === 'false') value = false

              // header value support
              let headerValue
              if (column.slice(-5) === '.name') {
                headerValue = column.slice(0, column.length - 5)
              } else {
                headerValue = column
              }

              let filterObj
              if (column === 'fulfillable') {
                filterObj = {
                  id: Math.floor(Math.random() * 100000000),
                  column,
                  operator: rule[2],
                  secondaryOption: (value && value.mode) || null,
                  headerValue,
                  value
                }
              } else {
                filterObj = {
                  id: Math.floor(Math.random() * 100000000),
                  column,
                  operator: rule[2],
                  dateOption: (value && value.mode) || (value.from && 'range') || null,
                  headerValue,
                  value
                }
              }

              args.filters.filterSet.push(filterObj)
            }
            break
          }
        }
      }
    }

    if (archived || archived === '0') args.archived = archived
    if (density) args.density = density
    return { viewId, getRuleset: args }
  }
  return {}
}

export function encodeURIString (string) {
  let encoded = encodeURIComponent(string)
  // our delimiters need to be double-encoded
  encoded = encoded.replace(/,/g, '%252C')
  encoded = encoded.replace(/:/g, '%253A')
  encoded = encoded.replace(/[\/]/g, '%252F')
  encoded = encoded.replace(/;/g, '%253B')
  encoded = encoded.replace(/\%2[Bb]/g, '%252B')

  // We want these to be readable.  Revert encoding.
  encoded = encoded.replace(/\%5B/g, '[')
  encoded = encoded.replace(/\%5D/g, ']')
  encoded = encoded.replace(/\%3C/g, '<')
  encoded = encoded.replace(/\%3E/g, '>')
  encoded = encoded.replace(/\%3D/g, '=')
  encoded = encoded.replace(/\%20/g, '+')

  return encoded
}

export function filterToString (filter) {
  const eus = encodeURIString
  const e = filter
  let encodedValue

  if (e.operator === 'in') {
    encodedValue = JSON.stringify(e.value)
  } else if (typeof e.value === 'string' && checkIsDateString(e.value)) {
    // if it's a date
    // (it's userTimezone date, not local)
    encodedValue = 'isdate' + eus(e.value)
  } else if (e.value && e.value.from && e.value.to) {
    // if it's a date range
    encodedValue = 'isDateRange' + eus(JSON.stringify(e.value))
  } else if (typeof e.value === 'object') {
    encodedValue = JSON.stringify(e.value)
    encodedValue = eus(encodedValue)
  } else {
    encodedValue = eus(e.value)
  }
  let string = `f:${eus(e.column)}:${eus(e.operator)}:`

  if (typeof encodedValue === 'undefined' || encodedValue === 'undefined' || encodedValue === '') {
    // even if the value is empty, some filters only require the operator
    return string
  }

  string += encodedValue
  return string
}

export function getParsedQuery (ruleset) {
  // Example query:
  //    c:<conjunction>,f:<col1>:<operator1>:<val1>[,f:<col2>:<operator2>:<val2>[,f:<col3>:<operator3>:<val3>]],s:<s-col>:<order>[,s:<s-col2>:<order>
  //
  //    Rules are separated by commas and parts of the rule are separated by
  //    colons.  Those characters and the "/" character must not appear in
  //    values when encoding into a single string like this one, so we have
  //    to replace them with their HTML entities:
  //    "," -> "%2C" ("%252C")
  //    ":" -> "%3A" ("%253A")
  //    "/" -> "%2F" ("%252F")
  //
  //    v: means that this is a table view
  //    c: means that this is a "conjunction" rule
  //
  //    <conjunction> can be "and" or "or"
  //
  //    f: means that this is a "filter" rule
  //
  //    <col1>, <col2>, <col3> are predefined filterable columns or data
  //    properties.
  //    <val1>, <val2>, <val3> are the values to search for.
  //    <operator1>, <operator2>, <operator3> are the operations to apply
  //    ("=", "<", ">", "%LIKE%", "LIKE%", "%LIKE").
  //
  //    s: means that this is a "sort" rule
  //
  //    Sort rules are applied in the order they appear.  Sort rules end
  //    with :a for ascending order, :d for descending order
  //
  //    a: means archived mode is on - 1 or off - 0
  //    d: means density is 1 for compact or 0 for expanded
  //    q: means query
  //
  //    c:and,f:title:%like%:helmet,f:price:>:30,f:price:<:60,s:price:a,s:discovered_at:d,a:0
  //
  //    means:   WHERE title like "%helmet%" AND price > 30 AND price < 60
  //             ORDER BY price ASC, discovered_at DESC
  //
  //    filters = {
  //        "conjunction": "and",
  //        "filterSet":[
  //            { "column": "title", "operator":"like", "value": "%helmet%" }
  //            { "column": "price", "operator":">",    "value": "60"       }
  //            { "column": "price", "operator":"<",    "value": "90"       }
  //        ]
  //    }
  //    sort_by = [
  //        {"column":"price", "ascending":true},
  //        {"column":"discovered_at", "ascending":true},

  const eus = encodeURIString
  const conjunction = []

  if (
    ruleset.filters.filterSet.length > 0 &&
    ruleset.filters.conjunction !== 'and'
  ) {
    conjunction.push('c:' + eus(ruleset.filters.conjunction))
  }

  const filters = ruleset.filters.filterSet.map(e => filterToString(e)).filter(x => x !== '')

  let sortBy = ruleset.sortObjs.map(e => 's:' + eus(e.column) + ':' + (e.ascending ? 'a' : 'd'))

  if (sortBy[0] === 's:u:a' || sortBy[0] === 's:u:d') {
    // Sort:  unsorted
    sortBy = sortBy.splice(0, 1)
  }

  let returnArray = [...conjunction, ...filters, ...sortBy]
  if (ruleset.archived || ruleset.archived === 0) returnArray.push(`a:${eus(ruleset.archived)}`)
  if (ruleset.density) returnArray.push(`d:${eus(ruleset.density)}`)
  if (ruleset.viewId && ruleset.viewId !== -1) returnArray.push(`v:${eus(ruleset.viewId)}`)
  if (ruleset.query) returnArray.push(`q:${eus(ruleset.query)}`)
  return returnArray.join(';')
}

export function addGivenQueryNameToUrl (key, value) {
  let currentQuery = JSON.parse(JSON.stringify(router.app._route.query))
  if (!currentQuery) return
  if (value) {
    pushToRoute({ query: { ...currentQuery, [key]: value } })
  } else {
    delete currentQuery[key]
    pushToRoute({ query: currentQuery })
  }
}

export function removeGivenQueryNameFromUrl (key) {
  let currentQuery = JSON.parse(JSON.stringify(router.app._route.query))
  if (!currentQuery) return
  if (currentQuery[key]) {
    delete currentQuery[key]
    pushToRoute({ query: currentQuery })
  }
}

export function addGivenValuePartToUrl (queryKey, value) {
  // for example to add d:compact to ?SKUTable=a:0 to have ?SKUTable=a:0;d:compact
  // use addGivenValuePartToUrl ('SKUTable', 'd:compact')
  let currentQuery = JSON.parse(JSON.stringify(router.app._route.query))
  if (!currentQuery) return

  if (currentQuery[queryKey]) {
    let parts = String(currentQuery[queryKey]).split(';') || []

    const index = parts.findIndex(p => p.substring(0, 2) === value.substring(0, 2))
    // if value starting with value first letter exists, edit it, else add the value
    if (index !== -1) {
      if (parts[index] === value) return
      parts[index] = value
    } else {
      parts.push(value)
    }
    const updatedValue = parts.join(';')
    pushToRoute({ query: { ...currentQuery, [queryKey]: updatedValue } })
  } else {
    currentQuery[queryKey] = value
    pushToRoute({ query: currentQuery })
  }
}

export function removeGivenValuePartFromUrl (queryKey, valueMarker) {
  let currentQuery = JSON.parse(JSON.stringify(router.app._route.query))
  if (!currentQuery) return
  if (currentQuery[queryKey]) {
    // find index to remove
    const parts = String(currentQuery[queryKey]).split(';')
    const removeIndex = parts.findIndex(p => p.substring(0, 2) === valueMarker.substring(0, 2))
    // remove that part of the value
    parts.splice(removeIndex, 1)
    // push remaining value to the url
    const updatedValue = parts.join(';')
    if (updatedValue) {
      pushToRoute({ query: { ...currentQuery, [queryKey]: updatedValue } })
    } else {
      // if the value is empty, remove that query key
      delete currentQuery[queryKey]
      pushToRoute({ query: currentQuery })
    }
  }
}
