import {
  cloneDeep,
  transform,
  isEqual,
  isArray,
  isObject,
  isEmpty
} from 'lodash'
import numeral from 'numeral'
import { baseURL } from '@/config/axiosConfig'
import moment from 'moment-timezone'
import exactMath from 'exact-math'
import formatDigit from './formatDigit'

export const REQUEST_DATE_FORMAT = 'YYYY-MM-DD'

/**
 * Compares the differences between the values of two objects
 * and returns an object with the differences. Returns an
 * empty object if there are no differences.
 *
 * Credit: https://gist.github.com/Yimiprod/7ee176597fef230d1451
 *
 * @param object
 * @param base
 * @param ignoredKeys
 */
export const difference = (object, base, ignoredKeys = ['updated_at', 'created_at', 'category_main', 'attributesByIds']) => {
  // Break references
  object = cloneDeep(object)
  base = cloneDeep(base)

  return transform(object, function (result, value, key) {
    // Handle numbers

    if (
      base[key] !== undefined &&
      ignoredKeys.indexOf(key) === -1 &&
      !isEqual(value, base[key])
    ) {
      if (isObject(value) && isObject(base[key])) {
        if (Object.keys(value).filter(i => !ignoredKeys.includes(i)).length !== Object.keys(base[key]).filter(i => !ignoredKeys.includes(i)).length) {
          result[key] = value // Different object lengths
        } else if (isArray(value) && isArray(base[key]) && value.length !== base[key].length) {
          result[key] = value // Different array lengths
        } else {
          // Find difference for next level
          const diff = difference(cloneDeep(value), cloneDeep(base[key]))

          // Only add difference when not empty
          if (!isEmpty(diff)) {
            // For array, we add entire value
            result[key] = isArray(value) || isObject(value) ? value : diff
          }
        }
      } else {
        result[key] = value
      }
    }
  })
}

export const sortName = (a, b) => {
  const nameA = a.name.toUpperCase()
  const nameB = b.name.toUpperCase()

  if (nameA < nameB) return -1
  if (nameA > nameB) return 1
  return 0
}

export const showAlert = (vm, options) => {
  if (options.permanent === undefined) options.permanent = false
  let notificationData = {
    closeButton: options.hasOwnProperty('closeButton'),
    permanent: options.permanent,
    id: options.id
  }
  if (options.html) notificationData.html = options.html
  // Hide any alerts with same id.
  if (options.id) hideAlert(vm, options.id)
  vm.$notification.$emit(options.type, notificationData)
}

export const hideAlert = (vm, id) => {
  vm.$notification.$emit('hide', id)
}

export const capitalize = (value) => {
  if (!value) return ''
  value = value.toString()
  return value.charAt(0).toUpperCase() + value.slice(1)
}

export const formatMoney = (value) => {
  if (!value) return '0.00'
  return numeral(value).format('0,0.00')
}

export const getImageUrl = (url) => {
  if (!url) return ''
  if (url.slice(0, 4) === 'http') return url
  return `${baseURL}${url}`
}

export const extractWarehouseName = (name, supplierName) => {
  if (!name) return name
  return name.substring(supplierName.length + 3)
}

export const booleanToString = (value) => {
  if (value === null || value === undefined) return 'No'
  return value ? 'Yes' : 'No'
}

export const isIncludes = (subject, search) => {
  if (!subject) return false
  return subject.toString().toLowerCase().indexOf(search.toString().toLowerCase()) !== -1
}

export const validateObjectRequiredProperties = (object, requiredProp, requiredLength) => {
  if (!isObject(object)) return false
  const keys = Object.keys(object)
  if (keys.length !== requiredLength) return false
  return keys.filter(key => requiredProp.includes(key)).length === requiredLength
}

export const collectionValidator = (collection, requiredProperties) => {
  const length = collection.length
  if (length === 0) return false
  let result = true
  const requiredLength = requiredProperties.length
  for (let i = 0; i < length; i++) {
    if (!validateObjectRequiredProperties(collection[i], requiredProperties, requiredLength)) {
      result = false
      break
    }
  }
  return result
}

export const hashCode = (s) => {
  return s.split('')
    .reduce((a, b) => {
      a = ((a << 5) - a) + b.charCodeAt(0)
      return a & a
    }, 0)
}

export const scrollToRefs = (vm, name, block = 'center') => {
  const el = vm.$refs[`${name}`]
  if (el) el[0].scrollIntoView({ behavior: 'smooth', block })
}

export const getRequestFiltersObject = (filters = {}) => {
  return Object.keys(filters).reduce((acc,key) => {
    if (filters.hasOwnProperty(key)) acc = {...acc, [`filter[${key}]`]: filters[key]}
    return acc
  },{})
}

export const dateRequestFormat = (value, format = 'YYYY-MM-DD') => {
  if (!moment.isMoment(value)) value = moment(value)
  return value.utc().format(format)
}

export const humanizeStringValue = (data, separator = '_', customize = null) => {
  let parts
  if (separator) parts = data.split(separator)
  // is PascalCase
  else parts = data.split(/(?=[A-Z])/)
  if (typeof customize === 'function') parts = parts.map(part => customize(part))

  return parts.join(' ')
}

export const getRandomId = (coefficient = 50000) => {
  return Math.ceil(Math.random() * coefficient)
}

export const toHumanFormat = (value) => {
  if (!value) return ''
  value = value.toString()
  return value.replaceAll('_', ' ')
}

export const convertBackendEntityToHuman = (value) => {
  if (!value) return ''

  const entity = value.split('\\').slice(-1)
  if (!entity || entity.length === 0) return ''

  return entity[0].split(/(?=[A-Z])/)
    .join(' ')
}

export const setTagsCollection = (tags) => {
  if (tags.length === 0) return []

  let result = []
  if (typeof tags[0] === 'object') result = tags
  if (typeof tags[0] === 'string') result = tags.map(tag => ({id:tag,name:tag}))

  return result
}

export const getSelectedRowsIds = (rows) => {
  return Object.entries(rows).reduce((acc,[key,value]) => {
    if (value) acc = [...acc, key]
    return acc
  },[])
}

export const getTableFilters = (filters = []) => {
  return {
    filters: JSON.stringify({
      conjunction: 'and',
      filterSet: filters.map(f => ({ ...f, id: getRandomId() })),
    }),
  }
}

export const toPercentage = (value, convertable = false) => {
  const percentage = convertable ? exactMath.mul(parseFloat(value),100) : value
  return `${formatDigit(percentage)} %`
}

export const apiModelName = (name) => name.split('\\').pop()

export const friendlyApiModelName = (data_name) => humanizeStringValue(apiModelName(data_name),null, null)

export const generateUniqueId = (ids = []) => {
  let id = getRandomId()
  while (ids.includes(id)) {
    id = getRandomId()
  }
  return id
}
