import {
  fetchWarehousesTransfers,
  getWarehouseTransferById,
  getWarehouseTransferReceipts,
  makeTransferDraft,
  openWarehouseTransfer,
  updateWarehouseTransfer
} from '@/services/WarehouseTransfersService'
import Vue from 'vue'
import {cloneDeep, isEmpty} from 'lodash'
import {difference} from '@/support/helpers'
import moment from 'moment-timezone'

export const WarehouseTransfersModule = {
  namespaced: true,
  state: {
    warehouseTransfers: {
      itemsList: [],
      tableSpecifications: {},
      maxPages: 0,
      meta: {},
      loading: false
    },
    selectedItem: {},
    selectedReceipt: {},
    selectedShipment: {},
    drawerMode: 'view',
    warehouseTransferDrawer: {
      show: false,
      hideNavigator: false,
      initialItemId: false,
      ignoreItemResetOnLaunch: false
    },
    warehouseTransferReceiptDrawer: {
      show: false,
      hideNavigator: false,
      initialItemId: false
    },
    warehouseTransferShipmentDrawer: {
      show: false,
      hideNavigator: false,
      initialItemId: false
    },
    receiptDrawerMode: 'view',
    shipmentDrawerMode: 'view',
    warehouseTransferLoading: false,
    editingTransfer: {},
    selectedTransferReceipts: [],
    selectedTransferShipments: [],

    warehouseTransferAddProductDrawer: {
      show: false
    },
    warehouseTransferAddProductDrawerMode: null,
    globalSettings: null,
    isFetch: false,
  },
  getters: {
    getNotes: state => state.selectedItem?.notes || [],
    isFetch: state => state.isFetch,
    getShowBarcode: state => {
      if (!state.globalSettings) return false
      const barcodeObj = state.globalSettings.find(i => i.key === 'sales_credit_show_barcode_in_frontend')
      if (!barcodeObj) return false
      return barcodeObj.value
    },
    getItemsList: state => state.warehouseTransfers.itemsList,
    getLoading: state => state.warehouseTransfers.loading,
    getTableSpecifications: state => state.warehouseTransfers.tableSpecifications,
    getMaxPages: state => state.warehouseTransfers.maxPages,
    getMeta: state => state.warehouseTransfers.meta,
    getSelectedItem: state => state.selectedItem,
    getSelectedTransferReceipt: state => state.selectedReceipt,
    getSelectedTransferShipment: state => state.selectedShipment,
    getSelectedTransferShipments: state => state.selectedTransferShipments,
    getSelectedItemIndex: state => {
      if (!state.selectedItem || !state.warehouseTransfers.itemsList.length) { return -1 }
      return state.warehouseTransfers.itemsList.findIndex(i => i.id === state.selectedItem.id)
    },
    getSelectedReceiptIndex: state => {
      if (!state.selectedReceipt || !state.selectedTransferReceipts.length) { return -1 }
      return state.selectedTransferReceipts.findIndex(i => i.id === state.selectedReceipt.id)
    },
    getSelectedShipmentIndex: state => {
      if (!state.selectedShipment || !state.selectedTransferShipments.length) { return -1 }
      return state.selectedTransferShipments.findIndex(i => i.id === state.selectedShipment.id)
    },
    getEditingTransfer: state => state.editingTransfer,
    getEditingFromWarehouse: state => {
      return state.editingTransfer.from_warehouse
    },
    getEditingToWarehouse: state => {
      return state.editingTransfer.to_warehouse
    },
    getDrawerMode: state => state.drawerMode,
    getWarehouseTransferDrawer: state => state.warehouseTransferDrawer,
    getWarehouseTransferLoading: state => state.warehouseTransferLoading,
    getReceiptDrawerMode: state => state.receiptDrawerMode,
    getShipmentDrawerMode: state => state.shipmentDrawerMode,
    getWarehouseTransferReceiptDrawer: state => state.warehouseTransferReceiptDrawer,
    getWarehouseTransferShipmentDrawer: state => state.warehouseTransferShipmentDrawer,
    getSomethingChanged: (state, getters) => {
      return !isEmpty(getters.getEditingDifference)
    },
    getEditingDifference: state => {
      if (!state.editingTransfer.id || !state.selectedItem.id) return {}
      const diff = difference(state.selectedItem, state.editingTransfer)
      // For shipping method, we match based on ids
      if (diff.shipping_method && state.selectedItem.shipping_method.id === state.editingTransfer.shipping_method.id) { delete diff.shipping_method }
      return diff
    },
    getSelectedTransferReceipts: state => state.selectedTransferReceipts,

    getWarehouseTransferAddProductDrawer: state => state.warehouseTransferAddProductDrawer,
    getWarehouseTransferAddProductDrawerMode: state => state.warehouseTransferAddProductDrawerMode
  },
  mutations: {
    SET_WAREHOUSES_TRANSFERS_ITEMS_LIST (state, value) {
      Vue.prototype.$set(state.warehouseTransfers, 'itemsList', value)
    },
    SET_WAREHOUSES_TRANSFERS_LOADING (state, value) {
      Vue.prototype.$set(state.warehouseTransfers, 'loading', value)
    },
    SET_WAREHOUSES_TRANSFERS_TABLE_SPECIFICATIONS (state, value) {
      Vue.prototype.$set(state.warehouseTransfers, 'tableSpecifications', value)
    },
    SET_WAREHOUSES_TRANSFERS_MAX_PAGES (state, value) {
      Vue.prototype.$set(state.warehouseTransfers, 'maxPages', value)
    },
    SET_WAREHOUSES_TRANSFERS_META (state, value) {
      Vue.prototype.$set(state.warehouseTransfers, 'meta', value)
    },
    SET_SELECTED_WAREHOUSE_TRANSFER (state, transfer) {
      state.selectedItem = transfer
    },
    SET_EDITING_WAREHOUSE_TRANSFER (state, transfer) {
      state.editingTransfer = transfer
    },
    SET_DRAWER_MODE (state, value) {
      state.drawerMode = value
    },
    SET_RECEIPT_DRAWER_MODE (state, value) {
      state.receiptDrawerMode = value
    },
    SET_SHIPMENT_DRAWER_MODE (state, value) {
      state.shipmentDrawerMode = value
    },
    SET_WAREHOUSE_TRANSFER_RECEIPT_DRAWER_FIELD (state, { field, value }) {
      Vue.prototype.$set(state.warehouseTransferReceiptDrawer, field, value)
    },
    SET_WAREHOUSE_TRANSFER_SHIPMENT_DRAWER_FIELD (state, { field, value }) {
      Vue.prototype.$set(state.warehouseTransferShipmentDrawer, field, value)
    },
    SET_WAREHOUSE_TRANSFER_DRAWER_FIELD (state, { field, value }) {
      Vue.prototype.$set(state.warehouseTransferDrawer, field, value)
    },
    SET_WAREHOUSE_TRANSFER_LOADING (state, loading) {
      state.warehouseTransferLoading = loading
    },
    SET_EDITING_TRANSFER_FIELD (state, { field, value }) {
      Vue.prototype.$set(state.editingTransfer, field, value)
    },
    SET_SELECTED_TRANSFER_RECEIPTS (state, value) {
      state.selectedTransferReceipts = value
    },
    SET_SELECTED_TRANSFER_RECEIPT (state, value) {
      state.selectedReceipt = value
    },
    SET_SELECTED_TRANSFER_SHIPMENT (state, value) {
      state.selectedShipment = value
    },
    SET_SELECTED_TRANSFER_SHIPMENTS (state, value) {
      state.selectedTransferShipments = value
    },

    SET_WAREHOUSE_TRANSFER_ADD_PRODUCT_DRAWER_FIELD (state, { field, value }) {
      Vue.prototype.$set(state.warehouseTransferAddProductDrawer, field, value)
    },
    SET_WAREHOUSE_TRANSFER_ADD_PRODUCT_DRAWER_MODE (state, value) {
      state.warehouseTransferAddProductDrawerMode = value
    },
    SET_GLOBAL_SETTINGS (state, value) {
      state.globalSettings = value
    },
    SET_IS_FETCH (state, value) {
      state.isFetch = value
    }
  },
  actions: {
    initiateSelectedItemEdit ({ state, commit }) {
      if (!state.selectedItem || !state.selectedItem.id) return
      // Fill in transfer date to current date if not set
      commit('SET_EDITING_WAREHOUSE_TRANSFER', cloneDeep(state.selectedItem))
    },

    async setSelectedWarehouseTransfer ({ commit }, ID) {
      if (!ID) return
      commit('SET_WAREHOUSE_TRANSFER_LOADING', true)
      const response = await getWarehouseTransferById(ID)

      if (response.data.eta) { response.data.eta = moment(response.data.eta).format('YYYY-MM-DD') }

      commit('SET_SELECTED_WAREHOUSE_TRANSFER', response.data)

      commit('SET_WAREHOUSE_TRANSFER_LOADING', false)
    },

    async setSelectedReceipt ({ state, commit }, id) {
      if (id === null) {
        commit('SET_SELECTED_TRANSFER_RECEIPT', null)
        return
      }

      const receipt = state.selectedTransferReceipts.find(receipt => receipt.id === id)
      if (receipt) {
        commit('SET_SELECTED_TRANSFER_RECEIPT', receipt)
      }
    },

    async setSelectedShipment ({ state, commit }, id) {
      if (id === null) {
        commit('SET_SELECTED_TRANSFER_SHIPMENT', null)
        return
      }
      const shipment = state.selectedTransferShipments.find(shipment => shipment.id === id)
      if (shipment) {
        commit('SET_SELECTED_TRANSFER_SHIPMENT', shipment)
      }
    },

    async updateSelectedTransfer ({ state, commit, getters }, payload) {
      try {
        const response = await updateWarehouseTransfer(state.selectedItem.id, payload)
        // Update the selected item
        commit('SET_SELECTED_WAREHOUSE_TRANSFER', response.data)
        // Also update the value in the list
        Vue.prototype.$set(state.warehouseTransfers.itemsList, getters.getSelectedItemIndex, response.data)
      } catch (e) {
        return Promise.reject(e)
      }
    },

    async openSelectedTransfer ({ state, commit, getters }, payload) {
      if (!state.selectedItem.id) return

      try {
        const response = await openWarehouseTransfer(state.selectedItem.id, payload)

        // Update the selected item
        commit('SET_SELECTED_WAREHOUSE_TRANSFER', response.data)
        // Also update the value in the list
        Vue.prototype.$set(state.warehouseTransfers.itemsList, getters.getSelectedItemIndex, response.data)
      } catch (e) {
        return Promise.reject(e)
      }
    },

    async makeSelectedTransferDraft ({ state, commit, getters }, payload) {
      try {
        const response = await makeTransferDraft(state.selectedItem.id, payload)

        // Update the selected item
        commit('SET_SELECTED_WAREHOUSE_TRANSFER', response.data)
        // Also update the value in the list
        Vue.prototype.$set(state.warehouseTransfers.itemsList, getters.getSelectedItemIndex, response.data)
      } catch (e) {
        return Promise.reject(e)
      }
    },

    async fetchSelectedReceipts ({ state, commit }) {
      if (!state.selectedItem) return
      try {
        const receipts = (await getWarehouseTransferReceipts(state.selectedItem.id)).data.map(receipt => {
          receipt.receipt_lines.map(line => {
            line.selected = true // Ensures that all are selected by default for edit detection
            return line
          })
          return receipt
        })
        commit('SET_SELECTED_TRANSFER_RECEIPTS', receipts)
      } catch (e) {
        return Promise.reject(e)
      }
    },

    // inventory adjustments requests
    async getItemsRequest ({ commit }, query) {
      try {
        commit('SET_WAREHOUSES_TRANSFERS_LOADING', true)
        const response = await fetchWarehousesTransfers(query)

        commit('SET_WAREHOUSES_TRANSFERS_ITEMS_LIST', response.data)
        if (response.table_specifications) commit('SET_WAREHOUSES_TRANSFERS_TABLE_SPECIFICATIONS', response.table_specifications)
      } catch (e) {
        throw e
      } finally {
        commit('SET_WAREHOUSES_TRANSFERS_LOADING', false)
      }
    },
    async getMetaRequest ({ commit, state }, query) {
      try {
        const response = await fetchWarehousesTransfers(query)
        if (response.meta && response.meta.last_page) commit('SET_WAREHOUSES_TRANSFERS_MAX_PAGES', response.meta.last_page)
        if (response.meta) commit('SET_WAREHOUSES_TRANSFERS_META', response.meta)
      } catch (e) {
        throw e
      }
    },
    async loadTableSpecifications ({ commit }, query) {
      try {
        const response = await fetchWarehousesTransfers(query)

        if (response.data?.table_specifications) commit('SET_WAREHOUSES_TRANSFERS_TABLE_SPECIFICATIONS', response.data.table_specifications)
        return Promise.resolve()
      } catch (e) {
        throw e
      }
    }
  }
}
