import {
  fetchPurchaseOrderLine,
  getAllPurchaseOrders,
  settingsPurchaseOrders
} from '@/services/orders/PurchaseOrdersService'
import { lookupPurchaseOrdersValues } from '@/services/LookupService'
import { getAllShippingCarriers } from '@/services/orders/ShippingCarriersService'
import moment from 'moment-timezone'
import Vue from 'vue'
import wasArrayChanged from '@/support/wasArrayChanged'
import { preparePOLines, rowPriceAfterDiscount } from '@/support/productItems'
import { availableItems } from '@/support/purchaseOrderHelpers'
import { DRAWER_MODE } from '@/assets/constants'
import { getFilter, getTableFilters } from '@/support/apiRequestHelpers'
import exactMath from 'exact-math'
import { fetchUnlinkedAmazonInboundShipments } from '@/services/AmazonIntegrationInboundService'

const setPOLinesFilter = (value = true) => getFilter('is_product', value)

const MODE_VALUES = DRAWER_MODE.PO

export const PurchaseOrdersModule = {
  namespaced: true,
  state: () => ({
    itemsList: [],
    tableSpecifications: {},
    maxPages: 0,
    meta: {},
    loading: false,

    itemByIdLoading: false,
    selectedItem: null,

    carriers: [], // aka shipping methods
    carriersLoading: false,

    editedOrder: {
      date: null,
      otherDate: null,
      supplier: null,
      store: null,
      purchaseOrderNumber: '',
      currency: null,
      paymentTerm: null,
      incoterm: null,
      purchaseOrderFormat: null,
      destination: null,
      shippingMethod: null,
      requestedShippingMethod: null,
      eta: '',
      supplierNotes: '',
      internalNotes: '',
      unmatchedShipment: '',
      trackingNumber: '',
      tags: []
    },
    ignoreItemResetOnLaunch: false,

    autogeneratePurchaseOrderNumber: true,

    orderProducts: [],
    drawerMode: null, // createPurchaseOrder, editPurchaseOrder and viewPurchaseOrder, initial value must be null!

    addressFields: [
      'name',
      'company',
      'phone',
      'address1',
      'address2',
      'address3',
      'city',
      'province',
      'province_code',
      'zip',
      'country',
      'country_code'
    ],
    itemFields: [
      'discount',
      'name',
      'nominal_code',
      'eta',
      'price',
      'quantity',
      'tax_rate_id'
    ],

    fetchedSelectValues: false,

    settings: [],
    settingsLoading: false,

    inventoryWarehouse: {
      showUserWarehouseDrawer: false,
      initialSupplierWarehouse: null,
      showSupplierWarehouseModal: false
    },
    selectedIsTaxIncluded: false,
    selectedOrderTaxRate: {
      tax_rate_id: null,
      tax_rate: null,
      amount: 0
    },
    selectedTaxes: [],

    initialValuesAreSet: false,

    // Indicates if a purchase order is being created from forecasting
    forecastingPurchaseOrder: null,

    unmatchedShipments: [],
    unmatchedShipmentsLoading: false,

    globalSettings: null,
    skuCurrencies: [],
    isInitProduct: false,
    linesLoading: false,
    swapItem: null,
    showSwapModal: false,
    inventoryModalCoordinates: { x: null, y: null },
    loadingNonProductLines: false,
    editedProducts: new Map(),
    initialEditedProducts: new Map(),
    editedOrderSummaries: {
      discount: 0,
      subtotal: 0,
      taxes: new Map()
    },
    deletedTaxes: new Map()
  }),
  getters: {
    getShowBarcode: state => {
      if (!state.globalSettings) return false
      const barcodeObj = state.globalSettings.find(i => i.key === 'sales_credit_show_barcode_in_frontend')
      if (!barcodeObj) return false
      return barcodeObj.value
    },
    getForecastingPurchaseOrder: state => state.forecastingPurchaseOrder,
    getItemsList: state => state.itemsList,
    getTableSpecifications: state => state.tableSpecifications,
    getMaxPages: state => state.maxPages,
    getMeta: state => state.meta,

    getItemByIdLoading: state => state.itemByIdLoading,
    getSelectedItem: state => state.selectedItem,
    getSelectedItemIndex: state => {
      if (!state.selectedItem) return 0
      const index = state.itemsList.findIndex(i => i.id === state.selectedItem.id)
      return index
    },

    getAutogeneratePurchaseOrderNumber: state => state.autogeneratePurchaseOrderNumber,

    getCarriers: state => state.carriers,
    getCarriersLoading: state => state.carriersLoading,
    getShippingMethods: (state, getters) => {
      let methodsArray = []
      getters.getCarriers.forEach((carrier, index) => {
        carrier.shipping_methods.forEach(method => {
          methodsArray.push({
            ...method,
            id: method.id,
            name: method.name,
            fullName: `${carrier.name} ${method.name}`
          })
        })
      })
      return methodsArray
    },

    getEditedOrder: state => state.editedOrder,
    getEditedOrderDate: state => state.editedOrder.date,
    getSelectedOtherDate: state => state.editedOrder.otherDate,
    getSelectedSupplier: state => state.editedOrder.supplier,
    getSelectedStore: state => state.editedOrder.store,
    getSelectedOrderNumber: state => state.editedOrder.purchaseOrderNumber,
    getSelectedCurrency: state => state.editedOrder.currency,
    getSelectedPOFormat: state => state.editedOrder.purchaseOrderFormat,
    getSelectedDestination: state => state.editedOrder.destination,
    getIsDropshipPurchaseOrder: state => {
      // destination could be warehouse or customer (in case of dropshipping)
      // if there's sales_order_id then it's dropshipping, hence destination should be the customer
      // customers are not editable, unlike warehouses
      if (state.selectedItem && (state.selectedItem.sales_order_id || state.editedOrder.sales_order_id === 0)) {
        return true
      } else {
        return false
      }
    },
    getSelectedTrackingNumber: state => state.editedOrder.trackingNumber,
    getSelectedUnmatchedShipment: state => state.editedOrder.unmatchedShipment,
    getSelectedShippingMethod: state => state.editedOrder.shippingMethod,
    getSelectedRequestedShippingMethod: state => state.editedOrder.requestedShippingMethod,
    getSelectedEta: state => state.editedOrder.eta,
    getSupplierNotes: state => state.editedOrder.supplierNotes,
    getInternalNotes: state => state.editedOrder.internalNotes,

    getPOFormats: () => [
      { name: 'Email PDF Attachment', value: 'email_pdf_attachment' }
    ],

    getSupplierChanged: state => {
      const initialValue = (state.selectedItem && state.selectedItem.supplier_id) || '-'
      const localValue = (state.editedOrder && state.editedOrder.supplier && state.editedOrder.supplier.id) || '-'
      return localValue !== initialValue
    },

    getStoreChanged: state => {
      const initialValue = (state.selectedItem && state.selectedItem.store_id) || '-'
      const localValue = (state.editedOrder && state.editedOrder.store && state.editedOrder.store.id) || '-'
      return localValue !== initialValue
    },

    getCurrencyChanged: state => {
      if (!state.selectedItem) return false
      const initialValue = state.selectedItem.currency_code || '-'
      const localValue = (state.editedOrder && state.editedOrder.currency && state.editedOrder.currency.code) || '-'
      return localValue !== initialValue
    },

    getDateChanged: (state, getters) => {
      if (!state.selectedItem || !getters.getEditedOrderDate) return false
      const initialValue = (moment(state.selectedItem.purchase_order_date).format()) || '-'
      const localValue = moment(getters.getEditedOrderDate).format() || '-'
      return initialValue !== localValue
    },

    getOrderNumberChanged: (state, getters) => {
      if (!state.selectedItem || !getters.getSelectedOrderNumber) return false
      const initialValue = state.selectedItem.po_number || '-'
      const localValue = getters.getSelectedOrderNumber || '-'
      return initialValue !== localValue
    },

    getOtherDateChanged: (state, getters) => {
      if (!state.selectedItem || !getters.getSelectedOtherDate) return false
      const initialValue = (moment(state.selectedItem.other_date).format()) || '-'
      const localValue = moment(getters.getSelectedOtherDate).format() || '-'
      return initialValue !== localValue
    },

    getPOFormatChanged: state => {
      if (!state.selectedItem) return false
      const initialValue = state.selectedItem.submission_format || '-'
      const localValue = (state.editedOrder && state.editedOrder.purchaseOrderFormat && state.editedOrder.purchaseOrderFormat.value) || '-'
      return localValue !== initialValue
    },

    getDestinationChanged: state => {
      if (!state.selectedItem) return false
      // only dropship orders have sales_order_id, and destination is not editable for those
      if (state.selectedItem && state.selectedItem.sales_order_id) return false

      const initialValue = state.selectedItem.destination_warehouse_id || '-'
      const localValue = (state.editedOrder && state.editedOrder.destination && state.editedOrder.destination.id) || '-'
      return localValue !== initialValue
    },

    getShippingMethodChanged: state => {
      const initialValue = (state.selectedItem && state.selectedItem.requested_shipping_method_id) || '-'
      const localValue = (state.editedOrder && state.editedOrder.shippingMethod && state.editedOrder.shippingMethod.id) || '-'
      return localValue !== initialValue
    },
    getRequestedShippingMethodChanged: state => {
      const initialValue = (state.selectedItem && state.selectedItem.requested_shipping_method) || '-'
      const localValue = (state.editedOrder && state.editedOrder.requestedShippingMethod) || '-'
      return localValue !== initialValue
    },
    getTrackingNumberChanged: state => {
      const initialValue = (state.selectedItem && state.selectedItem.tracking_number) || '-'
      const localValue = (state.editedOrder && state.editedOrder.trackingNumber) || '-'
      return localValue !== initialValue
    },

    getEtaChanged: state => {
      const initialValue = (state.selectedItem && state.selectedItem.estimated_delivery_date) || '-'
      const localValue = (state.editedOrder && state.editedOrder.eta) || '-'
      return localValue !== initialValue
    },

    getSupplierNotesChanged: state => {
      const initialValue = (state.selectedItem && state.selectedItem.supplier_notes) || '-'
      const localValue = (state.editedOrder && state.editedOrder.supplierNotes) || '-'
      return localValue !== initialValue
    },

    getInternalNotesChanged: state => {
      const initialValue = (state.selectedItem && state.selectedItem.internal_notes) || '-'
      const localValue = (state.editedOrder && state.editedOrder.internalNotes) || '-'
      return localValue !== initialValue
    },

    getOrderProducts: state => state.orderProducts,
    getOrderProductsChanged: (state, getters) => {
      return getters.getEditedProducts.size !== 0
    },
    getSomethingChanged (state, getters) {
      return getters.getSupplierChanged || getters.getStoreChanged || getters.getDateChanged ||
        getters.getCurrencyChanged || getters.getPOFormatChanged || getters.getDestinationChanged ||
        getters.getShippingMethodChanged || getters.getEtaChanged || getters.getSupplierNotesChanged ||
        getters.getInternalNotesChanged || getters.getOrderProductsChanged || getters.getTrackingNumberChanged ||
        getters.getRequestedShippingMethodChanged || getters.getOtherDateChanged || getters.getOrderNumberChanged ||
        getters.getIsTaxIncludedChanged || getters.getOrderTaxesChanged || getters.getIsTagsChanged
    },

    getAddressFields: state => state.addressFields,
    getItemFields: state => state.itemFields,

    getDrawerMode: state => state.drawerMode,
    getFetchedSelectValues: state => state.fetchedSelectValues,

    getSettingsList: state => state.settings,
    getSettingsLoading: state => state.settingsLoading,
    getInventoryWarehouse: state => state.inventoryWarehouse,

    getInitialValuesAreSet: state => state.initialValuesAreSet,

    getUnmatchedShipments: state => state.unmatchedShipments,
    getUnmatchedShipmentsLoading: state => state.unmatchedShipmentsLoading,

    getEditingIsBlockedForFbaDestination: state => {
      return state.drawerMode && state.drawerMode !== MODE_VALUES.CREATE && state.selectedItem && state.selectedItem.shipment_id
    },

    getSkuCurrencies: state => state.skuCurrencies,
    getLoading: state => state.loading,
    getIgnoreItemResetOnLaunch: state => state.ignoreItemResetOnLaunch,
    getSelectedIsTaxIncluded: state => state.selectedIsTaxIncluded,
    getIsTaxIncludedChanged: (state, getters) => {
      if (!state.selectedItem) return false
      return state.selectedItem?.is_tax_included !== getters.getSelectedIsTaxIncluded
    },
    getSelectedTaxes: state => state.selectedTaxes,
    getOrderTaxesChanged: state => {
      if (!state.selectedItem) return false

      return state.selectedTaxes.length
        ? state.selectedItem.tax_rate_id !== state.selectedTaxes[0].tax_rate_id
        : state.selectedItem.tax_rate_id !== null
    },
    getSelectedOrderTaxRate: state => state.selectedOrderTaxRate,
    getIsSetPurchaseOrderTaxRate: (_, getters) => {
      if (!getters.getSelectedItem || !getters.getSelectedItem.hasOwnProperty('tax_rate_id') || !getters.getSelectedItem.hasOwnProperty('tax_rate_id')) return false
      return getters.getSelectedItem?.tax_rate_id !== null || getters.getSelectedOrderTaxRate?.tax_rate_id
    },
    getDefaultSkuCurrency: (_, getters) => {
      const defaultCurrency = getters.getSkuCurrencies.filter(item => item.is_default)

      return defaultCurrency.length !== 0 ? defaultCurrency[0] : null
    },
    getSelectedItemAccounting: (_, getters) => getters.getSelectedItem.hasOwnProperty('accounting') ? getters.getSelectedItem.accounting : null,
    isCreateDrawerMode: (_, getters) => getters.getDrawerMode === MODE_VALUES.CREATE,
    isEditDrawerMode: (_, getters) => getters.getDrawerMode === MODE_VALUES.EDIT,
    isViewDrawerMode: (_, getters) => getters.getDrawerMode === MODE_VALUES.VIEW,
    getIsTagsChanged: state => {
      const localItems = state.editedOrder.tags || []
      const initialItems = state.selectedItem?.tags || []

      return wasArrayChanged(localItems, initialItems)
    },
    getEditedOrderTags: state => state.editedOrder.tags,
    isInitProduct: state => state.isInitProduct,
    isLinesLoading: state => state.linesLoading,
    swapItem: state => state.swapItem,
    showSwapModal: state => state.showSwapModal,
    getInventoryModalCoordinates: state => state.inventoryModalCoordinates,
    orderItems: (_, getters, rootState, rootGetters) => {
      const defaultNominalCode = rootGetters['NominalCodes/getDefaultPurchaseNominalCode']
      if (!getters.getSelectedItem?.items) return []

      return getters.getSelectedItem.items.map((i) => {
        const discountRate = !isNaN(i?.discount_rate) ? parseFloat(i.discount_rate) : 0
        const product = {
          ...i,
          discount: i.discount || exactMath.mul(discountRate, 100) || null,
          non_product_row: !i.sku || !i.sku.id,
          purchase_order_line_id: i.id,
          available_items_count: availableItems(i, getters.getIsDropshipPurchaseOrder, getters.getSelectedDestination)
        }
        if (!product.nominal_code) product.nominal_code = defaultNominalCode
        return product
      })
    },
    getProducts: (_, getters) => getters.orderItems.filter(item => !item.non_product_row),
    getNonProducts: (_, getters) => getters.orderItems.filter(item => item.non_product_row),
    getNonProductsLoading: state => state.loadingNonProductLines,
    getEditedProducts: state => state.editedProducts,
    getInitialEditedProducts: state => state.initialEditedProducts,
    getEditedOrderSubtotal: state => state.editedOrderSummaries.subtotal || 0,
    getEditedOrderDiscount: state => state.editedOrderSummaries.discount || 0,
    getEditedOrderTaxes: state => Array.from(state.editedOrderSummaries.taxes.values()),
    getEditedOrderTotal: (_, getters) => {
      return exactMath.add(getters.getEditedOrderSubtotal, getters.getEditedOrderTaxesAmount, getters.getNonProductAmount)
    },
    getEditedOrderTaxesAmount: (_, getters) => {
      if (getters.getSelectedIsTaxIncluded) return 0

      const taxes = getters.getSelectedOrderTaxRate.tax_rate_id
        ? [getters.getSelectedOrderTaxRate]
        : getters.getEditedOrderTaxes

      return taxes.reduce((acc, tax) => {
        acc = exactMath.add(parseFloat(tax.amount_in_tenant_currency ?? 0), acc)
        return acc
      }, 0)
    },
    getNonProductAmount: (_, getters) => {
      const editedNonProducts = Array.from(getters.getEditedProducts.values()).filter(p => p.non_product_row)
      const editedNonProductIds = editedNonProducts.map(p => p.id)

      if (editedNonProducts.length && editedNonProductIds.length) {
        return [
          ...getters.getNonProducts.filter(p => !editedNonProductIds.includes(p.id)),
          ...editedNonProducts
        ].reduce((acc, p) => {
          if (!p?.is_deleted) acc = exactMath.add(acc, rowPriceAfterDiscount(p))

          return acc
        }, 0)
      } else {
        return getters.getNonProducts.reduce((acc, p) => {
          if (!p?.is_deleted) acc = exactMath.add(acc, rowPriceAfterDiscount(p))
          return acc
        }, 0)
      }
    },
    getDeletedTaxes: state => state.deletedTaxes
  },
  mutations: {
    SET_FORECASTING_PURCHASE_ORDER (state, value) {
      state.forecastingPurchaseOrder = value
    },
    SET_ITEMS_LIST (state, items) {
      state.itemsList = items
    },
    SET_TABLE_SPECIFICATIONS (state, specs) {
      state.tableSpecifications = specs
    },
    SET_MAX_PAGES (state, pages) {
      state.maxPages = pages
    },
    SET_META (state, meta) {
      state.meta = meta
    },
    SET_ITEM_BY_ID_LOADING (state, value) {
      state.itemByIdLoading = value
    },
    SET_SELECTED_ITEM (state, value) {
      state.selectedItem = value
    },
    SET_CARRIERS (state, value) {
      state.carriers = value
    },
    SET_CARRIERS_LOADING (state, value) {
      state.carriersLoading = value
    },
    SET_EDITED_ORDER (state, value) {
      state.editedOrder = value
    },
    SET_EDITED_ORDER_FIELD (state, { field, value }) {
      Vue.prototype.$set(state.editedOrder, field, value)
    },
    SET_ORDER_PRODUCTS (state, value) {
      state.orderProducts = value
    },
    SET_DRAWER_MODE (state, value) {
      state.drawerMode = value
    },
    SET_FETCHED_SELECT_VALUES (state, value) {
      state.fetchedSelectValues = value
    },
    SET_SETTINGS (state, value) {
      state.settings = value
    },
    SET_SETTINGS_LOADING (state, value) {
      state.settingsLoading = value
    },
    SET_INVENTORY_WAREHOUSE (state, { value, field }) {
      Vue.prototype.$set(state.inventoryWarehouse, field, value)
    },
    SET_INITIAL_VALUES_ARE_SET (state, value) {
      state.initialValuesAreSet = value
    },
    SET_AUTOGENERATE_PURCHASE_ORDER_NUMBER (state, value) {
      state.autogeneratePurchaseOrderNumber = value
    },
    SET_UNMATCHED_SHIPMENTS (state, value) {
      state.unmatchedShipments = value
    },
    SET_UNMATCHED_SHIPMENTS_LOADING (state, value) {
      state.unmatchedShipmentsLoading = value
    },
    SET_SELECTED_UNMATCHED_SHIPMENT (state, value) {
      state.editedOrder.unmatchedShipment = value
    },
    SET_GLOBAL_SETTINGS (state, value) {
      state.globalSettings = value
    },
    SET_SKU_CURRENCIES (state, value) {
      state.skuCurrencies = value
    },
    SET_LOADING (state, value) {
      state.loading = value
    },
    SET_IGNORE_ITEM_RESET (state, value) {
      state.ignoreItemResetOnLaunch = value
    },
    SET_SELECTED_IS_TAX_INCLUDED (state, value) {
      state.selectedIsTaxIncluded = value
    },
    SET_SELECTED_TAXES (state, value) {
      if (value === undefined) return
      state.selectedTaxes = [...new Map(value.map(item => [item.id, item])).values()]
    },
    SET_SELECTED_ORDER_TAX_RATE (state, value) {
      state.selectedOrderTaxRate = value
    },
    REMOVE_TAX_BY_INDEX (state, index) {
      state.selectedTaxes.splice(index, 1)
    },
    SET_SELECTED_ITEM_FIELD (state, { field, value }) {
      Vue.prototype.$set(state.selectedItem, field, value)
    },
    SET_IS_INIT_PRODUCT (state, value) {
      state.isInitProduct = value
    },
    SET_LINES_LOADING (state, value) {
      state.linesLoading = value
    },
    SET_SWAP_ITEM (state, value) {
      state.swapItem = value
    },
    SET_SHOW_SWAP_MODAL (state, value) {
      state.showSwapModal = value
    },
    SET_INVENTORY_COORDINATES (state, value) {
      state.inventoryModalCoordinates = value
    },
    SET_NON_PRODUCT_LOADING (state, value) {
      state.loadingNonProductLines = value
    },
    SET_EDITED_PRODUCTS (state, value) {
      state.editedProducts = value
    },
    SET_INITIAL_EDITED_PRODUCTS (state, value) {
      state.initialEditedProducts = value
    },
    SET_EDITED_ORDER_SUMMARIES_FIELD (state, { field, value }) {
      Vue.prototype.$set(state.editedOrderSummaries, field, value)
    },
    SET_EDITED_ORDER_SUMMARIES (state, value) {
      state.editedOrderSummaries = value
    },
    SET_DELETED_TAXES (state, value) {
      state.deletedTaxes = value
    }
  },
  actions: {
    async getItemsRequest ({ commit, getters }, query) {
      try {
        commit('SET_LOADING', true)
        const response = await getAllPurchaseOrders(query)
        commit('SET_ITEMS_LIST', response.data)
        if (response.table_specifications) commit('SET_TABLE_SPECIFICATIONS', response.table_specifications)
      } catch (e) {
        throw e
      } finally {
        commit('SET_LOADING', false)
      }
    },
    async getMetaRequest ({ commit, getters }, query) {
      try {
        const response = await getAllPurchaseOrders(query)
        if (response.meta && response.meta.last_page) commit('SET_MAX_PAGES', response.meta.last_page)
        if (response.meta) commit('SET_META', response.meta)
      } catch (e) {
        throw e
      }
    },
    async loadTableSpecifications ({ commit, getters }, query) {
      try {
        const response = await getAllPurchaseOrders(query)
        if (response.data?.table_specifications) commit('SET_TABLE_SPECIFICATIONS', response.data.table_specifications)
        return Promise.resolve()
      } catch (e) {
        throw e
      }
    },
    async fetchLookupValues ({ commit }, { value, header, page = 0, limit = 10 }) {
      try {
        return await lookupPurchaseOrdersValues({ value, header, page, limit })
      } catch (e) {
        throw e
      }
    },
    async fetchAllCarriers ({ commit, getters }) {
      try {
        commit('SET_CARRIERS_LOADING', true)
        const response = await getAllShippingCarriers()
        commit('SET_CARRIERS', response.data)
      } catch (e) {
        throw e
      } finally {
        commit('SET_CARRIERS_LOADING', false)
      }
    },
    async fetchSettings ({ commit, getters }) {
      try {
        commit('SET_SETTINGS_LOADING', true)
        const response = await settingsPurchaseOrders()
        commit('SET_SETTINGS', response.data)
      } catch (e) {
        throw e
      } finally {
        commit('SET_SETTINGS_LOADING', false)
      }
    },
    async fetchUnmatchedShipments ({ commit, getters }, { instanceId, limit }) {
      try {
        commit('SET_UNMATCHED_SHIPMENTS_LOADING', true)
        const { data } = await fetchUnlinkedAmazonInboundShipments(instanceId, { limit: limit })

        commit('SET_UNMATCHED_SHIPMENTS', data)
      } catch (e) {
        return Promise.reject(e)
      } finally {
        commit('SET_UNMATCHED_SHIPMENTS_LOADING', false)
      }
    },
    fetchPurchaseOrderLines ({ commit, getters }, { id, query, filterBy = -1, filterSet = [] }) {
      if (getters.isLinesLoading) return
      // unlimited:
      //    get all lines for edit mode                                                      [filter_by -> -1]
      // limited:
      //    initial -> get all non_product and limited is_product lines                      [filter_by -> 0]
      //    pagination -> get limited is_product lines and to mix with is_not_product lines  [filter_by -> 1]
      const productFilters = filterBy === -1 ? {} : getTableFilters([setPOLinesFilter(), ...filterSet])
      const nonProductFilters = getTableFilters([setPOLinesFilter(false), ...filterSet])

      // productFilters.excluded = '[]'
      // nonProductFilters.excluded = '[]'

      commit('SET_LINES_LOADING', true)
      let actions
      if (filterBy !== 0) {
        actions = [ fetchPurchaseOrderLine(id, { ...query, ...productFilters }) ]
      } else {
        actions = [
          fetchPurchaseOrderLine(id, { ...query, ...productFilters }),
          fetchPurchaseOrderLine(id, { ...query, limit: -1, ...nonProductFilters })
        ]
        commit('SET_NON_PRODUCT_LOADING', true)
      }

      Promise.allSettled(actions)
        .then(response => {
          const data = response.reduce((acc, r) => {
            if (r?.value?.data) {
              const lines = r.value.data.length > query.limit && filterBy !== -1
                ? r.value.data.filter(line => line.sku.id === null)
                : r.value.data
              acc = [...acc, ...lines]
            }
            return acc
          }, [])
            .sort((a, b) => a.id < b.id ? 1 : -1)
          if (!data) return
          let products = preparePOLines(data.sort((a, b) => a.id < b.id ? 1 : -1))
          if (filterBy === 1) products = [...products, ...getters.getNonProducts]
          commit('SET_ORDER_PRODUCTS', products)
          commit('SET_SELECTED_ITEM_FIELD', { field: 'items', value: products })
        })
        .catch(e => {
          throw e
        })
        .finally(() => {
          commit('SET_LINES_LOADING', false)
          commit('SET_NON_PRODUCT_LOADING', false)
        })
    }
  }
}
