import Vue from 'vue'
import {
  getPurchaseInvoices,
  fetchPurchaseInvoicePayment,
  createPurchaseInvoicePayment,
  updatePurchaseInvoicePayment,
  deletePurchaseInvoicePayment
} from '@/services/orders/PurchaseInvoiceService'
import { isEqual, cloneDeep } from 'lodash'
import exactMath from 'exact-math'
import { getPaymentObject } from '@/support/paymentsHelpers'

export const PurchaseOrderInvoiceModule = {
  namespaced: true,
  state: {
    editedInvoice: {
      invoiceNumber: '',
      date: '',
      items: []
    },
    initialInvoice: {
      invoiceNumber: '',
      date: '',
      items: []
    },
    selectedItem: null,
    drawerMode: null, // payments, view, create
    purchaseInvoiceDrawer: {
      show: false,
      hideNavigator: false,
      initialItemId: false
    },

    receiveItems: [],

    invoicesData: {
      items: [],
      loading: false,
      isInitialised: false
    },
    itemFields: ['discount', 'price', 'quantity', 'tax', 'name'],
    itemsForNewInvoice: [],
    purchaseOrder: {
      item: null,
      format: null,
      supplier: null,
      currency: null,
      barcode: null,
      selectedIsTaxIncluded: false,
      selectedOrderTaxRate: {
        tax_rate_id: null,
        tax_rate: null
      },
      selectedTaxes: []
    },
    parentUid: null,
    paymentModal: {
      show: false,
      deleting: false,
      payment: null,
      isRefund: false,
      loading: false
    },
    payments: [],
    selectedInvoiceTotal: 0
  },
  getters: {
    getEditedInvoice: state => state.editedInvoice,
    getInitialInvoice: state => state.initialInvoice,
    getReceiveItems: state => state.receiveItems,
    getSelectedItem: state => state.selectedItem,
    getDrawerMode: state => state.drawerMode,
    getPurchaseInvoiceDrawer: state => state.purchaseInvoiceDrawer,

    getInvoiceNumberChanged: state => {
      const initialValue = state.initialInvoice.invoiceNumber || '-'
      const localValue = state.editedInvoice.invoiceNumber || '-'
      return localValue !== initialValue
    },
    getDateChanged: state => {
      const initialValue = state.initialInvoice.date || '-'
      const localValue = state.editedInvoice.date || '-'
      return localValue !== initialValue
    },
    getItemsChanged: state => {
      const initialItems = state.initialInvoice.items
      const localItems = state.editedInvoice.items
      // items added or removed
      if (localItems.length !== initialItems.length) return true
      // changes inside of items
      let initialObj = {}
      let localObj = {}

      for (let i = 0; i < initialItems.length; i++) {
        const initialItem = initialItems[i]
        const localItem = localItems[i]

        state.itemFields.forEach(field => {
          initialObj[field] = initialItem[field] || ''
          localObj[field] = localItem[field] || ''
        })

        if (!isEqual(initialObj, localObj)) {
          return true
        }
      }
      return false
    },
    getSomethingChanged: (state, getters) => {
      return getters.getInvoiceNumberChanged || getters.getDateChanged || getters.getItemsChanged
    },

    getInvoicesData: state => state.invoicesData,
    getItemsForNewInvoice: state => state.itemsForNewInvoice,

    getItemInvoicedQuantity: state => purchaseOrderItemId => {
      // filter invoices by given purchaseOrderItem
      let lines = []
      state.invoicesData.items.forEach(i => {
        const invoiceLines = i.purchase_invoice_lines.filter(l => l.purchase_order_line_id === purchaseOrderItemId)
        lines = [...lines, ...invoiceLines]
      })
      if (!lines.length) return 0
      // compute quantity_invoiced
      const quantity = lines.map(i => i.quantity_invoiced).reduce((acc, value) => Number(acc) + Number(value))
      return quantity
    },
    getPaymentModal: state => state.paymentModal,
    getPOItem: state => state.purchaseOrder.item,
    getPOFormat: state => state.purchaseOrder.format,
    getPOSupplier: state => state.purchaseOrder.supplier,
    getPOCurrency: state => state.purchaseOrder.currency,
    getPOBarcode: state => state.purchaseOrder.barcode,
    getParentUid: state => state.parentUid,
    getSelectedIsTaxIncluded: state => state.purchaseOrder.selectedIsTaxIncluded,
    getSelectedOrderTaxRate: state => state.purchaseOrder.selectedOrderTaxRate,
    getSelectedTaxes: state => state.purchaseOrder.selectedTaxes,
    getSelectedInvoicePayments: state => state.payments,
    isViewDrawerMode: (_, getters) => getters.getDrawerMode === 'view',
    totalPayment: (_, getters) => {
      if (getters.getSelectedInvoicePayments.length === 0) return 0
      return getters.getSelectedInvoicePayments.reduce((acc, payment) => acc += parseFloat(payment.amount), 0)
    },
    totalDue: (_, getters) => {
      if (!getters.getSelectedInvoiceTotal) return 0
      return exactMath.round(Number(getters.getSelectedInvoiceTotal) - getters.totalPayment, -2)
    },
    getSelectedInvoiceTotal: state => state.selectedInvoiceTotal
  },
  mutations: {
    SET_EDITED_INVOICE (state, value) {
      state.editedInvoice = value
    },
    SET_INITIAL_INVOICE (state, value) {
      state.initialInvoice = value
    },
    SET_EDITED_INVOICE_FIELD (state, { field, value }) {
      Vue.prototype.$set(state.editedInvoice, field, value)
    },
    SET_RECEIVE_ITEMS (state, value) {
      state.receiveItems = value
    },
    SET_SELECTED_ITEM (state, value) {
      state.selectedItem = value
    },
    SET_DRAWER_MODE (state, value) {
      state.drawerMode = value
    },
    SET_ITEMS_FOR_NEW_INVOICE (state, value) {
      state.itemsForNewInvoice = value
    },
    SET_PURCHASE_INVOICE_DRAWER_FIELD (state, { field, value }) {
      Vue.prototype.$set(state.purchaseInvoiceDrawer, field, value)
    },
    SET_INVOICES_DATA_FIELD (state, { field, value }) {
      Vue.prototype.$set(state.invoicesData, field, value)
    },
    SET_PURCHASE_ORDER_FIELDS (state, { field, value }) {
      Vue.prototype.$set(state.purchaseOrder, field, value)
    },
    SET_INVOICE_PURCHASE_ORDER_FIELD (state, { field, value }) {
      Vue.prototype.$set(state.purchaseOrder, field, value)
    },
    SET_PARENT_UID (state, value) {
      state.parentUid = value
    },
    SET_PAYMENT_MODAL_FIELD (state, { field, value }) {
      Vue.prototype.$set(state.paymentModal, field, value)
    },
    SET_INVOICE_PAYMENTS (state, value) {
      state.payments = value
    },
    SET_SELECTED_INVOICE_TOTAL (state, value) {
      state.selectedInvoiceTotal = value
    }
  },
  actions: {
    async fetchPurchaseInvoices ({ commit, getters }, purchaseId) {
      try {
        commit('SET_INVOICES_DATA_FIELD', { field: 'loading', value: true })
        const { data } = await getPurchaseInvoices(purchaseId)
        commit('SET_INVOICES_DATA_FIELD', { field: 'items', value: data })

        // following flag is required for cases when invoice drawer is been loaded from url,
        // we must start it after purchase order drawer is loaded (invoice drawer is always on top of po drawer)
        if (!getters.getInvoicesData.isInitialised) commit('SET_INVOICES_DATA_FIELD', { field: 'isInitialised', value: true })
        return Promise.resolve()
      } catch (e) {
        return Promise.reject(e)
      } finally {
        commit('SET_INVOICES_DATA_FIELD', { field: 'loading', value: false })
      }
    },
    async fetchPurchaseInvoicePayments ({ commit }, invoiceId) {
      try {
        const { data } = await fetchPurchaseInvoicePayment(invoiceId)
        commit('SET_INVOICE_PAYMENTS', data)
        return Promise.resolve()
      } catch (e) {
        return Promise.reject(e)
      }
    },
    async updatePurchaseInvoicePayment ({ commit, getters }, payload) {
      try {
        const { data } = await updatePurchaseInvoicePayment(getters.getSelectedItem.id, payload.id, payload.payment)
        const payments = cloneDeep(getters.getSelectedInvoicePayments)
          .map(payment => {
            if (payment.id === data.id) {
              payment.amount = data.amount
              payment.payment_date = data.payment_date
              payment.payment_type = data.payment_type
              payment.payment_type_id = data.payment_type_id
              payment.external_reference = data.external_reference
            }
            return payment
          })
        commit('SET_INVOICE_PAYMENTS', payments)
        return Promise.resolve()
      } catch (e) {
        return Promise.reject(e)
      }
    },
    async addPurchaseInvoicePayment ({ commit, getters, rootState }, payload) {
      try {
        const { data } = await createPurchaseInvoicePayment(getters.getSelectedItem.id, payload)
        let payment = getPaymentObject(data)
        const paymentType = rootState.PaymentTypesModule.types.find(type => type.id === data.payment_type_id)
        if (paymentType && paymentType.name) payment.payment_type.name = paymentType.name
        let payments = cloneDeep(getters.getSelectedInvoicePayments)
        payments.unshift(payment)
        commit('SET_INVOICE_PAYMENTS', payments)
        commit('SET_PAYMENT_MODAL_FIELD', { field: 'payment', value: null })
        return Promise.resolve()
      } catch (e) {
        return Promise.reject(e)
      }
    },
    async deletePurchaseInvoicePayment ({ commit, getters }, paymentId) {
      try {
        const data = await deletePurchaseInvoicePayment(getters.getSelectedItem.id, paymentId)
        let payments = getters.getSelectedInvoicePayments.filter(p => p.id !== paymentId)
        commit('SET_INVOICE_PAYMENTS', payments)
        return Promise.resolve(data)
      } catch (e) {
        return Promise.reject(e)
      }
    }
  }
}
