import { getProductByID, getProductSuppliers } from '@/services/ProductsService'
import { getProductInventoryMovements } from '@/services/InventoryMovementService'
import Vue from 'vue'
import { toArray } from 'lodash'
import { settingsBlemishedSkuPatternSettings } from '@/services/ProductBlemishedService'

export const ProductsModule = {
  namespaced: true,
  state: () => ({
    selectedProduct: null,
    productDetailsMenu: false,
    editProductNotificationId: Math.ceil(Math.random() * 50000),
    productProcessing: false,
    productByIdLoading: false,
    isProductEditing: false,
    inventoryMovement: {
      itemsList: [],
      tableSpecifications: {},
      maxPages: 0,
      meta: {},
      loading: false
    },
    dashboardData: {
      loading: false,
      analytics: [],
      chart: []
    },
    suppliers: [],
    settings: [],
    settingsLoading: false
  }),
  getters: {
    getSelectedProduct: state => state.selectedProduct,
    getEditNotificationId: state => state.editProductNotificationId,
    getProductByIdLoading: state => state.productByIdLoading,
    getProductProcessing: state => state.productProcessing,
    getProductEditing: state => state.isProductEditing,
    getInventoryMovementItemsList: state => state.inventoryMovement.itemsList,
    getInventoryMovementTableSpecifications: state => state.inventoryMovement.tableSpecifications,
    getInventoryMovementMaxPages: state => state.inventoryMovement.maxPages,
    getInventoryMovementMeta: state => state.inventoryMovement.meta,
    getInventoryMovementLoading: state => state.inventoryMovement.loading,
    getDashboardAnalytics: state => state.dashboardData.analytics,
    getDashboardLoading: state => state.dashboardData.loading,
    getDashboardChartData: state => state.dashboardData.chart,
    getTags: (_, getters) => getters.getSelectedProduct?.tags || [],
    getSuppliers: state => state.suppliers,
    isKitType: (_, getters) => getters.getSelectedProduct?.type === 'kit',
    isMatrixType: (_, getters) => getters.getSelectedProduct?.type === 'matrix',
    isBundleType: (_, getters) => getters.getSelectedProduct?.type === 'bundle',
    isStandardType: (_, getters) => getters.getSelectedProduct?.type === 'standard',
    getBlemishedSkuPattern: state => {
      if (!state.settings || state.settings.length === 0) return null
      return state.settings.find(setting => setting.key === 'blemished_sku_pattern') || null
    },
    getIsBlemishedSkuPattern: state => {
      if (!state.settings || state.settings.length === 0) return null
      return state.settings.find(setting => setting.key === 'use_blemished_sku_pattern') || null
    }
  },
  mutations: {
    // Set selected product mutation
    SET_SELECTED_PRODUCT (state, product) {
      state.selectedProduct = Object.assign({}, product)
    },
    // Set selected product mutation
    SET_SELECTED_PRODUCT_FIELD (state, { field, value }) {
      Vue.set(state.selectedProduct, field, value)
    },
    // Set Product Details Menu Mutation
    SET_PRODUCT_DETAILS_MENU (state, val) {
      state.productDetailsMenu = val
    },
    SET_PRODUCT_BY_ID_LOADING (state, value) {
      state.productByIdLoading = value
    },
    SET_PRODUCT_PROCESSING (state, value) {
      state.productProcessing = value
    },
    SET_PRODUCT_ATTRIBUTES (state, attributes) {
      Vue.set(state.selectedProduct, 'attributes_grouped', attributes)
    },
    SET_PRODUCT_EDITING (state, value) {
      state.isProductEditing = value
    },
    SET_INVENTORY_MOVEMENT_ITEMS_LIST (state, value) {
      Vue.prototype.$set(state.inventoryMovement, 'itemsList', value)
    },
    SET_INVENTORY_MOVEMENT_TABLE_SPECIFICATIONS (state, value) {
      Vue.prototype.$set(state.inventoryMovement, 'tableSpecifications', value)
    },
    SET_INVENTORY_MOVEMENT_MAX_PAGES (state, value) {
      Vue.prototype.$set(state.inventoryMovement, 'maxPages', value)
    },
    SET_INVENTORY_MOVEMENT_META (state, value) {
      Vue.prototype.$set(state.inventoryMovement, 'meta', value)
    },
    SET_INVENTORY_MOVEMENT_LOADING (state, value) {
      Vue.prototype.$set(state.inventoryMovement, 'loading', value)
    },
    SET_DASHBOARD_DATA (state, { field, value }) {
      Vue.prototype.$set(state.dashboardData, field, value)
    },
    SET_SUPPLIERS (state, value) {
      state.suppliers = value
    },
    SET_SETTINGS (state, value) {
      state.settings = value
    },
    SET_SETTINGS_LOADING (state, value) {
      state.settingsLoading = value
    }
  },
  actions: {
    async getProductID ({ commit }, productID) {
      try {
        const product = await getProductByID(productID)
        commit('SET_SELECTED_PRODUCT', product)
      } catch (e) {
        throw e
      }
    },
    // Set Product Details Menu Actions
    setProductDetailsMenu ({ commit }, val) {
      commit('SET_PRODUCT_DETAILS_MENU', val)
    },

    // Sets the main category and path of selected product
    setSelectedProductMainCategory ({ commit, getters }, payload) {
      const product = getters.getSelectedProduct
      product.category_main = payload.category
      product.category_main_path = payload.path
      commit('SET_SELECTED_PRODUCT', product)
    },

    // Adds additional categories
    addCategoriesToSelectedProduct ({ commit, getters }, payload) {
      const product = getters.getSelectedProduct
      product.category_others.push(payload.category)
      product.category_others_path = toArray(product.category_others_path)
      payload.path.forEach(category => {
        // We only put unique elements
        const exists = product.category_others_path.filter(element => element.id === category.id).length > 0
        if (!exists) {
          product.category_others_path.push(category)
        }
      })
      commit('SET_SELECTED_PRODUCT', product)
    },

    // inventory movement requests
    async getInventoryMovementItemsRequest ({ commit, state }, query) {
      query.productId = state.selectedProduct.id
      try {
        commit('SET_INVENTORY_MOVEMENT_LOADING', true)
        const response = await getProductInventoryMovements(query)
        commit('SET_INVENTORY_MOVEMENT_ITEMS_LIST', response.data)

        if (response.table_specifications) commit('SET_INVENTORY_MOVEMENT_TABLE_SPECIFICATIONS', response.table_specifications)
      } catch (e) {
        throw e
      } finally {
        commit('SET_INVENTORY_MOVEMENT_LOADING', false)
      }
    },
    async getInventoryMovementMetaRequest ({ commit, state }, query) {
      query.productId = state.selectedProduct.id
      try {
        const response = await getProductInventoryMovements(query)
        if (response.meta && response.meta.last_page) commit('SET_INVENTORY_MOVEMENT_MAX_PAGES', response.meta.last_page)
        if (response.meta) commit('SET_INVENTORY_MOVEMENT_META', response.meta)
      } catch (e) {
        throw e
      }
    },
    async loadInventoryMovementTableSpecifications ({ commit }, query) {
      try {
        const response = await getProductInventoryMovements(query)

        if (response.data?.table_specifications) commit('SET_INVENTORY_MOVEMENT_TABLE_SPECIFICATIONS', response.data.table_specifications)
        return Promise.resolve()
      } catch (e) {
        throw e
      }
    },
    async fetchSuppliers ({ commit }, id) {
      try {
        const { data } = await getProductSuppliers(id)
        if (data) commit('SET_SUPPLIERS', data)
      } catch (e) {
        throw e
      }
    },
    async fetchBlemishedSkuSettings ({ commit }) {
      try {
        commit('SET_SETTINGS_LOADING', true)
        const { data } = await settingsBlemishedSkuPatternSettings()
        if (!data) return
        commit('SET_SETTINGS', data)
      } catch (e) {
      } finally {
        commit('SET_SETTINGS_LOADING', false)
      }
    }
  }
}
