import Vue from 'vue'
import { fetchIntegrations, getIntegrationInstanceById } from '@/services/IntegrationsService'
import moment from "moment-timezone";

export const IntegrationsModule = {
  namespaced: true,
  state: {
    integrationsData: {
      items: [],
      loading: false,
      categories: [{ name: 'All', value: 'all' }],
      startDate: moment().subtract(1, 'months').startOf('day'),
      endDate: moment(),
    },
    selectedIntegration: null,
    selectedInstance: null,
    selectedSalesChannel: null,
    integrationForInitialSetup: null,
    automatedWarehouses: null,
    hidedSalesChannels: [],
    integrationPath: null,
  },
  getters: {
    getIntegrationsData: state => state.integrationsData,
    getSelectedIntegration: state => state.selectedIntegration,
    getIntegrationPath: state => state.integrationPath,
    getSelectedInstance: state => state.selectedInstance,
    getSelectedSalesChannel: state => state.selectedSalesChannel,
    getIntegrationForInitialSetup: state => state.integrationForInitialSetup,
    getAutomatedWarehouses: state => state.automatedWarehouses,
    getActiveSalesChannels: state => state.integrationsData.items
      .filter(channel => channel?.integration_type === 'sales_channel' && channel?.integration_instances.length),
    getHidedSalesChannels: state => state.hidedSalesChannels,
    getStartDate: state => state.startDate,
    getEndDate: state => state.endDate
  },
  mutations: {
    SET_INTEGRATIONS_DATA_FIELD (state, { field, value }) {
      Vue.prototype.$set(state.integrationsData, field, value)
    },
    SET_SELECTED_INTEGRATION (state, value) {
      state.selectedIntegration = value
    },
    SET_INTEGRATION_PATH (state, value) {
      state.integrationPath = value
    },
    SET_SELECTED_INSTANCE (state, value) {
      state.selectedInstance = value
    },
    SET_SELECTED_SALES_CHANNEL (state, value) {
      state.selectedSalesChannel = value
    },
    SET_INTEGRATION_FOR_INITIAL_SETUP (state, value) {
      state.integrationForInitialSetup = value
    },
    SET_AUTOMATED_WAREHOUSES (state, value) {
      state.automatedWarehouses = value
    },
    UPDATE_INTEGRATION_DATA (state, data) {
      state.selectedIntegration = data.integration
      state.selectedInstance = data
      state.selectedSalesChannel = data.sales_channel
    },
    SET_START_DATE (state, value) {
      state.startDate = value
    },
    SET_END_DATE (state, value) {
      state.endDate = value
    },
  },
  actions: {
    async fetchIntegrations ({ commit, getters }) {
      if (getters.getIntegrationsData.loading) return
      try {
        commit('SET_INTEGRATIONS_DATA_FIELD', { field: 'loading', value: true })
        const { data } = await fetchIntegrations()
        commit('SET_INTEGRATIONS_DATA_FIELD', { field: 'items', value: data.filter(item => !getters.getHidedSalesChannels.includes(item.name)) })

        let categories = data.map(i => {
          if (i.integration_sub_type) {
            return {
              name: i.integration_sub_type,
              value: i.integration_sub_type,
              variableName: 'integration_sub_type'
            }
          }
          return {
            name: i.integration_type,
            value: i.integration_type,
            variableName: 'integration_type'
          }
        })
        categories = [{ name: 'All', value: 'all' }, ...categories]
        commit('SET_INTEGRATIONS_DATA_FIELD', { field: 'categories', value: categories })
      } catch (e) {
        return Promise.reject(e)
      } finally {
        commit('SET_INTEGRATIONS_DATA_FIELD', { field: 'loading', value: false })
      }
    },
    async fetchIntegrationById ({ commit }, id) {
      try {
        const { data } = await getIntegrationInstanceById(id)
        if (data) commit('SET_SELECTED_INSTANCE', data)
      } catch (e) {
        return Promise.reject(e)
      }
    }
  }
}
