import axiosConfig from '@/config/axiosConfig'
import buildUrlParams from '@/support/buildUrlParams'
import {getAxiosRequest} from '@/support/apiRequestHelpers'
import axios from 'axios'

const CancelToken = axios.CancelToken

let cancelPOLinesRequest = () => false

const baseURL = '/purchase-orders'

const getUrl = (path) => `${baseURL}/${path}`

export const getAllPurchaseOrders = async (options = {}) => {
  try {
    const { data } = await axiosConfig.get(baseURL, buildUrlParams(options))
    return data
  } catch (e) {
    return Promise.reject(e)
  }
}

export const getPurchaseOrderByID = async (orderID) => {
  const url = `${baseURL}/${orderID}`

  try {
    const { data } = await axiosConfig.get(url)
    return data.data
  } catch (e) {
    return Promise.reject(e)
  }
}

export const createPurchaseOrder = async (orderData) => {
  try {
    const { data } = await axiosConfig.post(baseURL, orderData)
    return data
  } catch (e) {
    return Promise.reject(e)
  }
}

export const updatePurchaseOrderByID = async (orderID, orderData) => {
  const url = `${baseURL}/${orderID}`

  try {
    const { data } = await axiosConfig.patch(url, orderData)
    return data
  } catch (e) {
    return Promise.reject(e)
  }
}

export const importPurchaseOrders = async (importData) => {
  const url = `${baseURL}/import`

  try {
    const { data } = await axiosConfig.post(url, { ...importData })
    return data
  } catch (e) {
    return Promise.reject(e)
  }
}

export const deletePurchaseOrderLine = async (orderLineID) => {
  const url = `/purchase-order-lines/${orderLineID}`

  try {
    const { data } = await axiosConfig.delete(url)
    return data
  } catch (e) {
    return Promise.reject(e)
  }
}

export const archivePurchaseOrderByID = async (orderId) => {
  const url = `${baseURL}/${orderId}/archive`

  try {
    const { data } = await axiosConfig.put(url)
    return data
  } catch (e) {
    return Promise.reject(e)
  }
}

export const unarchivePurchaseOrderByID = async (orderId) => {
  const url = `${baseURL}/${orderId}/unarchived`

  try {
    const { data } = await axiosConfig.put(url)
    return data
  } catch (e) {
    return Promise.reject(e)
  }
}

export const archivePurchaseOrders = async (sendData) => {
  const url = `${baseURL}/archive`

  try {
    const { data } = await axiosConfig.put(url, sendData)
    return data
  } catch (e) {
    return Promise.reject(e)
  }
}

export const unarchivePurchaseOrders = async (sendData) => {
  const url = `${baseURL}/unarchive`

  try {
    const { data } = await axiosConfig.put(url, sendData)
    return data
  } catch (e) {
    return Promise.reject(e)
  }
}

export const deletePurchaseOrderByID = async (orderID) => {
  const url = `${baseURL}/${orderID}`

  try {
    const { data } = await axiosConfig.delete(url)
    return data
  } catch (e) {
    return Promise.reject(e)
  }
}

export const deletePurchaseOrders = async (sendData) => {
  try {
    const { data } = await axiosConfig.delete(baseURL, { params: sendData })
    return data
  } catch (e) {
    return Promise.reject(e)
  }
}

export const settingsPurchaseOrders = async (sendData) => {
  let url = 'settings/purchase-orders'
  try {
    const { data } = await axiosConfig.get(url)
    return data
  } catch (e) {
    return Promise.reject(e)
  }
}

export const updatePurchaseOrdersSettings = async (settings) => {
  const url = '/settings/purchase-orders'

  try {
    const { data } = await axiosConfig.put(url, settings)
    return data
  } catch (e) {
    return Promise.reject(e)
  }
}

export const fetchPurchaseOrderDropshipShipments = async (id) => {
  let url = `${baseURL}/${id}/dropship-shipments`

  try {
    const { data } = await axiosConfig.get(url)
    return data
  } catch (e) {
    return Promise.reject(e)
  }
}

export const submitPurchaseOrder = async (id) => {
  const url = `${baseURL}/${id}/submit`
  try {
    const { data } = await axiosConfig.put(url)
    return data
  } catch (e) {
    return Promise.reject(e)
  }
}

export const isDeletablePurchaseOrder = async (ids) => {
  const url = `${baseURL}/deletable`
  try {
    const { data } = await axiosConfig.post(url, { ids })
    return data
  } catch (e) {
    return Promise.reject(e)
  }
}

export const duplicatePurchaseOrder = async (id) => {
  const url = `${baseURL}/${id}/duplicate`
  try {
    const { data } = await axiosConfig.post(url)
    return data
  } catch (e) {
    return Promise.reject(e)
  }
}

export const createReceivingDiscrepancyForPurchaseOrder = async (id) => {
  try {
    const { data } = await axiosConfig.post(`${baseURL}/${id}/receiving-discrepancy`)
    return data
  } catch (e) {
    return Promise.reject(e)
  }
}

export const fetchPurchaseOrderFbaInventory = async (orderId) => {
  const url = `${baseURL}/${orderId}/fba-inventory-movements`
  try {
    const { data } = await axiosConfig.get(url)
    return data
  } catch (e) {
    return Promise.reject(e)
  }
}

export const previewPurchaseOrdersImport = async (sendData) => {
  const url = `${baseURL}/import/preview`
  try {
    const { data } = await axiosConfig.post(url, sendData)
    return data
  } catch (e) {
    return Promise.reject(e)
  }
}

export const importPurchaseOrdersData = async (sendData) => {
  const url = `${baseURL}/import`
  try {
    const { data } = await axiosConfig.post(url, sendData)
    return data
  } catch (e) {
    return Promise.reject(e)
  }
}

export const buildPurchaseOrder = async (sendData) => {
  const url = `${baseURL}/build`
  try {
    const { data } = await axiosConfig.post(url, sendData)
    return data
  } catch (e) {
    return Promise.reject(e)
  }
}

export const exportBuildPurchaseOrder = async (sendData) => {
  const url = `${baseURL}/build-export`
  try {
    const { data } = await axiosConfig.post(url, sendData)
    return data
  } catch (e) {
    return Promise.reject(e)
  }
}

export const getOrderNotes = async (orderId) => {
  try {
    const { data } = await axiosConfig.get(`${baseURL}/${orderId}/notes`)
    return data
  } catch (e) {
    return Promise.reject(e)
  }
}

export const addOrderNote = async (orderId, payload) => {
  try {
    const { data } = await axiosConfig.post(`${baseURL}/${orderId}/notes`, { ...payload })
    return data
  } catch (e) {
    return Promise.reject(e)
  }
}

export const deleteNote = async (orderId, noteId) => {
  try {
    const { data } = await axiosConfig.delete(`${baseURL}/${orderId}/notes/${noteId}`)
    return data
  } catch (e) {
    return Promise.reject(e)
  }
}

export const translateCsvListForPurchaseOrders = async (payload) => {
  try {
    const { data } = await axiosConfig.post(`${baseURL}/import-lines`, payload)
    return data
  } catch (e) {
    return Promise.reject(e)
  }
}

export const approvePurchaseOrders = async (payload) => {
  try {
    const { data } = await axiosConfig.post(`${baseURL}/approve`, payload)
    return data
  } catch (e) {
    return Promise.reject(e)
  }
}

export const submitPurchaseOrders = async (payload) => {
  try {
    const { data } = await axiosConfig.post(`${baseURL}/submit`, payload)
    return data
  } catch (e) {
    return Promise.reject(e)
  }
}

export const fetchPurchaseOrderLine = async (orderId, payload) => {
  // cancel any pending request
  cancelPOLinesRequest('canceled previous fetch purchase order lines request')
  // create cancel token
  let cancelToken = new CancelToken(function executor (c) {
    // An executor function receives a cancel function as a parameter
    cancelPOLinesRequest = c
  })

  return getAxiosRequest(getUrl(`${orderId}/lines?included=["price","quantity","received","discount","discount_rate","eta","tax_allocation","tax_rate","tax_rate_id","img_url","barcode","sku","supplier_sku","name", "backorder_coverage", "backorder_releases"]
`), {params: {...payload, cancelToken}})
}

export const applyDiscountRateToPurchaseOrder = async (orderId, discountRate) => {
  try {
    const { data } = await axiosConfig.post(`${baseURL}/${orderId}/apply-discount-rate`, { discount_rate: discountRate })
    return data
  } catch (e) {
    return Promise.reject(e)
  }
}

export const applyPricingTierToPurchaseOrder = async (orderId, supplierPricingTierId) => {
  try {
    const { data } = await axiosConfig.post(`${baseURL}/${orderId}/apply-pricing-tier/${supplierPricingTierId}`)
    return data
  } catch (e) {
    return Promise.reject(e)
  }
}

export const shipDropshipPurchaseOrder = async (orderId, payload) => {
  try {
    const { data } = await axiosConfig.post(`${baseURL}/${orderId}/dropship`, payload)
    return data
  } catch (e) {
    return Promise.reject(e)
  }
}

export const sendPurchaseOrderASN = async (id) => {
  const url = `${baseURL}/${id}/asn`
  try {
    const { data } = await axiosConfig.post(url)
    return data
  } catch (e) {
    return Promise.reject(e)
  }
}
