import axiosConfig from '@/config/axiosConfig'
import buildUrlParams from '@/support/buildUrlParams'
import {isFailedRequest} from "@/support/integrations/integrationInstanceHelpers";

const baseURL = '/customers'

export const getAllCustomers = async (options = {}) => {
  try {
    const { data } = await axiosConfig.get(baseURL, buildUrlParams(options))
    return data
  } catch (e) {
    return Promise.reject(e)
  }
}

export const getCustomerById = async (id) => {
  try {
    const { data } = await axiosConfig.get(`${baseURL}/${id}`)
    return data
  } catch (e) {
    return Promise.reject(e)
  }
}

export const getUnpaidOrders = async (id) => {
  try {
    const { data } = await axiosConfig.get(`${baseURL}/${id}/unpaid-orders`)
    return data
  } catch (e) {
    return Promise.reject(e)
  }
}

export const getOpenSalesCredits = async (id) => {
  try {
    const { data } = await axiosConfig.get(`${baseURL}/${id}/open-sales-credits`)
    return data
  } catch (e) {
    return Promise.reject(e)
  }
}

// Create Customer
export const createCustomer = async (customerData) => {
  try {
    const { data } = await axiosConfig.post(baseURL, { ...customerData })
    return data
  } catch (e) {
    return Promise.reject(e)
  }
}

// Update Customer
export const updateCustomer = async (customerID, customerData) => {
  const url = `${baseURL}/${customerID}`

  try {
    const { data } = await axiosConfig.put(url, { ...customerData })
    return data
  } catch (e) {
    return Promise.reject(e)
  }
}

export const addOtherAddress = async ({ customerID, addressData }) => {
  const url = `${baseURL}/${customerID}/addresses`

  try {
    const { data } = await axiosConfig.post(url, addressData)
    return data
  } catch (e) {
    return Promise.reject(e)
  }
}

export const checkIfEditableAddress = async (ids) => {
  const url = `/addresses/locked`

  try {
    const { data } = await axiosConfig.post(url, { ids })
    return data
  } catch (e) {
    return Promise.reject(e)
  }
}

export const setDefaultShippingAddress = async ({ customerID, addressID, check }) => {
  const url = `${baseURL}/${customerID}/set-default-shipping-address/${addressID}`

  try {
    const { data } = await axiosConfig.put(url, { check })
    return data
  } catch (e) {
    return Promise.reject(e)
  }
}

export const setDefaultBillingAddress = async ({ customerID, addressID, check }) => {
  const url = `${baseURL}/${customerID}/set-default-billing-address/${addressID}`

  try {
    const { data } = await axiosConfig.put(url, { check })

    return data
  } catch (e) {
    return Promise.reject(e)
  }
}

export const archiveCustomerByID = async (id) => {
  const url = `${baseURL}/${id}/archive`

  try {
    const { data } = await axiosConfig.put(url)
    return data
  } catch (e) {
    return Promise.reject(e)
  }
}

export const unarchiveCustomerByID = async (id) => {
  const url = `${baseURL}/${id}/unarchived`

  try {
    const { data } = await axiosConfig.put(url)
    return data
  } catch (e) {
    return Promise.reject(e)
  }
}

export const archiveCustomers = async (sendData) => {
  const url = `${baseURL}/archive`

  try {
    const { data } = await axiosConfig.put(url, sendData)
    return data
  } catch (e) {
    return Promise.reject(e)
  }
}

export const unarchiveCustomers = async (sendData) => {
  const url = `${baseURL}/unarchive`

  try {
    const { data } = await axiosConfig.put(url, sendData)
    return data
  } catch (e) {
    return Promise.reject(e)
  }
}

export const deleteCustomerByID = async (id) => {
  const url = `${baseURL}/${id}`

  try {
    const { data } = await axiosConfig.delete(url)
    return data
  } catch (e) {
    return Promise.reject(e)
  }
}

export const deleteCustomers = async (sendData) => {
  try {
    const { data } = await axiosConfig.delete(baseURL, { params: sendData })
    return data
  } catch (e) {
    return Promise.reject(e)
  }
}

export const isDeletableCustomers = async (ids) => {
  try {
    const { data } = await axiosConfig.post(`${baseURL}/deletable`, { ids })
    return data
  } catch (e) {
    return Promise.reject(e)
  }
}

export const allocateSalesCreditToOrders = async (customerId, salesCreditId, payload) => {
  try {
    const { data } = await axiosConfig.post(`${baseURL}/${customerId}/sales-credits/${salesCreditId}/allocate`, payload)
    return data
  } catch (e) {
    return Promise.reject(e)
  }
}

export const previewCustomersImport = async (sendData) => {
  const url = `${baseURL}/import/preview`
  try {
    const { data } = await axiosConfig.post(url, sendData)
    return data
  } catch (e) {
    return Promise.reject(e)
  }
}

export const importCustomersData = async (sendData) => {
  const url = `${baseURL}/import`
  try {
    const { data } = await axiosConfig.post(url, sendData)
    return data
  } catch (e) {
    return Promise.reject(e)
  }
}

export const syncCustomersWithXero = async () => {
  const url = `${baseURL}/xero/sync`
  try {
    const { data } = await axiosConfig.get(url)
    if (isFailedRequest(data)) return Promise.reject({response:{data}})
    return data
  } catch (e) {
    return Promise.reject(e)
  }
}
