import axiosConfig from '@/config/axiosConfig'
import buildUrlParams from '@/support/buildUrlParams'
import { getAxiosRequest } from '@/support/apiRequestHelpers'
import { isFailedRequest } from '@/support/integrations/integrationInstanceHelpers'

const baseURL = '/xero'

export const fetchTransactions = async (options = {}) => {
  try {
    const { data } = await axiosConfig.get(baseURL, buildUrlParams(options))
    if (isFailedRequest(data)) return Promise.reject({ response: { data } })
    return data
  } catch (e) {
    return Promise.reject(e)
  }
}

export const syncTransactionById = async (id) => {
  const url = `${baseURL}/${id}/sync`
  try {
    const { data } = await axiosConfig.post(url)
    if (isFailedRequest(data)) return Promise.reject({ response: { data } })
    return data
  } catch (e) {
    return Promise.reject(e)
  }
}

export const syncMultipleTransactions = async (sendData) => {
  const url = `${baseURL}/sync-multiple`
  try {
    const { data } = await axiosConfig.post(url, sendData)
    if (isFailedRequest(data)) return Promise.reject({ response: { data } })
    return data
  } catch (e) {
    return Promise.reject(e)
  }
}

export const syncTransactions = async (instanceId) => {
  const url = `${baseURL}/${instanceId}/sync-transactions`
  try {
    const { data } = await axiosConfig.post(url)
    if (isFailedRequest(data)) return Promise.reject({ response: { data } })
    return data
  } catch (e) {
    return Promise.reject(e)
  }
}

export const checkIfXeroIsEnabled = async () => {
  const url = `${baseURL}/check-if-xero-is-enabled`
  try {
    const { data } = await axiosConfig.get(url)
    return data
  } catch (e) {
    return Promise.reject(e)
  }
}

export const downloadTaxRates = async () => {
  try {
    const { data } = await axiosConfig.post(`${baseURL}/tax-rates/download`)
    if (isFailedRequest(data)) return Promise.reject({ response: { data } })
    return data
  } catch (e) {
    console.log(e)
    return Promise.reject(e)
  }
}

export const getXeroTaxRates = async () => {
  const url = `${baseURL}/tax-rates`
  try {
    const { data } = await axiosConfig.get(url)
    if (isFailedRequest(data)) return Promise.reject({ response: { data } })
    return data
  } catch (e) {
    console.log(e)
    return Promise.reject(e)
  }
}

export const getXeroTaxRatesMappings = async () => {
  const url = `${baseURL}/tax-mappings`
  try {
    const data = await axiosConfig.get(url)
    if (isFailedRequest(data)) return Promise.reject({ response: { data } })
    return data
  } catch (e) {
    console.log(e)
    return Promise.reject(e)
  }
}

export const mapXeroTaxRates = async (sendData) => {
  const url = `${baseURL}/map-tax-rates`
  try {
    const { data } = await axiosConfig.put(url, sendData)
    if (isFailedRequest(data)) return Promise.reject({ response: { data } })
    return data
  } catch (e) {
    console.log(e)
    return Promise.reject(e)
  }
}

export const deleteTaxRateMapping = async (mappingId) => {
  const url = `${baseURL}/tax-rate-mapping/${mappingId}`
  try {
    const { data } = await axiosConfig.delete(url)
    if (isFailedRequest(data)) return Promise.reject({ response: { data } })
    return data
  } catch (e) {
    console.log(e)
    return Promise.reject(e)
  }
}

// Get Mapping Nominal Codes From Xero
export const downloadAccounts = async () => {
  try {
    const { data } = await axiosConfig.post(`${baseURL}/accounts/download`)
    if (isFailedRequest(data)) return Promise.reject({ response: { data } })
    return data
  } catch (e) {
    return Promise.reject(e)
  }
}

// Create Xero SKU Nominal Codes
export const createSkuNominalCodes = async (options) => {
  try {
    const { data } = await axiosConfig.post(`${baseURL}/accounts/create-nominal-codes`, options)
    if (isFailedRequest(data)) return Promise.reject({ response: { data } })
    return data
  } catch (e) {
    return Promise.reject(e)
  }
}

export const getAccounts = async (options) => {
  try {
    const { data } = await axiosConfig.get(`${baseURL}/accounts`, buildUrlParams(options))
    if (isFailedRequest(data)) return Promise.reject({ response: { data } })
    return data
  } catch (e) {
    return Promise.reject(e)
  }
}

// get Tax Rates
export const fetchTaxRatesForMapping = async () => {
  try {
    const { data } = await axiosConfig.get(`${baseURL}?included=["name","rate"]&limit=-1`)
    return data
  } catch (e) {
    return Promise.reject(e)
  }
}

export const syncContacts = async () => {
  try {
    const { data } = await axiosConfig.post(`${baseURL}/contacts/sync`)
    if (isFailedRequest(data)) return Promise.reject({ response: { data } })
    return data
  } catch (e) {
    return Promise.reject(e)
  }
}

// authorize
export const authXero = async () => await getAxiosRequest(`${baseURL}/authorize`)
