import axiosConfig from '../config/axiosConfig'
import buildUrlParams from '@/support/buildUrlParams'

const baseURL = '/warehouses'

export const fetchWarehouses = async (options = {}) => {
  try {
    const { data } = await axiosConfig.get(baseURL, buildUrlParams(options))
    return data
  } catch (e) {
    return Promise.reject(e)
  }
}

export const createWarehouse = async (warehouseData) => {
  try {
    const { data } = await axiosConfig.post(baseURL, { ...warehouseData })
    return data
  } catch (e) {
    return Promise.reject(e)
  }
}

export const getWarehouseByID = async (ID) => {
  try {
    const { data } = await axiosConfig.get(`${baseURL}/${ID}`)

    return data
  } catch (e) {
    return Promise.reject(e)
  }
}

export const updateWarehouse = async (warehouseID, warehouseData) => {
  const url = `${baseURL}/${warehouseID}`

  try {
    const { data } = await axiosConfig.put(url, { ...warehouseData })
    return data
  } catch (e) {
    return Promise.reject(e)
  }
}

export const archiveWarehouseByID = async (productID) => {
  const url = `${baseURL}/${productID}/archive`

  try {
    const { data } = await axiosConfig.put(url)
    return data
  } catch (e) {
    return Promise.reject(e)
  }
}

export const unarchiveWarehouseByID = async (productID) => {
  const url = `${baseURL}/${productID}/unarchived`

  try {
    const { data } = await axiosConfig.put(url)
    return data
  } catch (e) {
    return Promise.reject(e)
  }
}

export const archiveWarehouses = async (sendData) => {
  const url = `${baseURL}/archive`

  try {
    const { data } = await axiosConfig.put(url, sendData)
    return data
  } catch (e) {
    return Promise.reject(e)
  }
}

export const unarchiveWarehouses = async (sendData) => {
  const url = `${baseURL}/unarchive`

  try {
    const { data } = await axiosConfig.put(url, sendData)
    return data
  } catch (e) {
    return Promise.reject(e)
  }
}

export const deleteWarehouseByID = async (productID) => {
  const url = `${baseURL}/${productID}`

  try {
    const { data } = await axiosConfig.delete(url)
    return data
  } catch (e) {
    return Promise.reject(e)
  }
}

export const deleteWarehouses = async (sendData) => {
  try {
    const { data } = await axiosConfig.delete(baseURL, { params: sendData })
    return data
  } catch (e) {
    return Promise.reject(e)
  }
}

export const isDeletableWarehouses = async (ids) => {
  try {
    const { data } = await axiosConfig.post(`${baseURL}/deletable`, { ids })
    return data
  } catch (e) {
    return Promise.reject(e)
  }
}

export const isArchivable = async (ids) => {
  try {
    const { data } = await axiosConfig.post(`${baseURL}/is-archivable`, { ids })
    return data
  } catch (e) {
    return Promise.reject(e)
  }
}

/**
 * Gets the summary of inventory
 * for the given warehouse
 *
 * @param {*} warehouseId
 */
export const getInventorySummaries = async (warehouseId) => {
  try {
    const { data } = await axiosConfig.get(`${baseURL}/${warehouseId}/inventory-summary`)
    return data
  } catch (e) {
    return Promise.reject(e)
  }
}

/**
 * Gets the inventory activity
 * for the given warehouse
 *
 * @param {*} warehouseId
 */
export const getInventoryActivity = async (warehouseId) => {
  try {
    const { data } = await axiosConfig.get(`${baseURL}/${warehouseId}/inventory-activity`)
    return data
  } catch (e) {
    return Promise.reject(e)
  }
}

/**
 * Gets the inventory shipments
 * received at the warehouse
 *
 * @param {*} warehouseId
 */
export const getInventoryShipments = async (warehouseId, from, to) => {
  try {
    const { data } = await axiosConfig.get(`${baseURL}/${warehouseId}/inventory-shipments?from=${from}&to=${to}`)
    return data
  } catch (e) {
    return Promise.reject(e)
  }
}

export const getIncomingPurchaseOrders = async (warehouseId) => {
  try {
    const { data } = await axiosConfig.get(`${baseURL}/${warehouseId}/incoming-purchase-orders`)
    return data
  } catch (e) {
    return Promise.reject(e)
  }
}

export const previewWarehousesImport = async (sendData) => {
  const url = `${baseURL}/import/preview`
  try {
    const { data } = await axiosConfig.post(url, sendData)
    return data
  } catch (e) {
    return Promise.reject(e)
  }
}

export const importWarehousesData = async (sendData) => {
  const url = `${baseURL}/import`
  try {
    const { data } = await axiosConfig.post(url, sendData)
    return data
  } catch (e) {
    return Promise.reject(e)
  }
}

export const fetch3PLAndDirectWarehouses = async (options = {}) => {
  try {
    const filters = {
      conjunction: 'or',
      filterSet: [{ column: 'type', operator: '=', value: '3pl' }, { column: 'type', operator: '=', value: 'direct' }]
    }
    const { data } = await fetchWarehouses({ filters: filters, limit: -1 })
    return data
  } catch (e) {
    return Promise.reject(e)
  }
}
