import axiosConfig from '../config/axiosConfig'
import buildUrlParams from '@/support/buildUrlParams'

const baseURL = '/warehouses/transfers'

export const fetchWarehousesTransfers = async (options = {}) => {
  try {
    const { data } = await axiosConfig.get(baseURL, buildUrlParams(options))
    return data
  } catch (e) {
    return Promise.reject(e)
  }
}

export const getWarehouseTransferById = async (id) => {
  try {
    const { data } = await axiosConfig.get(`${baseURL}/${id}`)
    return data
  } catch (e) {
    return Promise.reject(e)
  }
}

export const getWarehouseTransferReceipts = async (transferId) => {
  try {
    const { data } = await axiosConfig.get(`${baseURL}/${transferId}/receipts`)
    return data
  } catch (e) {
    return Promise.reject(e)
  }
}

export const createWarehousesTransfer = async (sendData) => {
  try {
    const { data } = await axiosConfig.post(baseURL, sendData)
    return data
  } catch (e) {
    return Promise.reject(e)
  }
}

export const createWarehouseTransferReceipt = async (transferId, payload) => {
  try {
    const { data } = await axiosConfig.post(`${baseURL}/${transferId}/receive`, payload)
    return data
  } catch (e) {
    return Promise.reject(e)
  }
}

export const updateWarehouseTransferReceipt = async (transferId, receiptId, payload) => {
  try {
    const { data } = await axiosConfig.put(`${baseURL}/${transferId}/receipts/${receiptId}`, payload)
    return data
  } catch (e) {
    return Promise.reject(e)
  }
}

export const deleteTransferReceipt = async (transferId, receiptId) => {
  try {
    const { data } = await axiosConfig.delete(`${baseURL}/${transferId}/receipts/${receiptId}`)
    return data
  } catch (e) {
    return Promise.reject(e)
  }
}

export const updateWarehouseTransfer = async (id, sendData) => {
  try {
    const { data } = await axiosConfig.put(`${baseURL}/${id}`, sendData)
    return data
  } catch (e) {
    return Promise.reject(e)
  }
}

export const openWarehouseTransfer = async (transferId, payload) => {
  const url = `${baseURL}/${transferId}/open`
  try {
    const { data } = await axiosConfig.put(url, payload)
    return data
  } catch (e) {
    return Promise.reject(e)
  }
}

export const duplicateWarehouseTransfer = async (transferId) => {
  const url = `${baseURL}/${transferId}/duplicate`
  try {
    const { data } = await axiosConfig.post(url)
    return data
  } catch (e) {
    return Promise.reject(e)
  }
}

export const makeTransferDraft = async (transferId, payload) => {
  const url = `${baseURL}/${transferId}/draft`
  try {
    const { data } = await axiosConfig.put(url, payload)
    return data
  } catch (e) {
    return Promise.reject(e)
  }
}

export const archiveWarehouseTransferByID = async (id) => {
  const url = `${baseURL}/${id}/archive`

  try {
    const { data } = await axiosConfig.put(url)
    return data
  } catch (e) {
    return Promise.reject(e)
  }
}

export const unarchiveWarehouseTransferByID = async (id) => {
  const url = `${baseURL}/${id}/unarchived`

  try {
    const { data } = await axiosConfig.put(url)
    return data
  } catch (e) {
    return Promise.reject(e)
  }
}

export const archiveWarehouseTransfers = async (sendData) => {
  const url = `${baseURL}/archive`

  try {
    const { data } = await axiosConfig.put(url, sendData)
    return data
  } catch (e) {
    return Promise.reject(e)
  }
}

export const unarchiveWarehouseTransfers = async (sendData) => {
  const url = `${baseURL}/unarchive`

  try {
    const { data } = await axiosConfig.put(url, sendData)
    return data
  } catch (e) {
    return Promise.reject(e)
  }
}

export const deleteWarehouseTransferByID = async (id) => {
  const url = `${baseURL}/${id}`

  try {
    const { data } = await axiosConfig.delete(url)
    return data
  } catch (e) {
    return Promise.reject(e)
  }
}

export const deleteWarehouseTransfers = async (sendData) => {
  try {
    const { data } = await axiosConfig.delete(baseURL, { params: sendData })
    return data
  } catch (e) {
    return Promise.reject(e)
  }
}

export const isDeletableWarehouseTransfers = async (ids) => {
  try {
    const { data } = await axiosConfig.post(`${baseURL}/deletable`, { ids })
    return data
  } catch (e) {
    return Promise.reject(e)
  }
}

export const fetchWarehouseTransferFbaInventory = async (id) => {
  const url = `warehouse-transfer/${id}/fba-inventory-movements`
  try {
    const { data } = await axiosConfig.get(url)
    return data
  } catch (e) {
    return Promise.reject(e)
  }
}

export const translateCsvListForWarehouseTransfers = async (sendData) => {
  try {
    const { data } = await axiosConfig.post(`${baseURL}/import-lines`, sendData)
    return data
  } catch (e) {
    return Promise.reject(e)
  }
}

export const getWarehouseTransferNotes = async (transferId) => {
  try {
    const { data } = await axiosConfig.get(`${baseURL}/${transferId}/notes`)
    return data
  } catch (e) {
    return Promise.reject(e)
  }
}

export const addWarehouseTransferNote = async (transferId, payload) => {
  try {
    const { data } = await axiosConfig.post(`${baseURL}/${transferId}/notes`, { ...payload })
    return data
  } catch (e) {
    return Promise.reject(e)
  }
}

export const deleteWarehouseTransferNote = async (transferId, noteId) => {
  try {
    const { data } = await axiosConfig.delete(`${baseURL}/${transferId}/notes/${noteId}`)
    return data
  } catch (e) {
    return Promise.reject(e)
  }
}

export const sendWarehouseTransferASN = async (id) => {
  const url = `${baseURL}/${id}/asn`
  try {
    const { data } = await axiosConfig.post(url)
    return data
  } catch (e) {
    return Promise.reject(e)
  }
}

export const createReceivingDiscrepancyForWarehouseTransfer = async (id) => {
  try {
    const { data } = await axiosConfig.post(`${baseURL}/${id}/receiving-discrepancy`)
    return data
  } catch (e) {
    return Promise.reject(e)
  }
}

export const updateWarehouseTransferLines = async (transferId, payload) => {
  try {
    const { data } = await axiosConfig.patch(`${baseURL}/${transferId}/lines`, payload)
    return data
  } catch (e) {
    return Promise.reject(e)
  }
}
