import axiosConfig from '@/config/axiosConfig'
import buildUrlParams from '@/support/buildUrlParams'
import { isFailedRequest } from '@/support/integrations/integrationInstanceHelpers'

const baseURL = '/suppliers'
const inventoriesURL = '/supplier-inventories'

export const fetchSuppliers = async (options = {}) => {
  try {
    const { data } = await axiosConfig.get(baseURL, buildUrlParams(options))
    return data
  } catch (e) {
    return Promise.reject(e)
  }
}

export const fetchSupplierInventories = async (options = {}) => {
  try {
    const { data } = await axiosConfig.get(inventoriesURL, buildUrlParams(options))
    return data
  } catch (e) {
    return Promise.reject(e)
  }
}

export const getSupplierById = async (id) => {
  try {
    const { data } = await axiosConfig.get(`${baseURL}/${id}`)
    return data
  } catch (e) {
    return Promise.reject(e)
  }
}

export const getSupplierPricingTiers = async (id, payload = {}) => {
  try {
    const { data } = await axiosConfig.get(`${baseURL}/${id}/pricing-tiers`, buildUrlParams(payload))
    return data
  } catch (e) {
    return Promise.reject(e)
  }
}

// Get All Suppliers
export const getAllSuppliers = async () => {
  try {
    const { data } = await axiosConfig.get(`${baseURL}?limit=-1&visible_only=0`)
    return data
  } catch (e) {
    return Promise.reject(e)
  }
}

// Create New Supplier
export const createSupplier = async (supplierData) => {
  try {
    const { data } = await axiosConfig.post(baseURL, { ...supplierData })
    return data
  } catch (e) {
    return Promise.reject(e)
  }
}

// Update Supplier
export const updateSupplier = async (supplierID, supplierData) => {
  const url = `${baseURL}/${supplierID}`

  try {
    const { data } = await axiosConfig.put(url, { ...supplierData })
    return data
  } catch (e) {
    return Promise.reject(e)
  }
}

export const createSupplierProduct = async (payload) => {
  const url = `${baseURL}/supplier-products`
  return axiosConfig.post(url, payload)
}

export const getPricingTiers = async () => {
  const url = `/supplier-pricing-tiers?limit=-1`
  const tiers = (await axiosConfig.get(url)).data.data
  return tiers.map(tier => {
    tier.supplier_pricing_tier_id = tier.id
    tier.supplier_pricing_tier_name = tier.name
    tier.currency = tier.currency_code
    tier.price = null

    return tier
  })
}

export const addOtherAddress = async ({ supplierId, addressData }) => {
  const url = `${baseURL}/${supplierId}/addresses`

  try {
    const { data } = await axiosConfig.post(url, addressData)
    return data
  } catch (e) {
    return Promise.reject(e)
  }
}

export const createSupplierWarehouse = async ({ supplierId, warehouseData }) => {
  const url = `${baseURL}/${supplierId}/warehouses`

  try {
    const { data } = await axiosConfig.post(url, warehouseData)
    return data
  } catch (e) {
    return Promise.reject(e)
  }
}

export const getSupplierWarehouses = async (id) => {
  try {
    const { data } = await axiosConfig.get(`${baseURL}/${id}/warehouses?visible_only=0`)
    return data
  } catch (e) {
    return Promise.reject(e)
  }
}

export const fetchSupplierProducts = async ({ id, visibleOnly = 0, excluded = [], limit = 20, query = '', page = '', cancelToken = () => false }) => {
  try {
    let url = `${baseURL}/${id}/products?visible_only=${visibleOnly}&excluded=${excluded}&limit=${limit}&query=${query}`
    if (page) url += `&page=${page}`
    const { data } = await axiosConfig.get(url, { cancelToken })
    return data
  } catch (e) {
    return Promise.reject(e)
  }
}

export const updateSupplierInventory = async (inventoryId, payload) => {
  const url = `${inventoriesURL}/${inventoryId}`

  if (!payload.source) { delete payload.source }

  const { data } = await axiosConfig.put(url, { ...payload })
  return data
}

export const archiveSupplierById = async (supplierId) => {
  const url = `${baseURL}/${supplierId}/archive`

  const { data } = await axiosConfig.put(url)
  return data
}

export const unarchiveSupplierById = async (supplierId) => {
  const url = `${baseURL}/${supplierId}/unarchived`

  const { data } = await axiosConfig.put(url)
  return data
}

export const archiveSuppliers = async (sendData) => {
  const url = `${baseURL}/archive`

  try {
    const { data } = await axiosConfig.put(url, sendData)
    return data
  } catch (e) {
    return Promise.reject(e)
  }
}

export const unarchiveSuppliers = async (sendData) => {
  const url = `${baseURL}/unarchive`

  try {
    const { data } = await axiosConfig.put(url, sendData)
    return data
  } catch (e) {
    return Promise.reject(e)
  }
}

export const deleteSupplierByID = async (supplierId) => {
  const url = `${baseURL}/${supplierId}`

  try {
    const { data } = await axiosConfig.delete(url)
    return data
  } catch (e) {
    return Promise.reject(e)
  }
}

export const deleteSuppliers = async (sendData) => {
  try {
    const { data } = await axiosConfig.delete(baseURL, { params: sendData })
    return data
  } catch (e) {
    return Promise.reject(e)
  }
}

export const isDeletableSuppliers = async (ids) => {
  try {
    const { data } = await axiosConfig.post(`${baseURL}/deletable`, { ids })
    return data
  } catch (e) {
    return Promise.reject(e)
  }
}

export const getLinkedOrdersForWarehouse = async (warehouseId) => {
  try {
    const { data } = await axiosConfig.post(`${baseURL}/warehouses/${warehouseId}/linked-orders`)
    return data
  } catch (e) {
    return Promise.reject(e)
  }
}

export const importSupplierInventory = async (supplierId, sendData) => {
  const url = `${inventoriesURL}/${supplierId}/import`
  try {
    const { data } = await axiosConfig.post(url, sendData)
    return data
  } catch (e) {
    return Promise.reject(e)
  }
}

export const previewSuppliersImportData = async (sendData) => {
  const url = `${baseURL}/import/preview`
  try {
    const { data } = await axiosConfig.post(url, sendData)
    return data
  } catch (e) {
    return Promise.reject(e)
  }
}

export const importSuppliersData = async (sendData) => {
  const url = `${baseURL}/import`
  try {
    const { data } = await axiosConfig.post(url, sendData)
    return data
  } catch (e) {
    return Promise.reject(e)
  }
}

export const previewInventoryImport = async (supplierId, sendData) => {
  const url = `${baseURL}/${supplierId}/import/preview`
  try {
    const { data } = await axiosConfig.post(url, sendData)
    return data
  } catch (e) {
    return Promise.reject(e)
  }
}

export const importInventoryData = async (supplierId, sendData) => {
  const url = `${baseURL}/${supplierId}/import`
  try {
    const { data } = await axiosConfig.post(url, sendData)
    return data
  } catch (e) {
    return Promise.reject(e)
  }
}

export const fetchSupplierStock = async (supplierId, sendData) => {
  const url = `${baseURL}/${supplierId}/products-stocks`
  try {
    const { data } = await axiosConfig.post(url, sendData)
    return data
  } catch (e) {
    return Promise.reject(e)
  }
}

export const syncSuppliersWithXero = async () => {
  const url = `${baseURL}/xero/sync`
  try {
    const { data } = await axiosConfig.get(url)
    if (isFailedRequest(data)) return Promise.reject({response:{data}})
    return data
  } catch (e) {
    return Promise.reject(e)
  }
}
