import axiosConfig from '@/config/axiosConfig'
import buildUrlParams from '@/support/buildUrlParams'
import { getAxiosRequest } from '@/support/apiRequestHelpers'

const baseURL = '/reporting'

const getUrl = (path) => `${baseURL}/${path}`

export const getShipByScheduleItems = async (options = {}) => {
  try {
    const { data } = await axiosConfig.get(`${baseURL}/ship-by-schedule`, buildUrlParams(options))
    return data
  } catch (e) {
    return Promise.reject(e)
  }
}

// Get Sales Order By ID
export const getSalesOrderByID = async (orderID, included) => {
  const url = `${baseURL}/${orderID}`
  const params = {}
  if (included) params.included = JSON.stringify(included)

  try {
    const { data } = await axiosConfig.get(url, { params })
    return data.data
  } catch (e) {
    return Promise.reject(e)
  }
}

// Create or Update Order
export const createSalesOrder = async (orderData) => {
  try {
    const { data } = await axiosConfig.post(baseURL, orderData)
    return data
  } catch (e) {
    return Promise.reject(e)
  }
}

export const duplicateSalesOrder = async (id) => {
  const url = `${baseURL}/${id}/duplicate`
  try {
    const { data } = await axiosConfig.post(url)
    return data
  } catch (e) {
    return Promise.reject(e)
  }
}

export const resendSalesOrder = async ({ id, sendData }) => {
  const url = `${baseURL}/${id}/resend`
  try {
    const { data } = await axiosConfig.post(url, sendData)
    return data
  } catch (e) {
    return Promise.reject(e)
  }
}

// Update Sales Order By ID
export const updateSalesOrderByID = async (orderID, orderData) => {
  const url = `${baseURL}/${orderID}`
  try {
    const { data } = await axiosConfig.put(url, { ...orderData })
    return data
  } catch (e) {
    return Promise.reject(e)
  }
}

// Import Sales Orders
export const importSalesOrders = async (importData) => {
  const url = `${baseURL}/import`

  try {
    const { data } = await axiosConfig.post(url, { ...importData })
    return data
  } catch (e) {
    return Promise.reject(e)
  }
}

// Sales Order Fulfillment
export const fetchSalesOrderFulfillment = async (id) => {
  const url = `/sales-order-fulfillments/${id}`

  try {
    const { data } = await axiosConfig.get(url)
    return data
  } catch (e) {
    return Promise.reject(e)
  }
}

// Sales Order Fulfillment
export const fetchShipStationSalesOrderFulfillment = async (from, to) => {
  const url = `/sales-order-fulfillments/shipstation?from=${from}&to=${to}`

  try {
    const { data } = await axiosConfig.get(url)
    return data
  } catch (e) {
    return Promise.reject(e)
  }
}

export const fetchSalesOrderFulfillmentsByType = async (type, from, to) => {
  const url = `/sales-order-fulfillments/${type}?from=${from}&to=${to}`

  try {
    const { data } = await axiosConfig.get(url)
    return data
  } catch (e) {
    return Promise.reject(e)
  }
}

export const editSalesOrderFulfillment = async (id, sendData) => {
  const url = `/sales-order-fulfillments/${id}`

  try {
    const { data } = await axiosConfig.put(url, sendData)
    return data
  } catch (e) {
    return Promise.reject(e)
  }
}

export const editSalesOrderFbaFulfillment = async (id, sendData) => {
  const url = `/sales-order-fulfillments/fba/${id}`

  try {
    const { data } = await axiosConfig.put(url, sendData)
    return data
  } catch (e) {
    return Promise.reject(e)
  }
}

export const fulfillSalesOrder = async (id, sendData) => {
  const url = `${baseURL}/${id}/fulfill`

  try {
    const { data } = await axiosConfig.post(url, sendData)
    return data
  } catch (e) {
    return Promise.reject(e)
  }
}

export const fulfillFbaSalesOrder = async (id, sendData) => {
  const url = `${baseURL}/${id}/fulfill-fba`

  try {
    const { data } = await axiosConfig.post(url, sendData)
    return data
  } catch (e) {
    return Promise.reject(e)
  }
}

export const cancelFbaSalesOrder = async (fulfillmentId) => {
  const url = `/sales-order-fulfillments/fba/${fulfillmentId}/cancel`

  try {
    const { data } = await axiosConfig.post(url)
    return data
  } catch (e) {
    return Promise.reject(e)
  }
}

export const dropshipSalesOrder = async (id, sendData) => {
  const url = `${baseURL}/${id}/dropship-request`

  try {
    const { data } = await axiosConfig.post(url, sendData)
    return data
  } catch (e) {
    return Promise.reject(e)
  }
}

export const archiveOrderByID = async (orderId) => {
  const url = `${baseURL}/${orderId}/archive`

  try {
    const { data } = await axiosConfig.put(url)
    return data
  } catch (e) {
    return Promise.reject(e)
  }
}

export const unarchiveOrderByID = async (orderId) => {
  const url = `${baseURL}/${orderId}/unarchived`

  try {
    const { data } = await axiosConfig.put(url)
    return data
  } catch (e) {
    return Promise.reject(e)
  }
}

export const archiveOrders = async (sendData) => {
  const url = `${baseURL}/archive`

  try {
    const { data } = await axiosConfig.put(url, sendData)
    return data
  } catch (e) {
    return Promise.reject(e)
  }
}

export const unarchiveOrders = async (sendData) => {
  const url = `${baseURL}/unarchive`

  try {
    const { data } = await axiosConfig.put(url, sendData)
    return data
  } catch (e) {
    return Promise.reject(e)
  }
}

export const deleteOrderByID = async (orderID) => {
  const url = `${baseURL}/${orderID}`

  try {
    const { data } = await axiosConfig.delete(url)
    return data
  } catch (e) {
    return Promise.reject(e)
  }
}

export const deleteOrders = async (sendData) => {
  try {
    const { data } = await axiosConfig.delete(baseURL, { params: sendData })
    return data
  } catch (e) {
    return Promise.reject(e)
  }
}

// check sales orders deleatbility
export const isDeletable = async (ids) => {
  try {
    const { data } = await axiosConfig.post(`${baseURL}/deletable`, { ids })
    return data
  } catch (e) {
    return Promise.reject(e)
  }
}

export const settingsSalesOrders = async (sendData) => {
  let url = 'settings/sales-orders'
  try {
    const { data } = await axiosConfig.get(url)
    return data
  } catch (e) {
    return Promise.reject(e)
  }
}

// Save or Update sales orders settings
export const saveOrUpdateSalesOrdersSettings = async (settings) => {
  const url = '/settings/sales-orders'

  try {
    const { data } = await axiosConfig.put(url, { ...settings })
    return data
  } catch (e) {
    return Promise.reject(e)
  }
}

export const getOrderNotes = async (orderId) => {
  try {
    const { data } = await axiosConfig.get(`${baseURL}/${orderId}/notes`)
    return data
  } catch (e) {
    return Promise.reject(e)
  }
}

export const addOrderNote = async (orderId, payload) => {
  try {
    const { data } = await axiosConfig.post(`${baseURL}/${orderId}/notes`, { ...payload })
    return data
  } catch (e) {
    return Promise.reject(e)
  }
}

export const deleteNote = async (orderId, noteId) => {
  try {
    const { data } = await axiosConfig.delete(`${baseURL}/${orderId}/notes/${noteId}`)
    return data
  } catch (e) {
    return Promise.reject(e)
  }
}

export const addPayment = async (orderId, payload) => {
  try {
    const { data } = await axiosConfig.post(`${baseURL}/${orderId}/payments`, { ...payload })
    return data
  } catch (e) {
    return Promise.reject(e)
  }
}

export const getPayments = async (orderId) => {
  try {
    const { data } = await axiosConfig.get(`${baseURL}/${orderId}/payments`)
    return data
  } catch (e) {
    return Promise.reject(e)
  }
}

export const deletePayment = async (salesOrderId, paymentId) => {
  try {
    const { data } = await axiosConfig.delete(`${baseURL}/${salesOrderId}/payments/${paymentId}`)
    return data
  } catch (e) {
    return Promise.reject(e)
  }
}

export const updatePayment = async (salesOrderId, paymentId, payload) => {
  try {
    const { data } = await axiosConfig.put(`${baseURL}/${salesOrderId}/payments/${paymentId}`, payload)
    return data
  } catch (e) {
    return Promise.reject(e)
  }
}

export const deleteFulfillment = async (fulfillmentId, params) => {
  try {
    const { data } = await axiosConfig.delete(`sales-order-fulfillments/${fulfillmentId}`, {params})
    return data
  } catch (e) {
    return Promise.reject(e)
  }
}

export const exchangeSalesOrder = async (orderId, payload) => {
  try {
    const { data } = await axiosConfig.post(`${baseURL}/${orderId}/exchange`, payload)
    return data
  } catch (e) {
    return Promise.reject(e)
  }
}

export const fetchSalesOrderDocumentById = async (id) => {
  const url = `${baseURL}/${id}/document`
  try {
    const { data } = await axiosConfig.get(url)
    return data
  } catch (e) {
    return Promise.reject(e)
  }
}

export const fetchOrderShipstationFulfillments = async (orderId) => {
  const url = `${baseURL}/${orderId}/shipstation-order`
  try {
    const { data } = await axiosConfig.get(url)
    return data
  } catch (e) {
    return Promise.reject(e)
  }
}

export const fetchOrderShipstationFulfillment = async (fulfillmentId) => {
  const url = `${baseURL}/${fulfillmentId}/shipstation-fulfillment`
  try {
    const { data } = await axiosConfig.get(url)
    return data
  } catch (e) {
    return Promise.reject(e)
  }
}
export const fetchOrderStarshipitFulfillment = async (fulfillmentId) => {
  const url = `starshipit/view-order/${fulfillmentId}`
  try {
    const { data } = await axiosConfig.get(url)
    return data
  } catch (e) {
    return Promise.reject(e)
  }
}
export const fetchSalesOrderFinancials = async (orderId) => {
  const url = `${baseURL}/${orderId}/financials`
  try {
    const { data } = await axiosConfig.get(url)
    return data
  } catch (e) {
    return Promise.reject(e)
  }
}

export const fetchSalesOrderFbaInventory = async (orderId) => {
  const url = `${baseURL}/${orderId}/fba-inventory-movements`
  try {
    const { data } = await axiosConfig.get(url)
    return data
  } catch (e) {
    return Promise.reject(e)
  }
}

export const fetchSalesOrderFbaShipments = async (orderId) => {
  const url = `${baseURL}/${orderId}/fba-shipments`
  try {
    const { data } = await axiosConfig.get(url)
    return data
  } catch (e) {
    return Promise.reject(e)
  }
}

export const previewSalesOrdersImport = async (sendData) => {
  const url = `${baseURL}/import/preview`
  try {
    const { data } = await axiosConfig.post(url, sendData)
    return data
  } catch (e) {
    return Promise.reject(e)
  }
}

export const importSalesOrdersData = async (sendData) => {
  const url = `${baseURL}/import`
  try {
    const { data } = await axiosConfig.post(url, sendData)
    return data
  } catch (e) {
    return Promise.reject(e)
  }
}

export const fulfillSalesOrders = async (sendData) => {
  const url = `${baseURL}/fulfill`
  try {
    const { data } = await axiosConfig.post(url, sendData)
    return data
  } catch (e) {
    return Promise.reject(e)
  }
}

export const syncOrderWithSalesChannel = async (integrationInstanceId, orderId) => {
  try {
    const {data} = await axiosConfig.post(`/integration-instances/${integrationInstanceId}/orders/${orderId}/refresh`)
    return data
  } catch (e) {
    return Promise.reject(e)
  }
}

export const approveSalesOrders = async (payload) => {
  try {
    const { data } = await axiosConfig.post(`${baseURL}/approve`, payload)
    return data
  } catch (e) {
    return Promise.reject(e)
  }
}

export const fetchOrdersWithUnallocatedItem = async (sku) => await getAxiosRequest(`${baseURL}/orders-have-unallocated-item/${sku}`)

export const compareFulfillmentsWithSalesChannel = async (orderId) => await getAxiosRequest(getUrl(`${orderId}/compare-fulfillments-with-sales-channel`))
