import axiosConfig from '../config/axiosConfig'
import buildUrlParams from '@/support/buildUrlParams'

const baseURL = '/categories'

// Get category detail
export const getCategory = async (categoryId) => {
  try {
    return await axiosConfig.get(`${baseURL}/${categoryId}`)
  } catch (e) {
    return Promise.reject(e)
  }
}

// Gets the product categories tree
export const getCategoriesTree = async () => {
  const url = `${baseURL}/tree`

  try {
    const { data } = await axiosConfig.get(url)
    return data.data
  } catch (e) {
    return Promise.reject(e)
  }
}

// Assigns a category to the product
export const assignCategoryToProduct = async (productId, categoryId, isPrimary = 1) => {
  const url = `${baseURL}/assign-category-to-product`
  try {
    return await axiosConfig.post(url, {
      category_id: categoryId,
      product_id: productId,
      is_primary: isPrimary
    })
  } catch (e) {
    return Promise.reject(e)
  }
}

export const updateCategory = async (categoryId, name, parentId, attributes) => {
  try {
    const url = `${baseURL}/${categoryId}`
    return await axiosConfig.put(url, {
      name: name,
      parent_id: parentId,
      attributes: attributes.direct,
      attribute_groups: attributes.grouped
    })
  } catch (e) {
    return Promise.reject(e)
  }
}

export const deleteCategory = async (categoryId) => {
  try {
    const url = `${baseURL}/${categoryId}`
    return await axiosConfig.delete(url)
  } catch (e) {
    return Promise.reject(e)
  }
}

export const archiveCategory = async (categoryId) => {
  try {
    const url = `${baseURL}/${categoryId}/archive`
    return await axiosConfig.put(url)
  } catch (e) {
    return Promise.reject(e)
  }
}

export const archiveBulkCategories = async (payload) => {
  try {
    const url = `${baseURL}/archive`
    return await axiosConfig.put(url, {
      ids: payload.ids
    })
  } catch (e) {
    return Promise.reject(e)
  }
}

export const unarchiveCategory = async (categoryId) => {
  try {
    const url = `${baseURL}/${categoryId}/unarchived`
    return await axiosConfig.put(url)
  } catch (e) {
    return Promise.reject(e)
  }
}

export const unarchiveBulkCategories = async (payload) => {
  try {
    const url = `${baseURL}/unarchive`
    return await  axiosConfig.put(url, {
      ids: payload.ids
    })
  } catch (e) {
    return Promise.reject(e)
  }
}

/**
 * Deletes a bulk of categories
 * @param payload
 * @returns {Promise<any[]>}
 */
export const deleteBulkCategories = async (payload) => {
  try {
    const url = `${baseURL}`
    return await axiosConfig.delete(url, { data: payload })
  } catch (e) {
    return Promise.reject(e)
  }
}

export const createCategory = async (name, parentId, attributeIds = []) => {
  try {
    const url = `${baseURL}`
    return await axiosConfig.post(url, {
      name: name,
      parent_id: parentId,
      attributes: attributeIds
    })
  } catch (e) {
    return Promise.reject(e)
  }
}

export const getProductsCount = async (categoryId) => {
  const url = `${baseURL}/${categoryId}`
  try {
    const { data } = await axiosConfig.get(url)
    return data.data
  } catch (e) {
    return Promise.reject(e)
  }
}

// Get Product Brands
export const getAllProductCategories = async (options = {}) => {
  try {
    const { data } = await axiosConfig.get(baseURL, buildUrlParams(options))
    return data
  } catch (e) {
    return Promise.reject(e)
  }
}

/**
 * Decides which of the provided category ids are deletable
 * @returns {array}
 */
export const canDeleteCategories = async (ids) => {
  const url = `${baseURL}/deletable`
  return (await axiosConfig.post(url, {
    ids: ids
  })).data
}

export const previewCategoriesImport = async (sendData) => {
  const url = `${baseURL}/import/preview`
  try {
    const { data } = await axiosConfig.post(url, sendData)
    return data
  } catch (e) {
    return Promise.reject(e)
  }
}

export const importCategoriesData = async (sendData) => {
  const url = `${baseURL}/import`
  try {
    const { data } = await axiosConfig.post(url, sendData)
    return data
  } catch (e) {
    return Promise.reject(e)
  }
}
