import axiosConfig from '@/config/axiosConfig'
import buildUrlParams from '@/support/buildUrlParams'
import { isFailedRequest } from '@/support/integrations/integrationInstanceHelpers'

const baseURL = '/nominal-codes'

// Get All Nominal Codes
export const getAllNominalCodes = async (options = {}) => {
  try {
    const { data } = await axiosConfig.get(baseURL, buildUrlParams(options))
    return data
  } catch (e) {
    return Promise.reject(e)
  }
}

// Create Nominal Code
export const createNominalCode = async (nominalCodeData) => {
  try {
    const { data } = await axiosConfig.post(baseURL, { ...nominalCodeData })
    return data
  } catch (e) {

    if (e.response) {
      return e.response
    }
  }
}

// Update Nominal Code
export const updateNominalCode = async (nominalCodeID, nominalCodeData) => {
  const url = `${baseURL}/${nominalCodeID}`

  try {
    const { data } = await axiosConfig.put(url, { ...nominalCodeData })
    return data
  } catch (e) {
    return Promise.reject(e)
  }
}

export const Types = ['Revenue', 'Expense', 'Asset', 'Liability', 'Equity', 'Bank', 'Current', 'Inventory', 'Fixed', 'Directcosts', 'Currliab', 'Termliab', 'Sales']

// Archive Nominal Code By ID
export const archiveNominalCodeByID = async (nominalCodeID) => {
  const url = `${baseURL}/${nominalCodeID}/archive`

  try {
    const { data } = await axiosConfig.put(url)
    return data
  } catch (e) {
    return Promise.reject(e)
  }
}

// UnArchive Nominal Code By ID
export const unarchiveNominalCodeByID = async (nominalCodeID) => {
  const url = `${baseURL}/${nominalCodeID}/unarchived`

  try {
    const { data } = await axiosConfig.put(url)
    return data
  } catch (e) {
    return Promise.reject(e)
  }
}

// Delete Nominal Code By ID
export const deleteNominalCodeByID = async (nominalCodeID) => {
  const url = `${baseURL}/${nominalCodeID}`

  try {
    const { data } = await axiosConfig.delete(url)
    return data
  } catch (e) {
    return Promise.reject(e)
  }
}

// Get Mapping Nominal Codes
export const mappingNominalCodes = async (fetchByPaymentTypes = false) => {
  const url = '/settings/mapping-nominal-codes'

  try {
    const { data } = await axiosConfig.get(url, { params: { fetchByPaymentTypes } })
    return data
  } catch (e) {
    return Promise.reject(e)
  }
}

// Get Mapping Nominal Codes From Xero
export const mappingNominalCodesFromXero = async () => {
  try {
    const { data } = await axiosConfig.post('/xero/accounts/download')
    if (isFailedRequest(data)) return Promise.reject({response:{data}})
    return data
  } catch (e) {
    return Promise.reject(e)
  }
}

// Update nominal codes mapping
export const updateNominalCodes = async (mappings) => {
  const url = '/settings/mapping-nominal-codes'

  try {
    const { data } = await axiosConfig.put(url, { ...mappings })
    return data
  } catch (e) {
    return Promise.reject(e)
  }
}

// Archive Nominal Codes By IDs
export const archiveNominalCodes = async (sendData) => {
  const url = `${baseURL}/archive`

  try {
    const { data } = await axiosConfig.put(url, sendData)
    return data
  } catch (e) {
    return Promise.reject(e)
  }
}

// unArchive Nominal Codes By IDs
export const unArchiveNominalCodes = async (sendData) => {
  const url = `${baseURL}/unarchive`

  try {
    const { data } = await axiosConfig.put(url, sendData)
    return data
  } catch (e) {
    return Promise.reject(e)
  }
}

// delete Nominal Codes By IDs
export const deleteNominalCodes = async (sendData) => {
  const url = `${baseURL}`

  try {
    const { data } = await axiosConfig.delete(url, { data: sendData })
    return data
  } catch (e) {
    return Promise.reject(e)
  }
}

// check nominal codes deleatbility
export const isDeletable = async (ids) => {
  try {
    const { data } = await axiosConfig.post(`${baseURL}/deletable`, { ids })
    return data
  } catch (e) {
    return Promise.reject(e)
  }
}

// check nominal codes archivablity
export const isArchivable = async (ids) => {
  try {
    const { data } = await axiosConfig.post(`${baseURL}/archivable`, { ids })
    return data
  } catch (e) {
    return Promise.reject(e)
  }
}

export const previewNominalCodesImport = async (sendData) => {
  const url = `${baseURL}/import/preview`
  try {
    const { data } = await axiosConfig.post(url, sendData)
    return data
  } catch (e) {
    return Promise.reject(e)
  }
}

export const importNominalCodesData = async (sendData) => {
  const url = `${baseURL}/import`
  try {
    const { data } = await axiosConfig.post(url, sendData)
    return data
  } catch (e) {
    return Promise.reject(e)
  }
}
