// Imports
import Vue from 'vue'
import {
  defaultView,
  filterOptions,
  dateFilterOptions,
  fulfillableSecondaryOptions,
  dateFilterWithinOptions,
  backorderedOptions
} from '../assets/config'
import { reject, isEqual, filter } from 'lodash'
import uniqueArray from '@/support/uniqueArray'

export const SKUTableModule = {
  namespaced: true,
  state: () => ({
    tableData: [],
    tableHeaders: [],
    tableViews: [defaultView],
    defaultHeaders: [],
    conjunctionList: [{ id: 1, name: 'And', value: 'and' }, { id: 2, name: 'Or', value: 'or' }],
    searchQuery: '',
    selectedRows: {},
    selectedAllRows: false,
    filteringCriteria: {
      conjunction: 'and',
      filterSet: []
    },
    sortingCriteria: [],
    pageLimit: 10,
    currentPage: 1,
    selectedView: defaultView,
    tableConfigurations: null,
    tableConfigurationsLoading: false,
    tableDensity: 'compact',
    filterComplexFields: null,
    useGroupedColumnsFlag: false,
    simpleCells: [],
    meta: 0,
    lookupOptions: [],
    stringFilterTypes: [],
    filtersWithInitialValues: [],
    useTableViews: true,
    loading: false,
    filterDropdown: false,
    enableDelete: false,
    enableArchive: false,
    disableImport: false,
    showArchivedItems: false,
    enableEdit: false,
    enableSync: false,
    enableDuplicate: false,
    enableCreateStockTake: false,
    reloadBlockers: {
      viewReloadIsBlocked: true,
      filtersReloadIsBlocked: true,
      sortingsReloadIsBlocked: true
    },
    totalsLoading: false,
    enumContext: '',
    drawerData: '',
    customFilters: [],
    rewriteFilters: [],
    customSettingsActions: [],
    disableFutureDate: false,
    dateTimeColumns: [],
    dateColumns: [],
    moneyColumns: [],
    disableExport: false,
    customFilterComplexFields: new Map(),
    enadleFutureDateColumns: [],
  }),
  getters: {
    getTableData: state => state.tableData,
    getTableHeaders: state => {
      if (!state.tableHeaders) return []
      return state.tableHeaders.slice().sort((a, b) => a.order > b.order ? 1 : -1)
    },
    getVisibleHeaders: (state, getters) => {
      let headers = state.tableHeaders.filter(h => h.visible).sort((a, b) => a.order > b.order ? 1 : -1)

      if (state.useGroupedColumnsFlag) {
        headers = reject(headers, (header) => {
          let reject

          if (header.isGroupBase) {
            // is groupBase, but has 0 visible children
            const children = headers.filter(h => h.group === header.group).filter(h => h.value !== header.value)
            let hasNoVisibleChildren = true
            if (children.length) {
              for (let i = 0; i < children.length; i++) {
                const child = children[i]
                if (child.visible) {
                  hasNoVisibleChildren = false
                  break
                }
              }
            }

            reject = hasNoVisibleChildren
          } else {
            // is not groupBase and group base is among visible headers
            let visibleGroupBase = headers.filter(h => h.group === header.group).find(h => h.value === h.groupData)
            reject = Boolean(!header.isGroupBase && visibleGroupBase)
          }

          return reject
        })
      }

      // always put frozen columns first
      if (getters.getFrozenColumnsValue) {
        headers.sort((a, b) => getters.getFrozenColumnsValue.includes(a.value) ? -1 : 1)
      }

      return headers
    },
    headersIncludingEmptyGroupsAndChildren: state => state.tableHeaders.filter(h => h.visible).sort((a, b) => a.order > b.order ? 1 : -1),
    getIncluded: (state, getters) => {
      const columns = getters.headersIncludingEmptyGroupsAndChildren
      if (!columns.length) return []

      const columnsValues = columns.map(column => {
        // if the value has '.', then we send the first part only: 'price.Retail.value' becomes 'price'
        return column.value.split('.')[0]
      })
      const uniqueValues = uniqueArray(columnsValues)

      return uniqueValues
    },
    getTableViews: (state, getters) => {
      if (getters.getHasCustomDefaultView) {
        let viewsClone = JSON.parse(JSON.stringify(state.tableViews))
        const defaultView = viewsClone[0]
        const arrayWithoutDefault = state.tableViews.slice(1)
        arrayWithoutDefault.splice(1, 0, defaultView)
        return arrayWithoutDefault
      } else {
        return state.tableViews
      }
    },
    getDefaultView: (state, getters) => {
      // check sku.io view first
      let wanted = getters.getTableViews.find(v => v.id === -1)
      return wanted || null
    },
    getCustomDefaultView: (state, getters) => {
      let wanted = getters.getTableViews.find(v => v.is_default)
      return wanted || false
    },
    getHasCustomDefaultView: state => !!state.tableViews.find(v => v.is_default),
    getDefaultHeaders: state => state.defaultHeaders.slice().sort((a, b) => a.order > b.order ? 1 : -1),
    getCurrentPage: state => state.currentPage,
    getTableFilters: state => state.filteringCriteria.filterSet,
    getLastFilterIndex: state => state.filteringCriteria.filterSet.length - 1,
    getConjunctionList: state => state.conjunctionList,
    getSelectedConjunction: state => state.filteringCriteria.conjunction,
    getSelectedRows: state => state.selectedRows,
    getSelectedRowsObjects: (state, getters) => {
      let rows = []

      if (Object.keys(getters.getSelectedRows).length) {
        // getSelectedRows has structure {'id': true/false}
        const entries = Object.entries(getters.getSelectedRows)
        // filter items with 'true' values
        const selectedIdsArray = entries.filter(i => i[1]).map(i => i[0])
        // filter table objects, which are among selected ids selected
        rows = getters.getTableData.filter(row => selectedIdsArray.includes(String(row.id)))
      }
      return rows
    },
    getSelectedAllRows: state => state.selectedAllRows,
    selectedRowsLength: state => Object.values(state.selectedRows).filter(row => row).length,
    getSearchQuery: state => state.searchQuery,
    getSortingCriteria: state => state.sortingCriteria,
    getFilteringCriteria: state => state.filteringCriteria,
    getPageLimit: state => state.pageLimit,
    getSelectedView: state => state.selectedView,
    selectedViewChanged: (state, getters) => {
      let filters1 = JSON.parse(JSON.stringify(state.selectedView.content.filters))
      filters1.filterSet.forEach(f => delete f.id)
      let filters2 = JSON.parse(JSON.stringify(state.filteringCriteria))
      filters2.filterSet.forEach(f => delete f.id)

      return !isEqual(filters1, filters2) ||
      !isEqual(state.selectedView.content.sort, state.sortingCriteria) ||
      !isEqual(state.selectedView.content.density, state.tableDensity) ||
      !isEqual(getters.getSelectedView.content.columns, getters.getTableHeaders) ||
      !isEqual(getters.getSelectedView.content.showArchivedItems, getters.getShowArchivedItems) ||
      !!getters.getSelectedView.content.query !== !!getters.getSearchQuery
    },
    getTableCompact: state => state.tableDensity === 'compact',
    getTableDensity: state => state.tableDensity,
    getTableConfigurations: state => state.tableConfigurations,
    getTableConfigurationsLoading: state => state.tableConfigurationsLoading,
    getFrozenColumnsValue: state => (state.tableConfigurations && state.tableConfigurations.frozen_columns) || null,
    getFilterComplexFields: state => state.filterComplexFields,
    getUseGroupColumnsFlag: state => state.useGroupedColumnsFlag,
    getSimpleCells: state => state.simpleCells,
    getMeta: state => state.meta,
    getLookupOptions: state => state.lookupOptions,
    getStringFilterTypes: state => state.stringFilterTypes,
    getFiltersWithInitialValues: state => state.filtersWithInitialValues,
    getUseTableViews: state => state.useTableViews,
    getLoading: state => state.loading,
    getFilterDropdown: state => state.filterDropdown,
    getEnableDelete: state => state.enableDelete,
    getEnableArchive: state => state.enableArchive,
    getEnableEdit: state => state.enableEdit,
    getEnableSync: state => state.enableSync,
    getEnableCreateStockTake: state => state.enableCreateStockTake,
    getShowArchivedItems: state => state.showArchivedItems,
    getViewReloadIsBlocked: state => state.reloadBlockers.viewReloadIsBlocked,
    getFiltersReloadIsBlocked: state => state.reloadBlockers.filtersReloadIsBlocked,
    getSortingsReloadIsBlocked: state => state.reloadBlockers.sortingsReloadIsBlocked,
    getTotalsLoading: state => state.totalsLoading,
    getDrawerData: state => state.drawerData,
    getEnumContext: state => state.enumContext,
    getEnableDuplicate: state => state.enableDuplicate,
    getDisableImport: state => state.disableImport,
    getHeaderType: (state, getters, rootState, rootGetters) => (header) => {
      if (!header) return ''

      if (state.enumContext && rootGetters['EnumsModule/getAll']({ context: state.enumContext, variableName: header.value }).length) {
        return 'stringSelect'
      }

      let stringTypes = ['string', 'longtext', 'image']

      if (state.stringFilterTypes) stringTypes = [...stringTypes, ...state.stringFilterTypes]

      if (stringTypes.includes(header.type)) {
        return 'string'
      } else if (['integer', 'decimal', 'float',].includes(header.type)) {
        return 'number'
      } else if (['boolean', 'checkbox'].includes(header.type)) {
        return 'boolean'
      } else if (['date/time', 'date'].includes(header.type)) {
        return 'date'
      } else if ('dropdown') {
        return header.type
      }
      console.log(`unhandled header type in getHeaderType method: ${header.type}`)
    },

    getFilterOptions: (state, getters) => (header) => {
      if (!header) return []
      // special type for 'backordered' filter
      if (header.value === 'fulfillable') {
        return filterOptions('fulfillableOptions')
      }
      if (header.value === 'custom_backordered') {
        return filterOptions('backorderedOptions')
      }
      // use special type for product.attributes
      if (header.value.substring(0, 11) === 'attributes.') {
        return filterOptions('string-extended')
      }

      return filterOptions(getters.getHeaderType(header))
    },

    getSelectedOption: (state, getters) => ({ option, header }) => {
      const filterOptions = getters.getFilterOptions(header)

      if (!filterOptions) return []
      const selectedOption = filterOptions.find(o => o.value === option)
      return selectedOption
    },

    getSelectedOptionName: (state, getters) => ({ option, header }) => {
      let obj = getters.getSelectedOption({ option, header })
      return (obj && obj.name) || ''
    },

    getSelectedColumn: (state, getters) => (filter) => {
      return [...getters.getDefaultHeaders, ...getters.getCustomFilters].find(header => header.value === filter.headerValue)
    },

    getSelectedDateOption: (state, getters) => (filter) => {
      if (filter.dateOption) {
        const option = dateFilterOptions.find(o => o.value === filter.dateOption) ||
          dateFilterWithinOptions.find(o => o.value === filter.dateOption)
        return option.name
      } else if (filter.operator !== 'isWithin') {
        return dateFilterOptions.find(o => o.value === 'exactDate').name
      }
      return dateFilterWithinOptions.find(o => o.value === 'nextNumberOfDays').name
    },

    getSelectedSecondaryOption: (state, getters) => (filter) => {
      if (filter.secondaryOption) {
        // set other options when needed
        const options = fulfillableSecondaryOptions
        const option = options.find(o => o.value === filter.secondaryOption)

        return option ? option.name : ''
      }
      return ''
    },

    getSelectedDropdownOption: (state, getters) => (filter, options) => {
      if (filter.secondaryOption) {
        // set other options when needed
        const options = fulfillableSecondaryOptions
        const option = options.find(o => o.value === filter.secondaryOption)

        return option ? option.name : ''
      }
      return ''
    },

    getDateOptionObject: (state, getters) => (filter) => {
      if (filter.dateOption) {
        return dateFilterOptions.find(o => o.value === filter.dateOption) ||
          dateFilterWithinOptions.find(o => o.value === filter.dateOption)
      } else if (filter.operator !== 'isWithin') {
        return dateFilterOptions.find(o => o.value === 'exactDate')
      }
      return dateFilterWithinOptions.find(o => o.value === 'nextNumberOfDays')
    },

    getSecondaryOptionObject: (state, getters) => (filter) => {
      if (!filter.secondaryOption) return {}

      // set other options when needed
      const options = fulfillableSecondaryOptions
      return options.find(o => o.value === filter.secondaryOption) || {}
    },

    getSelectedObject: () => (filter) => {
      if (!filter.operator) return {}

      // set other options when needed
      const options = backorderedOptions
      return options.find(o => o.value === filter.operator) || {}
    },

    getOperatorsThatDontNeedValue: state => ['isEmpty', 'isNotEmpty', 'isAssigned', 'isNotAssigned'],

    getCustomFilters: state => state.customFilters,
    getRewriteFilters: state => state.rewriteFilters,
    getCustomSettingsActions: state => state.customSettingsActions,
    getDisableFutureDate: state => state.disableFutureDate,
    getDateTimeColumns: state => state.dateTimeColumns,
    getDateColumns: state => state.dateColumns,
    getMoneyColumns: state => state.moneyColumns,
    getDisableExport: state => state.disableExport,
    getGroupChildren: (_,getters) => (group) => {
      // check among tableHeaders, because visibleHeaders are additionally filtered to exclude individual
      // fields if grouped columns are visible
      return getters.getTableHeaders.filter(h => h.group === group && h.visible && !h.isGroupBase) || []
    },
    getCustomFilterComplexFields: state => state.customFilterComplexFields,
    getEnableFutureDateColumns: state => state.enadleFutureDateColumns,
  },
  mutations: {
    // Set Table Data Mutation
    SET_TABLE_DATA (state, data) {
      state.tableData = data
    },
    // Set Table Headers Mutation
    SET_TABLE_HEADERS (state, headers) {
      state.tableHeaders = headers
    },
    // Set Table Views
    SET_TABLE_VIEWS (state, views) {
      state.tableViews = [defaultView, ...views]
    },
    // Set Default Headers
    SET_DEFAULT_HEADERS (state, views) {
      state.defaultHeaders = [...views]
    },
    // Set Search Query Mutation
    SET_SEARCH_QUERY (state, value) {
      state.searchQuery = value
    },
    // Set Current Page
    SET_CURRENT_PAGE (state, page) {
      state.currentPage = page
    },
    // Select Row
    SET_ROW_SELECTED (state, { rowID, value }) {
      state.selectedRows[rowID] = Vue.set(state.selectedRows, rowID, value)
      state.selectedRows = { ...state.selectedRows }
    },
    SET_SELECTED_RAWS (state, value) {
      state.selectedRows = value
    },
    SET_SELECTED_ALL_ROWS (state, value) {
      state.selectedAllRows = value
    },
    DUPLICATE_ROW (state, row) {
      // const foundRow = state.tableData.find(r => r.id === row.id)
      // const foundRowIndex = state.tableData.indexOf(foundRow)
      // state.tableData.slice(foundRowIndex, 0, row)
      console.log('not implemented')
    },
    DELETE_ROW (state, row) {
      const foundRow = state.tableData.find(r => r.id === row.id)
      const foundRowIndex = state.tableData.indexOf(foundRow)
      state.tableData.splice(foundRowIndex, 1)
    },
    SET_HIDE_COLUMN (state, column) {
      const foundColumn = state.tableHeaders.find(h => h.value === column.value)
      foundColumn.visible = false
    },
    SET_FILTER_CONJUNCTION (state, value) {
      state.filteringCriteria.conjunction = value
    },
    SET_FILTER (state, filter) {
      state.filteringCriteria.filterSet.push(filter)
    },
    SET_SORT (state, sort) {
      state.sortingCriteria.push(sort)
    },
    SET_FILTER_HEADER (state, { header, filterID }) {
      const foundColumn = state.filteringCriteria.filterSet.find(h => h.id === filterID)
      if (state.filterComplexFields && state.filterComplexFields.includes(header.type)) {
        foundColumn.column = `${header.value}.name`
      } else {
        foundColumn.column = header.value
      }
      foundColumn.headerValue = header.value
    },
    REMOVE_FILTER_HEADER (state, id) {
      const foundColumn = state.filteringCriteria.filterSet.find(h => h.id === id)
      const foundIndex = state.filteringCriteria.filterSet.indexOf(foundColumn)

      state.filteringCriteria.filterSet.splice(foundIndex, 1)
    },
    REMOVE_SORT (state, id) {
      const foundColumn = state.sortingCriteria.find(h => h.id === id)
      const foundIndex = state.sortingCriteria.indexOf(foundColumn)

      state.sortingCriteria.splice(foundIndex, 1)
    },
    SET_FILTER_VALUE (state, { value, filterID }) {
      const foundColumn = state.filteringCriteria.filterSet.find(h => h.id === filterID)
      Vue.set(foundColumn, 'value', value)
    },
    SET_FILTER_COLUMN (state, { value, filterID }) {
      const foundColumn = state.filteringCriteria.filterSet.find(h => h.id === filterID)
      Vue.set(foundColumn, 'column', value)
    },
    SET_FILTER_OPTION (state, { option, filterID }) {
      const foundColumn = state.filteringCriteria.filterSet.find(h => h.id === filterID)
      Vue.set(foundColumn, 'operator', option.value)
    },
    SET_FILTER_DATE_OPTION (state, { dateOption, filterID }) {
      const foundColumn = state.filteringCriteria.filterSet.find(h => h.id === filterID)
      Vue.set(foundColumn, 'dateOption', dateOption.value)
    },
    SET_FILTER_SECONDARY_OPTION (state, { secondaryOption, filterID }) {
      let foundColumn = state.filteringCriteria.filterSet.find(h => h.id === filterID)
      Vue.set(foundColumn, 'secondaryOption', secondaryOption.value)
    },
    SET_SORT_HEADER (state, { header, sortID }) {
      const foundColumn = state.sortingCriteria.find(h => h.id === sortID)
      if (state.filterComplexFields && state.filterComplexFields.includes(header.type)) {
        foundColumn.column = `${header.value}.name`
      } else {
        foundColumn.column = header.value
      }
      foundColumn.headerValue = header.value
    },
    SET_SORT_DIRECTION (state, { ascending, sortID }) {
      const foundColumn = state.sortingCriteria.find(h => h.id === sortID)
      foundColumn.ascending = ascending
    },
    SET_FIELD_CONFIG (state, item) {
      const foundConfig = state.fieldsConfig.find(h => h.uuid === item.uuid)
      const foundIndex = state.fieldsConfig.indexOf(foundConfig)

      if (foundConfig) {
        state.fieldsConfig.splice(foundIndex, 1)
      } else {
        state.fieldsConfig.push(item)
      }
    },
    SET_FILTERING_CRITERIA (state, criteria) {
      state.filteringCriteria = criteria
    },
    SET_SORTING_CRITERIA (state, criteria) {
      state.sortingCriteria = criteria
    },
    UPDATE_SORTING_CRITERIA (state) {
      state.sortingCriteria = [...state.sortingCriteria]
    },
    SET_PAGE_LIMIT (state, limit) {
      state.pageLimit = limit
    },
    SET_SELECTED_VIEW (state, view) {
      state.selectedView = view
    },
    SET_SELECTED_VIEW_NAME (state, name) {
      state.selectedView.name = name
    },
    SET_TABLE_CONFIGURATIONS (state, value) {
      state.tableConfigurations = Object.assign({}, value)
    },
    SET_TABLE_CONFIGURATIONS_LOADING (state, value) {
      state.tableConfigurationsLoading = value
    },
    SET_TABLE_DENSITY (state, value) {
      state.tableDensity = value
    },
    SET_FILTER_COMPLEX_FIELDS (state, value) {
      state.filterComplexFields = value
    },
    SET_USE_GROUPED_COLUMNS (state, value) {
      state.useGroupedColumnsFlag = value
    },
    SET_SIMPLE_CELLS (state, value) {
      state.simpleCells = value
    },
    SET_META (state, value) {
      state.meta = value
    },
    SET_LOOKUP_OPTIONS (state, value) {
      state.lookupOptions = value
    },
    SET_STRING_FILTER_TYPES (state, value) {
      state.stringFilterTypes = value
    },
    SET_FILTERS_WITH_INITIAL_VALUES (state, value) {
      state.filtersWithInitialValues = value
    },
    SET_USE_TABLE_VIEWS (state, value) {
      state.useTableViews = value
    },
    SET_LOADING (state, value) {
      state.loading = value
    },
    SET_FILTER_DROPDOWN (state, value) {
      state.filterDropdown = value
    },
    SET_ENABLE_DELETE (state, value) {
      state.enableDelete = value
    },
    SET_ENABLE_ARCHIVE (state, value) {
      state.enableArchive = value
    },
    SET_ENABLE_SYNC (state, value) {
      state.enableSync = value
    },
    SET_ENABLE_EDIT (state, value) {
      state.enableEdit = value
    },
    SET_SHOW_ARCHIVED_ITEMS (state, value) {
      state.showArchivedItems = value
    },

    SET_VIEW_RELOAD_IS_BLOCKED (state, value) {
      Vue.prototype.$set(state.reloadBlockers, 'viewReloadIsBlocked', value)
    },
    SET_FILTERS_RELOAD_IS_BLOCKED (state, value) {
      Vue.prototype.$set(state.reloadBlockers, 'filtersReloadIsBlocked', value)
    },
    SET_SORTINGS_RELOAD_IS_BLOCKED (state, value) {
      Vue.prototype.$set(state.reloadBlockers, 'sortingsReloadIsBlocked', value)
    },
    SET_TOTALS_LOADING (state, value) {
      state.totalsLoading = value
    },
    SET_ENUM_CONTEXT (state, value) {
      state.enumContext = value
    },
    SET_DRAWER_DATA (state, value) {
      state.drawerData = value
    },
    SET_ENABLE_DUPLICATE (state, value) {
      state.enableDuplicate = value
    },
    SET_DISABLE_IMPORT (state, value) {
      state.disableImport = value
    },
    CLEAR_FILTERS (state) {
      if (state.searchQuery) state.searchQuery = ''
      state.filteringCriteria = {
        conjunction: 'and',
        filterSet: []
      }
    },
    RESET_STORE (state) {
      state.tableData = []
      state.tableHeaders = []
      state.tableViews = [defaultView]
      state.defaultHeaders = []
      state.conjunctionList = [{ id: 1, name: 'And', value: 'and' }, { id: 2, name: 'Or', value: 'or' }]
      state.searchQuery = ''
      state.selectedRows = {}
      state.selectedAllRows = false
      state.filteringCriteria = {
        conjunction: 'and',
        filterSet: []
      }
      state.sortingCriteria = []
      state.pageLimit = 10
      state.currentPage = 1
      state.selectedView = defaultView
      state.tableConfigurations = null
      state.tableDensity = 'compact'
      state.filterComplexFields = null
      state.useGroupedColumnsFlag = false
      state.simpleCells = []
      state.meta = 0
      state.stringFilterTypes = []
      state.filtersWithInitialValues = []
      state.useTableViews = true
      state.loading = false
      state.filterDropdown = false
      state.enableDelete = false
      state.enableArchive = false
      state.showArchivedItems = false
      state.reloadBlockers = {
        viewReloadIsBlocked: false,
        filtersReloadIsBlocked: false,
        sortingsReloadIsBlocked: false
      }
      state.totalsLoading = false
    },
    SET_CUSTOM_FILTERS (state, value) {
      state.customFilters = value
    },
    SET_REWRITE_FILTERS (state, value) {
      state.rewriteFilters = value
    },
    SET_ENABLE_CREATE_STOCK_TAKES (state, value) {
      state.enableCreateStockTake = value
    },
    SET_DISABLE_CREATE_STOCK_TAKES (state, value) {
      state.disableCreateStockTake = value
    },
    SET_DISABLE_FUTURE_DATE (state, value) {
      state.disableFutureDate = value
    },
    SET_FILTERS (state, value) {
      Vue.prototype.$set(state.filteringCriteria, 'filterSet', value)
    },
    SET_CUSTOM_SETTINGS_ACTIONS (state, value) {
      state.customSettingsActions = value
    },
    SET_ENABLE_CREATE_STOCK_TAKES (state, value) {
      state.enableCreateStockTake = value
    },
    SET_DATE_TIME_COLUMNS (state, value) {
      state.dateTimeColumns = value
    },
    SET_DATE_COLUMNS (state, value) {
      state.dateColumns = value
    },
    SET_MONEY_COLUMNS (state, value) {
      state.moneyColumns = value
    },
    SET_DISABLE_EXPORT (state, value) {
      state.disableExport = value
    },
    SET_CUSTOM_FILTER_COMPLEX_FIELDS (state, value) {
      state.customFilterComplexFields = value
    },
    SET_ENABLE_FUTURE_DATE_COLUMNS (state, value) {
      state.enadleFutureDateColumns = value
    },
  },
  actions: {
    // Set Table Data
    setTableData ({ commit }, data) {
      commit('SET_TABLE_DATA', data)
    },
    // Set Table Headers
    setTableHeaders ({ commit }, headers) {
      commit('SET_TABLE_HEADERS', headers)
    },
    // Set Search Query Action
    setSearchQuery ({ commit }, value) {
      commit('SET_SEARCH_QUERY', value)
    },
    // Set Current Page Actions
    setCurrentPage ({ commit }, page) {
      commit('SET_CURRENT_PAGE', page)
    },
    duplicateRow ({ commit }, row) {
      commit('DUPLICATE_ROW', row)
    },
    deleteRow ({ commit }, row) {
      commit('DELETE_ROW', row)
    },
    setHideColumn ({ commit }, column) {
      commit('SET_HIDE_COLUMN', column)
    },
    setFilterConjunction ({ commit }, value) {
      commit('SET_FILTER_CONJUNCTION', value)
    },
    setFilter ({ commit }, filter) {
      commit('SET_FILTER', filter)
    },
    setFilterHeader ({ commit }, { header, filterID }) {
      commit('SET_FILTER_HEADER', { header, filterID })
    },
    removeFilterHeader ({ commit }, id) {
      // reset lookup options, because all filters share the same state, and not doing so would cause a bug
      commit('SET_LOOKUP_OPTIONS', [])
      commit('REMOVE_FILTER_HEADER', id)
    },
    removeFiltersWithoutValue ({ commit, getters }) {
      // check if any filter has no value and remove such filters
      getters.getTableFilters.forEach(f => {
        if (getters.getOperatorsThatDontNeedValue.includes(f.operator)) return Promise.resolve()
        if (f.value === '' || f.value === null || f.value === undefined) {
          commit('SET_LOOKUP_OPTIONS', [])
          commit('REMOVE_FILTER_HEADER', f.id)
        }
      })
      return Promise.resolve()
    },
    clearFiltersAfterSelected ({_, getters, dispatch}, {id}) {
      if (!id) return Promise.resolve()

      const filters = JSON.parse(JSON.stringify(getters.getTableFilters))
      const index = filters.findIndex(f => f.id === id)

      return dispatch('removeFilters', filters.slice(index + 1))
    },
    clearFilters ({_, getters, dispatch}) {
      return dispatch('removeFilters', JSON.parse(JSON.stringify(getters.getTableFilters)))
    },
    clearAll ({commit, getters, dispatch}) {
      commit('SET_SEARCH_QUERY', '')
      return dispatch('removeFilters', JSON.parse(JSON.stringify(getters.getTableFilters)))
    },
    setFilterValue ({ commit }, { value, filterID }) {
      commit('SET_FILTER_VALUE', { value, filterID })
    },
    setFilterColumn ({ commit }, { value, filterID }) {
      commit('SET_FILTER_COLUMN', { value, filterID })
    },
    setCustomFilter ({ commit }, value) {
      commit('SET_CUSTOM_FILTER', value)
    },
    setFilterOption ({ commit }, { option, filterID }) {
      commit('SET_FILTER_OPTION', { option, filterID })
    },
    setFilterDateOption ({ commit }, { dateOption, filterID }) {
      commit('SET_FILTER_DATE_OPTION', { dateOption, filterID })
    },
    setFilterSecondaryOption ({ commit }, { secondaryOption, filterID }) {
      commit('SET_FILTER_SECONDARY_OPTION', { secondaryOption, filterID })
    },
    setSort ({ commit }, sort) {
      commit('SET_SORT', sort)
    },
    setSortHeader ({ commit }, { header, sortID }) {
      commit('SET_SORT_HEADER', { header, sortID })
    },
    setSortDirection ({ commit }, { ascending, sortID }) {
      commit('SET_SORT_DIRECTION', { ascending, sortID })
    },
    removeSort ({ commit }, sortID) {
      commit('REMOVE_SORT', sortID)
    },
    setPageLimit ({ commit }, limit) {
      commit('SET_PAGE_LIMIT', limit)
    },
    setFilteringCriteria ({ commit }, criteria) {
      commit('SET_FILTERING_CRITERIA', criteria)
    },
    setSortingCriteria ({ commit }, criteria) {
      commit('SET_SORTING_CRITERIA', criteria)
    },
    updateSortingCriteria ({ commit }) {
      commit('UPDATE_SORTING_CRITERIA')
    },
    // Set Table Views
    setTableViews ({ commit }, views) {
      commit('SET_TABLE_VIEWS', views)
    },
    // Set Selected View Action
    setSelectedView ({ commit, getters }, view) {
      commit('SET_VIEW_RELOAD_IS_BLOCKED', false)
      commit('SET_SELECTED_VIEW', view)

      // we don't want to link view-object saved as selected view with view.object saved for filter, sort, columns,
      // because we want to reset all previous local changes to table-view when we switch between views back and force
      view = JSON.parse(JSON.stringify(view))

      if (view.value && view.value === 'default') {
        view.content.columns = getters.getDefaultHeaders
      }

      if (view.content.columns) {
        commit('SET_TABLE_HEADERS', view.content.columns)
      }

      if (view.content.filters) {
        commit('SET_FILTERS_RELOAD_IS_BLOCKED', true)
        commit('SET_FILTERING_CRITERIA', view.content.filters)
      }

      if (view.content.sort) {
        commit('SET_SORTINGS_RELOAD_IS_BLOCKED', true)
        commit('SET_SORTING_CRITERIA', (view.content.sort))
      }

      if (view.content.density) {
        commit('SET_TABLE_DENSITY', view.content.density)
      }

      if (Object.keys(view.content).includes('showArchivedItems')) {
        commit('SET_SHOW_ARCHIVED_ITEMS', view.content.showArchivedItems)
      }

      if (view.content.query) {
        commit('SET_SEARCH_QUERY', view.content.query)
      } else {
        commit('SET_SEARCH_QUERY', '')
      }
    },
    resetTableViewToDefault ({ commit }) {

    },
    setSelectedViewName ({ commit }, name) {
      commit('SET_SELECTED_VIEW_NAME', name)
    },
    setTableDensity ({ commit }, value) {
      commit('SET_TABLE_DENSITY', value)
    },
    setFilterComplexFields ({ commit }, value) {
      commit('SET_FILTER_COMPLEX_FIELDS', value)
    },
    setUseGroupedColumns ({ commit }, value) {
      commit('SET_USE_GROUPED_COLUMNS', value)
    },
    setSimpleCells ({ commit }, value) {
      commit('SET_SIMPLE_CELLS', value)
    },
    removeFilters ({commit},filters) {
      filters.forEach(filter => {
        commit('SET_LOOKUP_OPTIONS', [])
        commit('REMOVE_FILTER_HEADER', filter.id)
      })
      commit('SET_FILTERS_RELOAD_IS_BLOCKED', false)
      return Promise.resolve()
    },
    setCustomFilterComplexFields ({ commit }, value) {
      commit('SET_CUSTOM_FILTER_COMPLEX_FIELDS', value)
    },
  }
}
