import formatDigit from '../support/formatDigit'
import { toArray, cloneDeep } from 'lodash'

export default {

  methods: {
    /**
     * Prepares product with minimum initializations
     * @param product
     * @returns {*}
     */
    prepareProduct (product) {
      // add basic data to open the drawer asap
      if (product.suppliers === undefined) { product.suppliers = [] }

      if (product.attributes_grouped === undefined) {
        product.attributes_grouped = {
          direct: [],
          grouped: []
        }
      }

      // Setup images
      if (!product.images) product.images = []

      // Set up the product categories
      if (product.categories === undefined) { product.categories = [] }
      product = this.setupCategories(product)

      // We re-format the digits of numerical properties of the product
      product.length = product.length !== undefined ? formatDigit(product.length) : undefined
      product.weight = product.weight !== undefined ? formatDigit(product.weight) : undefined
      product.width = product.width !== undefined ? formatDigit(product.width) : undefined
      product.height = product.height !== undefined ? formatDigit(product.height) : undefined
      product.case_width = product.case_width !== undefined ? formatDigit(product.case_width) : undefined
      product.case_length = product.case_length !== undefined ? formatDigit(product.case_length) : undefined
      product.case_height = product.case_height !== undefined ? formatDigit(product.case_height) : undefined
      product.case_weight = product.case_weight !== undefined ? formatDigit(product.case_weight) : undefined

      if (product.barcode === null) product.barcode = ''

      if (product.price !== null && product.price !== undefined) {
        const prices = toArray(product.price).filter(price => price.is_default)
        prices.push(...toArray(product.price).filter(price => !price.is_default))
        const priceObjs = {}
        prices.forEach(price => {
          price.value = formatDigit(Number(price.value))
          priceObjs[price.name] = price
        })

        product.price = priceObjs
        for (let [key, value] of Object.entries(product.price)) {
          product.price[key]['value'] = formatDigit(Number(value.value))
        }
      }
      if (product?.type === 'kit' && !product.hasOwnProperty('components')) product.components = [] 

      product.tags = product.tags || []
      
      return product
    },

    /**
     * Prepares the main and additional categories
     * for the product.
     */
    setupCategories (product) {
      // If the product already has a category, we set the path for it,
      // otherwise, we show the category tree and allow the user to make
      // a choice.

      // We setup the categories on the product and in the store
      const categories = []
      // Add main category
      if (product.category_main !== null && product.category_main !== undefined) {
        const main = product.category_main

        // Since the parent id isn't available on the main category,
        // the strategy is to find it in the main category path.
        const mainPath = toArray(product.category_main_path)
        const matchInPath = mainPath.find(current => current.id === main.id)

        categories.push({
          id: main.id,
          parent_id: matchInPath ? matchInPath.parent_id : null,
          name: main.name,
          is_primary: true
        })
      }

      // Additional categories
      if (product.category_others) {
        const additional = product.category_others.map(category => {
          return {
            id: category.id,
            name: category.name,
            parent_id: category.parent_id,
            is_primary: false
          }
        })

        categories.push(...additional)
      }

      // Update categories in store.
      // this.SET_PRODUCT_CATEGORIES(cloneDeep(categories))

      // Update local product
      product.categories = cloneDeep(categories)

      return product
    },

    // It iteratively builds category path for the provided category
    // using the category set by recursively mapping parents
    buildCategoryParentPath (category, categorySet) {
      // Initialize with the given category
      const path = [category]

      // Find the parent of the category
      const matched = categorySet.filter((element) => {
        return element.id === category.parent_id
      })

      // We fetch parent if category is not root
      if (category.parent_id !== null && matched.length > 0) {
        path.unshift(...this.buildCategoryParentPath(matched[0], categorySet))
      }

      return path
    }
  }

}
