export default {
  methods: {
    /**
     * Aggregates the analytics data based on reporting date
     *
     */
    aggregateChartData (data, from, end) {
      if (!data?.length || !from || !end) return []
      // We aggregate based on the dates we have
      const aggregated = []

      const reference = this.moment(from)
      const endDate = this.moment(end)

      while (reference.isSameOrBefore(endDate)) {
        const date = reference.format('YYYY-MM-DD')
        const forDate = data.filter(row => this.moment(row.order_date).format('YYYY-MM-DD') === date)
        let revenue = 0
        let cost_order = 0
        let quantity_sold = 0
        forDate.forEach(row => {
          revenue += parseFloat(row.total_revenue || 0)
          cost_order += parseFloat(row.total_cost || 0)
          quantity_sold += parseFloat(row.quantity_sold || 0)
        })
        // We add the aggregated data
        aggregated.push({
          reporting_date: date,
          revenue,
          cost_order,
          quantity_sold
        })

        reference.add(1, 'days')
      }

      return aggregated
    },
    aggregatLabelsForPeriod ({ period = 'day', from, end }) {
      // this method looks at 'from' date and adds time-periods until we reach the 'end' date
      // each iteration pushes a 'label' to the array of labels

      let format = this.getFormat(period)
      let timeFrames = this.getTimeFrames({ period, from, end })
      from = timeFrames.from
      end = timeFrames.end

      let labels = []

      const reference = this.moment(from)
      const endDate = this.moment(end)

      // add label for each time-frame
      while (reference.isSameOrBefore(endDate)) {
        const date = reference.format(format)
        labels.push(date)
        reference.add(1, period + 's')
      }

      return labels
    },
    aggregateDataByPeriod ({ data, from, end, period = 'day', dateField = 'date', valueField = 'value' }) {
      // this method looks at 'from' date and adds time-periods until we reach the 'end' date
      // each iteration checks if such date has data to show, if no then it adds mock data with 0 value

      let format = this.getFormat(period)
      let timeFrames = this.getTimeFrames({ period, from, end })
      from = timeFrames.from
      end = timeFrames.end

      // we generate mock items for each date with no value, because each date must be represented even if its value = 0

      // expected data - array of objects like
      // {
      //   [dateField]
      //   id
      //   [valueField]
      //   group:{id, name}
      // }

      // We aggregate based on the dates we have within the from-end period
      const aggregated = []

      const reference = this.moment(from)
      const endDate = this.moment(end)

      // go through each time period one by one and check if our data has info for it
      while (reference.isSameOrBefore(endDate)) {
        // index date
        const date = reference.format(format)

        // if such date exists in the list then use it
        // else add a mock item with 0 value

        let wantedField
        switch (period) {
          case 'week':
          case 'month':
          case 'quarter':
          case 'year':
            wantedField = data.find(i => this.moment(i[dateField]).startOf(period).format(format) === date)
            break
          default:
            wantedField = data.find(i => this.moment(i[dateField]).format(format) === date)
            break
        }

        if (wantedField) {
          // existing item
          aggregated.push(wantedField)
        } else {
          // mock item
          aggregated.push({
            [dateField]: date,
            [valueField]: 0,
            group: {
              id: data[0]?.group?.id || '',
              name: data[0]?.group?.name || '',
              sku: date[0]?.group?.sku || ''
            }
          })
        }

        reference.add(1, period + 's')
      }

      return aggregated
    },
    getFormat (period) {
      let format
      switch (period) {
        case 'day':
        case 'week':
          format = 'YYYY-MM-DD'
          break
        case 'month':
          format = 'MMMM YYYY'
          break
        case 'quarter':
          format = 'Qo YYYY'
          break
        case 'year':
          format = 'YYYY'
          break
      }
      return format
    },
    getTimeFrames ({ period, from, end }) {
      let fromClone
      let endCLone
      // we use 'startOf' a given period, because user selects a specific date like 2020-12-12 but we need to
      // relate to the beginning of the month/week/quorter instead
      switch (period) {
        case 'week':
        case 'month':
        case 'quarter':
        case 'year':
          fromClone = this.moment(from).clone().startOf(period)
          endCLone = this.moment(end).clone().startOf(period)
          break
        default:
          fromClone = this.moment(from).clone()
          endCLone = this.moment(end).clone()
      }
      return { from: fromClone, end: endCLone }
    },
  }
}
