<?php

use App\Models\SalesOrderLine;
use Illuminate\Database\Migrations\Migration;
use Illuminate\Database\Schema\Blueprint;
use Illuminate\Support\Facades\Schema;

return new class extends Migration
{

    private string $insertTriggerName = 'before_insert_sales_order_lines';
    private string $updateTriggerName = 'before_update_sales_order_lines';


    /**
     * Run the migrations.
     */
    public function up(): void
    {
        // Add hash field with new structure.
        $this->beforeMigration();

        Schema::table('sales_order_lines', function (Blueprint $table) {
            $table->string('hash', 40)->after('subtotal_cache')->nullable();
        });

        // handle records
        $this->hashSalesOrderLineData();

        // Add unique index to hash field.
        Schema::table('sales_order_lines', function(Blueprint $table){
            $table->unique('hash');
        });

        // Add data triggers to automatically add hash
        $this->registerDataTriggers();

    }

    /**
     * Reverse the migrations.
     */
    public function down(): void
    {
        // Remove triggers
        DB::unprepared("DROP TRIGGER IF EXISTS $this->insertTriggerName");
        DB::unprepared("DROP TRIGGER IF EXISTS $this->updateTriggerName");

        // Drop hash column
        if(Schema::hasColumn('sales_order_lines', 'hash')){
            Schema::dropColumns('sales_order_lines', 'hash');
        }
    }


    private function beforeMigration(): void{
        if(Schema::hasColumn('sales_order_lines', 'hash')){
            Schema::dropColumns('sales_order_lines', 'hash');
        }
    }


    private function prepareFields(string $prefix = ''): string{
        return "COALESCE($prefix".implode(",''), COALESCE($prefix", SalesOrderLine::getUniqueFields()).",'')";
    }

    private function registerDataTriggers(): void{
        // Creation triggers

        $fields = $this->prepareFields('NEW.');

        DB::unprepared("
            CREATE TRIGGER $this->insertTriggerName
            BEFORE INSERT ON sales_order_lines
            FOR EACH ROW
            BEGIN
                SET NEW.hash = SHA1(CONCAT_WS('|', $fields));
            END;
        ");

        // Update triggers
        DB::unprepared("
            CREATE TRIGGER $this->updateTriggerName
            BEFORE UPDATE ON sales_order_lines
            FOR EACH ROW
            BEGIN
                SET NEW.hash = SHA1(CONCAT_WS('|', $fields));
            END;
        ");
    }


    private function hashSalesOrderLineData(): void{
        $fields = $this->prepareFields();
        DB::unprepared("
            UPDATE sales_order_lines set `hash` = SHA1(CONCAT_WS('|', $fields)) WHERE id > 0 
        ");
    }

};
