<?php

namespace Database\Factories;

use App\Helpers;
use App\Models\InventoryAdjustment;
use App\Models\Product;
use App\Models\SalesOrder;
use App\Models\SalesOrderLine;
use App\Models\Setting;
use App\Models\Warehouse;
use App\Services\SalesOrder\WarehouseRoutingMethod;
use Database\Factories\Concerns\HasFactoryDataRecycler;
use Illuminate\Database\Eloquent\Factories\Factory;

/**
 * @extends \Illuminate\Database\Eloquent\Factories\Factory<\App\Models\SalesOrderLine>
 */
class SalesOrderLineFactory extends Factory
{
    use HasFactoryDataRecycler;

    /**
     * Define the model's default state.
     *
     * @return array<string, mixed>
     */
    public function definition(): array
    {
        return [
            'sales_order_id' => SalesOrder::factory(),
            'description' => $this->faker->sentence(),
            'product_id' => Product::factory(),
            'nominal_code_id' => Helpers::setting(Setting::KEY_NC_MAPPING_SALES_ORDERS),
            'amount' => $this->faker->randomFloat(2, 0.01, 1000),
            'quantity' => $this->faker->numberBetween(5, 10),
            'warehouse_routing_method' => WarehouseRoutingMethod::WAREHOUSE,
            'warehouse_id' => Warehouse::factory(),
        ];
    }

    /**
     * Create inventory adjustments for sales order line so that there is sufficient stock.  If we don't do this
     * and there is no stock available, backorders will be created when making reservations.
     */
    public function noBackorders()
    {
        return $this->afterCreating(function (SalesOrderLine $salesOrderLine) {
            // Check if there is sufficient stock
            if ($salesOrderLine->quantity > $salesOrderLine->product->availableQuantity?->quantity) {
                /** @var InventoryAdjustment $inventoryAdjustment */
                InventoryAdjustment::factory()->create([
                    'product_id' => $salesOrderLine->product_id,
                    'quantity' => $salesOrderLine->quantity,
                    'warehouse_id' => $salesOrderLine->warehouse_id,
                    'adjustment_date' => $salesOrderLine->salesOrder->order_date,
                ]);
            }
        });
    }
}
