<?php

namespace Database\Factories;

use App\Models\Product;
use App\Models\ProductBrand;
use App\Models\ProductImage;
use App\Models\ProductPricing;
use App\Models\ProductPricingTier;
use App\Models\SupplierProduct;
use Database\Factories\Concerns\HasFactoryDataRecycler;
use Illuminate\Database\Eloquent\Factories\Factory;

// standard type

// archived product
// unarchived product

/**
 * After creating product.
 */
class ProductFactory extends Factory
{
    use HasFactoryDataRecycler;

    /**
     * Define the model's default state.
     */
    public function definition(): array
    {
        return [
            'sku' => $this->faker->ean13(),
            'type' => 'standard',
            'name' => $this->faker->realText(50, 2),
            'brand_id' => ProductBrand::factory(),
            'barcode' => $this->faker->ean13(),
            // weight
            'weight' => $this->faker->randomFloat(4, 1, 100),
            'weight_unit' => $this->faker->randomElement(Product::WEIGHT_UNITS),
            // dimension
            'length' => $this->faker->randomFloat(4, 1, 100),
            'width' => $this->faker->randomFloat(4, 1, 100),
            'height' => $this->faker->randomFloat(4, 1, 100),
            'dimension_unit' => $this->faker->randomElement(Product::DIMENSION_UNITS),
            // average cost
        ];
    }

    public function withProductPricing(): self
    {
        return $this->afterCreating(function (Product $product) {
            ProductPricing::factory()->create([
                'product_pricing_tier_id' => ProductPricingTier::query()->first()->id,
                'product_id' => $product->id,
                'price' => $this->faker->randomFloat(2, 1, 250),
            ]);
        });
    }

    public function withSupplierProduct(?FactoryDataRecycler $factoryDataRecycler = null): self
    {
        /** @var SupplierProductFactory $supplierProductFactory */
        $supplierProductFactory = SupplierProduct::factory();

        if ($factoryDataRecycler) {
            $supplierProductFactory = $supplierProductFactory->factoryDataRecycler($factoryDataRecycler);
        }

        return $this->has(
            $supplierProductFactory
        );
    }

    public function withBrand(): self
    {
        return $this->afterCreating(function (Product $product) {
            ProductBrand::factory()->create();
        });
    }

    public function withImages(int $numImages = 1): self
    {
        return $this->has(
            ProductImage::factory($numImages, [
                'is_primary' => true,
            ]),
            'images'
        );
    }

    public function standard()
    {
        return $this->state(['type' => Product::TYPE_STANDARD]);
    }

    public function archived()
    {
        return $this->state(function () {
            return [
                'archived_at' => $this->faker->dateTimeThisMonth(),
            ];
        });
    }

    public function unarchived()
    {
        return $this->state(['archived_at' => null]);
    }
}
