<?php

namespace Database\Factories;

use App\Models\SupplierProduct;
use Faker\Generator;
use Illuminate\Container\Container;
use Illuminate\Contracts\Container\BindingResolutionException;
use Illuminate\Database\Eloquent\Collection as EloquentCollection;
use Illuminate\Database\Eloquent\Factories\Factory;
use Illuminate\Database\Eloquent\Factories\Sequence;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Support\Str;

/**
 * This class is used to restrict factory creation to use specific model data
 *
 * @extends Factory<SupplierProduct>
 */
class FactoryDataRecycler
{
    protected Generator $faker;

    protected array $recycledData;

    public function __construct(array $recycledData = [])
    {
        $this->recycledData = $recycledData;
        try {
            $this->faker = $this->withFaker();
        } catch (BindingResolutionException) {
        }
    }

    /**
     * Get a new Faker instance.
     *
     * @throws BindingResolutionException
     */
    protected function withFaker(): Generator
    {
        return Container::getInstance()->make(Generator::class);
    }

    public function addRecycledData(
        EloquentCollection|Model|Factory $data,
        ?string $localKey = null,
        bool $uniqueModel = false,
    ): self {
        $this->recycledData[] = new RecycledData($data, $localKey, $uniqueModel);

        return $this;
    }

    public function process(Factory $factory): Factory
    {
        /** @var RecycledData|EloquentCollection|Model|Factory $data */
        foreach ($this->recycledData as $data) {
            if (! ($data instanceof RecycledData)) {
                $data = new RecycledData($data);
            }
            if ($data->data instanceof Factory) {
                $data->data = $data->data->create();
            }
            if ($data->data instanceof Model) {
                $data->data = new EloquentCollection($data->data);
            }
            $items = [];
            $localKey = $data->localKey ?? Str::singular($data->data->first()->getTable()).'_id';
            for ($i = 0; $i < $data->data->count(); $i++) {
                $items[] = [
                    $localKey => $data->uniqueModel ?
                        $this->faker->unique()->randomElement($data->data)->id :
                        $this->faker->randomElement($data->data)->id,
                ];
            }
            $factory = $factory->state(new Sequence(...$items));
        }

        return $factory;
    }
}
