<?php

namespace App;

use Illuminate\Support\Arr;
use Illuminate\Support\Str;

class Validator extends \Illuminate\Validation\Validator
{
    /**
     * Replace all error message place-holders with actual values.
     */
    public function makeReplacements($message, $attribute, $rule, $parameters): array
    {
        // replace attribute, input, values...
        $message = parent::makeReplacements($message, $attribute, $rule, $parameters);
        // replace index
        $message = $this->replaceIndexPlaceholder($message, $attribute);

        return [
            'message' => $message,
            'code' => $this->getRuleCode($rule, $attribute, $parameters),
            'data' => $this->getRuleData($rule, $attribute, $parameters),
        ];
    }

    /**
     * Get the Correct code from rule.
     *
     *
     * @return mixed|string
     */
    private function getRuleCode(string $rule, string $attribute, $parameters)
    {
        // after_attribute: SKU -> Invalid  = SKUInvalid
        $codes = [
            'Exists' => ['code' => 'Invalid', 'after_attribute' => false],
            'In' => ['code' => 'Invalid', 'after_attribute' => false],
            'NotIn' => ['code' => 'Invalid', 'after_attribute' => false],
            'Numeric' => ['code' => 'NonNumeric', 'after_attribute' => false],
            'Array' => 'IsNotArray',
            'Integer' => 'IsNotInteger',
            'Boolean' => 'IsNotBoolean',
            'RequiredWithout' => 'Required',
            'Required' => 'Required',
            'RequiredWithWithout' => 'Required',
            'Unique' => 'MustBeUnique',
            'Max' => 'ExceededMax',
            'Lt' => 'ExceededMax',
            'Gt' => 'ExceededMin',
            'RequiredIf' => 'RequiredOnlyIf',
            'Email' => 'MustBeValidEmail',
        ];

        $code = $codes[$rule] ?? $rule;

        $afterAttribute = $code['after_attribute'] ?? true;

        $code = is_array($code) ? $code['code'] : $code;

        // get displayable attribute
        $attribute = $this->getDisplayableAttribute($attribute);
        // add attribute to Code
        if ($afterAttribute) {
            $code = ucfirst(Str::camel($attribute)).$code;
        } else {
            $code = $code.ucfirst(Str::camel($attribute));
        }

        if ($rule === 'RequiredIf') {
            $code .= ucfirst(Str::camel($this->getDisplayableAttribute($parameters[0]))).'Is'.ucfirst(Str::camel(implode(' / ', $this->getAttributeList(array_slice($parameters, 1)))));
        } elseif ($rule === 'RequiredWith') {
            $code .= ucfirst(Str::camel(implode(' / ', $this->getAttributeList($parameters))));
        }

        return $code;
    }

    /**
     * Get the displayable name of the attribute.
     */
    public function getDisplayableAttribute($attribute): string
    {
        $attribute = parent::getDisplayableAttribute($attribute);

        $attributeSplit = explode('.', $attribute);
        if (($splitCount = count($attributeSplit)) > 1) {
            $attribute = parent::getDisplayableAttribute($attributeSplit[$splitCount - 1]);
        }

        if (is_numeric($attribute) && isset($attributeSplit[$splitCount - 2])) {
            $attribute = Str::singular($attributeSplit[$splitCount - 2]);
        }

        return $attribute;
    }

    /**
     * Replace the :index placeholder in the given message.
     */
    protected function replaceIndexPlaceholder($message, $attribute): string
    {
        $index = $this->getIndex($attribute);
        if (! is_null($index)) {
            $index = " (Index #$index)";
        }

        $message = str_replace(':index', $index, $message);

        return $message;
    }

    /**
     * Get the index of a given attribute.
     *
     *
     * @return mixed
     */
    protected function getIndex(string $attribute)
    {
        $attributeSplit = explode('.', $attribute);
        if (($splitCount = count($attributeSplit)) > 1) {
            if (is_numeric($attributeSplit[$splitCount - 2])) {
                return intval($attributeSplit[$splitCount - 2]);
            } else {
                return intval($attributeSplit[$splitCount - 1]);
            }
        }

        return null;
    }

    /**
     * Get the data needed for every rule.
     */
    private function getRuleData(string $rule, string $attribute, $parameters): array
    {
        $data = [];
        // index
        if (! is_null($index = $this->getIndex($attribute))) {
            $data['index'] = $index;
        }

        // get displayable attribute
        $displayableAttribute = $this->getDisplayableAttribute($attribute);

        $data[Str::snake($displayableAttribute)] = $this->getValue($attribute);

        if ($rule === 'RequiredWithout') {
            foreach ($parameters as $item) {
                $displayableAttribute = $this->getDisplayableAttribute($item);
                $data[Str::snake($displayableAttribute)] = $this->getValue($item);
            }
        } elseif ($rule === 'RequiredIf') {
            $displayableAttribute = $this->getDisplayableAttribute($parameters[0]);
            $data[Str::snake($displayableAttribute)] = $this->getValue($parameters[0]);
        }

        return $data;
    }

    /**
     * Determine if the data passes the validation rules.
     *
     * Overwrite this function for get messages instance from Our MessageBag
     */
    public function passes(): bool
    {
        // to get instance from our MessageBag
        $this->messages = new MessageBag();

        [$this->distinctValues, $this->failedRules] = [[], []];

        // We'll spin through each rule, validating the attributes attached to that
        // rule. Any error messages will be added to the containers with each of
        // the other error messages, returning true if we don't have messages.
        foreach ($this->rules as $attribute => $rules) {
            $attribute = str_replace('\.', '->', $attribute);

            foreach ($rules as $rule) {
                $this->validateAttribute($attribute, $rule);

                if ($this->shouldStopValidating($attribute)) {
                    break;
                }
            }
        }

        // Here we will spin through all of the "after" hooks on this validator and
        // fire them off. This gives the callbacks a chance to perform all kinds
        // of other validation that needs to get wrapped up in this operation.
        foreach ($this->after as $after) {
            call_user_func($after);
        }

        return $this->messages->isEmpty();
    }

    /**
     * {@inheritDoc}
     *
     * Currently this function only used when get validated attributes
     *
     * we remove attribute if it has "array" rule and have children attributes
     */
    public function getRules()
    {
        return array_filter($this->rules, function ($rule, $attribute) {
            if (in_array('array', $rule)) {
                $attributeChildren = array_filter($this->rules, function ($rule, $attr) use ($attribute) {
                    return Str::startsWith($attr, $attribute.'.');
                }, ARRAY_FILTER_USE_BOTH);

                return count($attributeChildren) ? false : true;
            }

            return true;
        }, ARRAY_FILTER_USE_BOTH);
    }

    /**
     * Validate that an attribute exists when another attribute exists and another one does not exists.
     *
     * @param  mixed  $value
     * @param  mixed  $parameters
     */
    public function validateRequiredWithWithout(string $attribute, $value, $parameters): bool
    {
        $this->requireParameterCount(2, $parameters, 'required_with_without');

        $with = $parameters[0];
        $without = $parameters[1];

        if ($this->validateRequired($with, $this->getValue($with)) && ! $this->validateRequired($without, $this->getValue($without))) {
            return $this->validateRequired($attribute, $value);
        }

        return true;
    }

    /**
     * Replace all place-holders for the required_with_without rule.
     */
    protected function replaceRequiredWithWithout(string $message, string $attribute, string $rule, array $parameters): string
    {
        return str_replace(':with', $this->getDisplayableAttribute($parameters[0]), $message);
    }

    protected function isImplicit($rule)
    {
        return parent::isImplicit($rule) || $rule == 'RequiredWithWithout';
    }

    public function addToData($data)
    {
        return $this->setData(array_replace($this->data, $data));
    }

    public function validateRules(array $rules)
    {
        if (empty($rules)) {
            return;
        }

        // add rules to validator to return in validated data and parse rules
        $this->addRules($rules);

        // validate only sent rules after parsing
        foreach (Arr::only($this->rules, array_keys($rules)) as $attribute => $attributeRules) {
            $attribute = str_replace('\.', '->', $attribute);

            foreach ($attributeRules as $rule) {
                $this->validateAttribute($attribute, $rule);

                if ($this->shouldStopValidating($attribute)) {
                    break;
                }
            }
        }
    }
}
