<?php
/**
 * Created by PhpStorm.
 * User: brightantwiboasiako
 * Date: 7/23/20
 * Time: 9:13 AM.
 */

namespace App\Services\SupplierInventory;

use App\Models\Product;
use App\Models\Supplier;
use App\Models\SupplierInventory;
use App\Repositories\SupplierInventoryRepository;
use App\Repositories\WarehouseRepository;
use App\Response;
use Generator;
use Illuminate\Contracts\Validation\Factory;

class SupplierInventoryCSVManager
{
    /**
     * @var SupplierInventoryRepository
     */
    private $inventories;

    /**
     * @var Factory
     */
    private $validator;

    /**
     * @var WarehouseRepository
     */
    private $warehouses;

    /**
     * SupplierInventoryCSVManager constructor.
     */
    public function __construct(SupplierInventoryRepository $inventories, Factory $validator, WarehouseRepository $warehouses)
    {
        $this->inventories = $inventories;
        $this->validator = $validator;
        $this->warehouses = $warehouses;
    }

    public function importInventories(Supplier $supplier, Generator $rows, bool $override, Response $messageBag): int
    {
        $successCount = 0;

        foreach ($rows as $index => $row) {
            if ($this->isInventoryRowValid($row, $messageBag, $index)) {
                $this->importRowForSupplier($supplier, $row, $override, $messageBag);
                $successCount++;
            }
        }

        return $successCount;
    }

    private function importRowForSupplier(Supplier $supplier, array $row, bool $override, Response $messageBag): void
    {
        // Get the warehouse for the supplier by warehouse name
        $warehouse = $this->warehouses->findWarehouseByNameForSupplier($row['warehouse_name'], $supplier->id);
        if (! $warehouse) {
            $this->addInvalidWarehouseWarning($row['warehouse_name'], $messageBag);

            return;
        }

        $product = Product::with([])->where('sku', $row['product_sku'])->first();
        $search = ['supplier_id' => $supplier->id, 'warehouse_id' => $warehouse->id, 'product_id' => $product->id];
        $inventory = $this->inventories->findBy($search);
        if ($override && $inventory) {
            // It's okay to override an existing supplier inventory
            $row['in_stock'] = $row['quantity'] ? null : $row['in_stock'];
            $inventory->update($row);
        }

        if (! $inventory) {
            // Inventory doesn't already exist
            $this->inventories->createInventory(array_merge($row, [
                'supplier_id' => $supplier->id,
                'product_id' => $product->id,
                'warehouse_id' => $warehouse->id,
            ]));
        }
    }

    public function exportInventories()
    {
    }

    /**
     * @return bool|mixed
     */
    private function isInventoryRowValid(array $row, Response $messageBag, int $index)
    {
        $rules = [
            'product_sku' => 'required|exists:products,sku',
            'warehouse_name' => 'required|exists:warehouses,name',
            'quantity' => 'required_without:in_stock|numeric|min:0',
            'in_stock' => 'required_without:quantity|boolean',
            'eta' => 'nullable|date',
            'source' => 'required|in:'.implode(',', SupplierInventory::SOURCES),
        ];

        $validation = $this->validator->make($row, $rules);

        if ($validation->fails()) {
            $messageBag->addWarningsFromValidator($validation, 'supplier_inventory'.$index);

            return false;
        }

        return true;
    }

    private function addInvalidWarehouseWarning($warehouseName, Response $messageBag)
    {
        $messageBag->addWarning(__('messages.supplier_product.invalid_supplier_warehouse', [
            'warehouse_name' => $warehouseName,
        ]), 'SupplierInventory'.Response::CODE_INVALID_ROW, 'warehouse_name', ['warehouse_name' => $warehouseName]);
    }
}
