<?php
namespace App\Services\StockTake;

use App\Abstractions\StockTakeItemDataInterface;
use App\Data\CreateInitialStockTakeData;
use App\Data\CreateInitialStockTakeItemData;
use App\Data\CreateStockTakeData;
use App\Data\StockTakeItemBulkData;
use App\Data\StockTakeItemData;
use App\Data\UpdateStockTakeData;
use App\Exceptions\CantDeleteStockTakeItemsForClosedStockTakeException;
use App\Exceptions\ClosedStockTakesCantChangeQuantityException;
use App\Exceptions\InsufficientInventoryStockException;
use App\Exceptions\InsufficientStockException;
use App\Exceptions\NegativeAdjustmentOnStockTakeCannotBeModifiedException;
use App\Models\FifoLayer;
use App\Models\Product;
use App\Models\StockTake;
use App\Models\StockTakeItem;
use App\Repositories\FifoLayerRepository;
use App\Repositories\ProductRepository;
use App\Repositories\StockTakeItemRepository;
use App\Repositories\StockTakeRepository;
use App\Services\InventoryManagement\InventoryManager;
use Carbon\Carbon;
use Exception;
use Illuminate\Support\Collection;
use Illuminate\Support\Facades\DB;
use InvalidArgumentException;
use Spatie\LaravelData\DataCollection;
use Spatie\LaravelData\Optional;
use Throwable;
use function PHPUnit\Framework\isInstanceOf;

/**
 * Class StockTakeManager.
 */
class StockTakeManager
{

    /**
     * StockTakeManager constructor.
     */
    public function __construct(
        private readonly StockTakeRepository $stockTakes,
        private readonly StockTakeItemRepository $stockTakeItems,
        private readonly FifoLayerRepository $fifoLayers,
        private readonly InventoryMovementStockCount $processor,
        private readonly ProductRepository $products,
    )
    {}

    /**
     * @throws Throwable
     */
    public function initiateCount(StockTake $stockTake, $dateOverride = null, bool $isIntegrityFix = false): StockTake
    {
        if ($stockTake->isClosed() || $stockTake->isOpen()) {
            throw new InvalidArgumentException('Only draft stock takes can be initiated.');
        }

        if ($stockTake->stockTakeItems()->count() === 0) {
            throw new InvalidArgumentException('Stock take must have products.');
        }

        if ($dateOverride) {
            $stockTake->date_count = Carbon::parse($dateOverride);
            $stockTake->save();
        }

        // We process the stock count.
        $stockTake = $this->processor->countInventory($stockTake, $isIntegrityFix);
        $stockTake->load('stockTakeItems');

        return $stockTake;
    }

    /**
     * @throws Throwable
     */
    public function finalizeStockTake(StockTake $stockTake, bool $autoApplyStock = true): StockTake
    {
        return DB::transaction(function () use ($stockTake, $autoApplyStock) {
            if (! $stockTake->isOpen()) {
                throw new InvalidArgumentException('Only open stock takes can be finalized.');
            }

            // Make sure stock take items have quantities
            if (! $stockTake->hasFullQuantities()) {
                throw new InvalidArgumentException('Each stock take item must have quantities.');
            }

            // We finalize the count by making any necessary adjustments

            // Reset the value changed
            $stockTake->setValueChanged(0);

            /**
             * We close the stock take before creating any
             * adjustments as positive adjustments may attempt
             * to release backorder queues which may have inventory
             * movements. Not closing the stock take first results in
             * a recursive bug about the stock take lock. This approach
             * is still fine as we're performing the operations in a transaction.
             */
            $stockTake->close();

            $stockTake->loadMissing('stockTakeItems.product.activeFifoLayers');
            $missing_inventory = collect();
            $stockTake->stockTakeItems->each(function (StockTakeItem $stockTakeItem) use ($stockTake, $missing_inventory, $autoApplyStock) {
                $this->finalizeStockTakeItem($stockTakeItem, $stockTake, $autoApplyStock, $missing_inventory);
            });
            if (count($missing_inventory)) {
                throw new InsufficientInventoryStockException('Insufficient inventory to stock adjust for following SKUs: '.implode(',', $missing_inventory->toArray()));
            }

            $stockTake->refresh();
            $stockTake->load('stockTakeItems');

            return $stockTake;
        });
    }

    public function revertToDraft($stockTakeId): StockTake
    {
        $stockTake = $this->stockTakes->findOrFail($stockTakeId);
        if (! $stockTake->isOpen()) {
            throw new InvalidArgumentException('Stock take must be open.');
        }

        // We reverse the open status
        $stockTake->revertToDraft();
        $stockTake->load('stockTakeItems');

        return $stockTake;
    }

    /**
     * @throws Throwable
     */
    public function createStockTake(CreateStockTakeData|CreateInitialStockTakeData $data): StockTake
    {
        if ($data->date_count instanceof Optional) {
            $data->date_count = now();
        }
        return DB::transaction(function () use ($data) {
            // Stock take is created in draft mode
            $data->status = StockTake::STOCK_TAKE_STATUS_DRAFT;
            $stockTake = $this->stockTakes->create($data->toArray());
            $this->upsertStockTakeItems($stockTake, $data->items);

            return $stockTake;
        });
    }

    /**
     * @throws Throwable
     */
    public function createInitialStockTake(CreateInitialStockTakeData $data, bool $autoApplyStock = true): StockTake
    {
        $stockTake = DB::transaction(function () use ($data) {
            return $this->createStockTake($data);
        });

        $this->initiateCount($stockTake);
        $this->finalizeStockTake($stockTake, $autoApplyStock);

        return $stockTake->refresh();
    }

    /**
     * @throws Throwable
     *
     * TODO: Consider rules that should apply for modifying stock takes and be sure to account for inventory movement changes that result
     */
    public function modifyStockTake(StockTake $stockTake, UpdateStockTakeData $data): StockTake
    {
        if ($stockTake->isOpen()) {
            if (!$stockTake->is_initial_count) {
                // TODO: Should date be modifiable for open stock take?  What are the implications?
                /*
                 * TODO: Prevent adding new items to open stock take?
                 *  Why? Inventory may have changed since the stock take opened
                 */
            }
        }
        if ($stockTake->isClosed()) {
            if (!$stockTake->is_initial_count && ! ($data->items instanceof Optional)) {
                // TODO: Should date be modifiable for closed stock take?  What are the implications?
                $data->items->each(function(StockTakeItemData $item) use ($stockTake) {
                    if ($item->to_delete) {
                        throw new CantDeleteStockTakeItemsForClosedStockTakeException("Cannot delete items from a closed stock take (Stock Take #$stockTake->id.)");
                    }
                    if (! ($item->qty_counted instanceof Optional)) {
                        throw new ClosedStockTakesCantChangeQuantityException("Cannot modify counted quantities for a closed stock take (Stock Take #$stockTake->id).");
                    }
                });
            }
        }

        return DB::transaction(function () use ($stockTake, $data) {
            $stockTake->fill($data->toArray());

            if ($stockTake->isDirty(['date_count'])) {
                $stockTake->stockTakeItems->each(function (StockTakeItem $stockTakeItem) use ($stockTake) {
                    if ($fifoLayer = $stockTakeItem->creationFifoLayer) {
                        $originatingMovement = $stockTakeItem->getOriginatingMovement();
                        $originatingMovement->inventory_movement_date = $stockTake->date_count;
                        $originatingMovement->save();
                        $fifoLayer->fifo_layer_date = $stockTake->date_count;
                        $fifoLayer->save();
                    }
                });
            }

            $stockTake->save();

            $stockTakeItems = $this->upsertStockTakeItems($stockTake, $data->items);
            if ($stockTakeItems->isNotEmpty() && $stockTake->isOpen()) {
                $this->processor->countInventory($stockTake->setValueChanged(0));
            }
            // Only closed stock takes have movements
            if ($stockTake->isClosed()) {
                // TODO: Account for valuation changes
                // TODO: Account for fifo layer changes /  inventory movement changes
                // eager load stock take items all at once with necessary eager loads
                // TODO: Failing to update inventory movement and fifo layer date for date_count change of closed stock take
                $stockTakeItems = StockTakeItem::with(['creationFifoLayer', 'fifoLayers', 'inventoryMovements'])->whereIn('id', $stockTakeItems->pluck('id'))->get();
                $stockTakeItems->each(function (StockTakeItem $stockTakeItem) use ($stockTake, &$totalValueChange) {
                    if ($stockTakeItem->quantity_adjusted > 0)
                    {
                        $creationFifoLayer = $stockTakeItem->creationFifoLayer;
                        // Skip if this is a new addition to the stock take
                        if (!$creationFifoLayer) {
                            $this->finalizeStockTakeItem($stockTakeItem, $stockTake, !$stockTake->is_initial_count);
                            return;
                        }
                        $creationFifoLayer->original_quantity = $stockTakeItem->quantity_adjusted;
                        $creationFifoLayer->total_cost = $stockTakeItem->unit_cost * $stockTakeItem->quantity_adjusted;
                        $creationFifoLayer->fifo_layer_date = $stockTake->date_count;

                        if ($creationFifoLayer->isDirty(['original_quantity', 'total_cost'])) {
                            $this->fifoLayers->updateUsagesForFifoLayer($creationFifoLayer);
                        }

                        if ($creationFifoLayer->isDirty(['fifo_layer_date'])) {
                            $creationFifoLayer->inventoryMovements->each(function ($movement) use ($creationFifoLayer) {
                                $movement->inventory_movement_date = $creationFifoLayer->fifo_layer_date;
                                $movement->save();
                            });
                        }

                        // On FIFO layer save, it won't let the FIFO layer be over-fulfilled
                        $creationFifoLayer->save();
                    } else if ($stockTakeItem->quantity_adjusted < 0) {
                        // Negative adjustments should not be able to be modified so this should never happen, Initial inventory
                        // counts don't have negative adjustments
                        throw new NegativeAdjustmentOnStockTakeCannotBeModifiedException("Negative stock take adjustments should not be able to be modified.");
                    }

                });
                $this->stockTakes->recalculateValueChange($stockTake);
            }

            return $stockTake;
        });
    }

    public function upsertStockTakeItems(StockTake $stockTake, DataCollection|Optional $stockTakeItemsData): Collection
    {
        if ($stockTakeItemsData instanceof Optional) {
            return collect();
        }

        // TODO: for initial stock take, we need to consider fifo and inventory movement implications
        $this->deleteStockTakeItemsMarkedForDeletion($stockTake, $stockTakeItemsData);

        $stockTakeItemsData = $stockTakeItemsData
            ->reject(fn($item) => @$item->to_delete)
            ->map(function (StockTakeItemDataInterface $stockTakeItemData) use ($stockTake)
        {
            $product = Product::findOrFail($stockTakeItemData->product_id);
            if (isset ($stockTakeItemData->qty_change) && !($stockTakeItemData->qty_change instanceof Optional)) {
                $stockTakeItemData->qty_counted = $this->products->getAvailableQuantityInWarehouse($product, $stockTake->warehouse) + $stockTakeItemData->qty_change;
            }
            if (
                $stockTakeItemData->unit_cost instanceof Optional &&
                !($stockTakeItemData->qty_counted instanceof Optional) &&
                $stockTakeItemData->qty_counted > 0
            ) {
                // Only update unit cost if it is not already set in the database or in the data object
                if (!$stockTake->stockTakeItems()->where('product_id', $product->id)->first()?->unit_cost) {
                    $stockTakeItemData->unit_cost = $product->getUnitCostAtWarehouse($stockTake->warehouse_id);
                }
            }

            // Convert the data object to an array and add the stock_take_id
            $stockTakeItemDataArray = $stockTakeItemData->toArray();
            $stockTakeItemDataArray['stock_take_id'] = $stockTake->id;

            return StockTakeItemBulkData::from($stockTakeItemDataArray);
        });

        return $this->stockTakeItems->save($stockTakeItemsData->toCollection(), StockTakeItem::class);
    }

    private function deleteStockTakeItemsMarkedForDeletion(StockTake $stockTake, DataCollection $stockTakeItemsData): void
    {
        $productIdsToDelete = $stockTakeItemsData
            ->reject(fn($item) => $item instanceof CreateInitialStockTakeItemData)
            ->filter(fn (StockTakeItemData $item) => $item->to_delete)
            ->toCollection()
            ->pluck('product_id');
        $this->stockTakes->deleteForProductIds($stockTake, $productIdsToDelete);
    }

    /**
     * @throws Throwable
     */
    private function finalizeStockTakeItem(
        StockTakeItem $stockTakeItem,
        StockTake $stockTake,
        bool $autoApplyStock,
        ?Collection $missing_inventory = null
    ): void {
        if ($stockTakeItem->snapshot_inventory != $stockTakeItem->qty_counted || $stockTake->is_initial_count) {
            $stockTakeItem->setRelation('stockTake', $stockTake);
            // Create an adjustment
            $manager = InventoryManager::with(
                $stockTake->warehouse_id,
                $stockTakeItem->product
            );
            try {
                $diff = $stockTake->is_initial_count ? $stockTakeItem->qty_counted : $stockTakeItem->qty_counted - $stockTakeItem->snapshot_inventory;

                if ($diff > 0) {
                    $manager->addToStock($diff, $stockTakeItem, true, $autoApplyStock,
                        $stockTakeItem->unit_cost ?? $stockTakeItem->product->getUnitCostAtWarehouse($stockTake->warehouse_id));
                } elseif ($diff < 0) {
                    $manager->takeFromStock(abs($diff), $stockTakeItem);
                }
            } catch (InsufficientStockException $e) {
                $missing_inventory->add($e->productSku);
            }
        }
    }
}
