<?php

namespace App\Services\StockTake;

use App\Exceptions\NegativeInventoryStockTakeException;
use App\Models\InventoryMovement;
use App\Models\StockTake;
use App\Models\StockTakeItem;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Support\Facades\DB;
use Throwable;

class InventoryMovementStockCount implements CountProcessor
{
    /**
     * @throws Throwable
     */
    public function countInventory(StockTake $stockTake, bool $isIntegrityFix = false): StockTake
    {
        // We count the inventory via inventory movements for products
        // under the stock take
        return DB::transaction(function () use ($stockTake, $isIntegrityFix) {
            // Open the stock count
            $stockTake->open();

            $negativeInventoryLines = [];
            // TODO: HasMany relation
            $stockTake->stockTakeItems()->with(['product.activeFifoLayers', 'product.inventoryMovements' => function ($builder) use ($stockTake, $isIntegrityFix) {
                $builder->where('warehouse_id', $stockTake->warehouse_id)
                    ->select('product_id', 'warehouse_id', DB::raw('sum(quantity) as quantity'))
                    ->where(function (Builder $builder) {
                        return $builder->where('inventory_status', InventoryMovement::INVENTORY_STATUS_ACTIVE)
                            ->orWhere('inventory_status', InventoryMovement::INVENTORY_STATUS_RESERVED);
                    })->groupBy(['product_id', 'warehouse_id']);
            }])->each(function (StockTakeItem $stockTakeItem) use ($stockTake, &$negativeInventoryLines, $isIntegrityFix) {
                $stockTakeItem->setRelation('stockTake', $stockTake);
                $unitsAvailable = $stockTakeItem->product->inventoryMovements->sum('quantity');

                $stockTakeItem->snapshot_inventory = $stockTake->is_initial_count ? 0 : $unitsAvailable;

                if ($stockTakeItem->snapshot_inventory < 0 && !$isIntegrityFix) {
                    $negativeInventoryLines[] = $stockTakeItem;

                    return;
                }

                $stockTakeItem->save();

                // Update value change
                if (is_null($stockTakeItem->qty_counted)) {
                    return;
                }
                $diff = $stockTakeItem->qty_counted - $stockTakeItem->snapshot_inventory;
                if ($diff > 0) {
                    $unitCost = $stockTakeItem->unit_cost ?? ($stockTakeItem->product->getCurrentFifoLayerForWarehouse($stockTake->warehouse)->avg_cost ?? $stockTakeItem->product->average_cost);
                } elseif ($diff < 0) {
                    $unitCost = $stockTakeItem->unit_cost ?? $stockTakeItem->product->getTotalCostByFifoLayers(abs($diff));
                } else {
                    $unitCost = 0;
                }

                $stockTakeItem->stockTake->addValueChanged($unitCost * $diff);
            });

            if (! empty($negativeInventoryLines) && !$isIntegrityFix) {
                throw new NegativeInventoryStockTakeException($negativeInventoryLines);
            }

            return $stockTake->refresh();
        });
    }
}
