<?php

namespace App\Services\Shopify;

use App\Exceptions\InsufficientStockException;
use App\Integrations\Shopify;
use App\Models\SalesOrderFulfillment;
use App\Models\SalesOrderFulfillmentLine;
use App\Models\Shopify\ShopifyOrderMapping;
use App\Repositories\Shopify\ShopifyOrderMappingRepository;
use Facades\App\Services\Shopify\Orders\Actions\ShopifyDownloadOrder;
use Illuminate\Support\Str;

class ShopifySubmitTrackingInfo extends \App\Services\SalesOrderFulfillment\SubmitTrackingInfo
{
    protected ShopifyOrderMappingRepository $shopifyOrderMappingRepository;

    public function __construct(SalesOrderFulfillment $salesOrderFulfillment)
    {
        parent::__construct($salesOrderFulfillment);
        $this->shopifyOrderMappingRepository = app(ShopifyOrderMappingRepository::class);
    }

    /**
     * {@inheritDoc}
     */
    public function submit(): array
    {
        customlog('submitShopifyFulfillments', 'Submit Shopify Fulfillments for '.$this->salesOrderFulfillment->salesOrder->sales_order_number.' ('.$this->salesOrderFulfillment->id.')', [], 7);
        if (config('app.env') !== 'production' && ! @$this->salesOrderFulfillment->salesOrder->salesChannel->integrationInstance->integration_settings['test']) {
            return ['success' => true, 'message' => 'The fulfillment submitted successfully (just on production)'];
        }

        // TODO: I think we no longer need this condition
        try {
            /** @var \App\Models\Shopify\ShopifyOrder $shopifyOrder */
            $shopifyOrder = \App\Models\Shopify\ShopifyOrder::with([])->where('sku_sales_order_id', $this->salesOrderFulfillment->sales_order_id)->firstOrFail();
        } catch (\Throwable $e) {
            return ['success' => false, 'message' => 'shopify order not found'];
        }

        // if it's mapped just mark as submitted
        if ($this->shopifyOrderMappingRepository->getMappingForTypeForShopifyOrder($shopifyOrder, ShopifyOrderMapping::LINK_TYPE_FULFILLMENTS, $this->salesOrderFulfillment->id)) {
            $this->salesOrderFulfillment->markAsSubmittedToSalesChannel(now());

            return ['success' => true, 'message' => 'The fulfillment is already submitted'];
        }

        // the order canceled or refunded
        if (! empty($shopifyOrder->json_object['cancelled_at']) || $shopifyOrder->json_object['financial_status'] == 'refunded') {
            // TODO: what should we do?
            $this->salesOrderFulfillment->markAsSubmittedToSalesChannel(now());

            return ['success' => true, 'message' => 'The shopify order canceled or refunded (marked as submitted)'];
        }

        $integrationInstance = $this->salesOrderFulfillment->salesOrder->salesChannel->integrationInstance;
        $inventoryData = $integrationInstance->getInventoryData();
        $shopifyLocationId = $inventoryData->locations->count() > 0 ?
            $inventoryData->locations->first()->id : null;
        // shopify fulfillment request
        $shopify = new Shopify($integrationInstance);
        $shopifyFulfillmentOrders = $shopify->getFulfillmentOrder($shopifyOrder->shopify_order_id);
        $fulfillmentOrder = collect($shopifyFulfillmentOrders)
            ->whereIn('status', ['open', 'in_progress'])
            //->where('delivery_method.method_type', 'shipping') // Need to look at more data before deciding if this is a good filter
            ->first();

        if (! empty($fulfillmentOrder)) {
            $fulfillmentLinesIdsAlreadyProcessed = [];
            $options = [
                'order_id' => $shopifyOrder->shopify_order_id,
                'location_id' => $shopifyLocationId,
                'notify_customer' => config('app.env') === 'production',
                'tracking_info' => [
                    'number' => $this->salesOrderFulfillment->tracking_number,
                    'url' => '',
                    'company' => $this->salesOrderFulfillment->fulfilledShippingMethod ? $this->salesOrderFulfillment->fulfilledShippingMethod->shippingCarrier->name : $this->salesOrderFulfillment->fulfilled_shipping_method,
                ],
                'line_items_by_fulfillment_order' => [
                    [
                        'fulfillment_order_id' => $fulfillmentOrder['id'],
                        'fulfillment_order_line_items' => $this->salesOrderFulfillment
                            ->salesOrderFulfillmentLines
//                            ->filter(function (SalesOrderFulfillmentLine $fulfillmentLine) {
//                                return !$fulfillmentLine->salesOrderLine->bundle;
//                            })
                            ->map(function (
                                SalesOrderFulfillmentLine $fulfillmentLine
                            ) use ($shopifyOrder, $fulfillmentOrder, &$fulfillmentLinesIdsAlreadyProcessed) {
                                if (in_array($fulfillmentLine->id, $fulfillmentLinesIdsAlreadyProcessed)) {
                                    return null;
                                }

                                // we send only the fulfillable quantity from the shopify order payload
                                $shopifyLine = collect($shopifyOrder->json_object['line_items'])->firstWhere('id',
                                    $fulfillmentLine->salesOrderLine->sales_channel_line_id);
                                // the line does not found in the shopify or the line is unfulfillable(already fulfilled or refunded)
                                if (! $shopifyLine || $shopifyLine['fulfillable_quantity'] == 0) {
                                    return null;
                                }

                                $quantity = $fulfillmentLine->quantity;

                                // Bundle handling
                                if ($bundle = $fulfillmentLine->salesOrderLine->bundle) {
                                    // get all fulfillment lines matching the bundle id
                                    $fulfillmentLinesFromBundle = $this->salesOrderFulfillment->salesOrderFulfillmentLines->filter(function ($fulfillmentLine) use ($bundle) {
                                        return $fulfillmentLine->salesOrderLine->bundle_id == $bundle->id;
                                    });

                                    /*
                                     * Each of these fulfillment lines for this particular fulfillment equate to a single shopify fulfillment line, so we need to figure
                                     * out the quantity of the bundle that was fulfilled
                                     */
                                    $bottleneckQuantity = $quantity;
                                    $fulfillmentLinesFromBundle->each(function ($fulfillmentLine) use ($bundle, &$bottleneckQuantity, &$fulfillmentLinesIdsAlreadyProcessed) {
                                        // get the component quantity
                                        $component = $bundle->components->where('id', $fulfillmentLine->salesOrderLine->product->id)->first();
                                        if($component){
                                            $componentQuantity = max(1, $component->pivot->quantity);

                                            // get the quantity of the fulfillment line
                                            $fulfilledQuantity = $fulfillmentLine->quantity;

                                            // get the quantity of the fulfillment line divided by the component quantity
                                            $quantityOfBundleShippedFromComponent = max(1, floor($fulfilledQuantity / $componentQuantity));

                                            // set the quantity of the fulfillment line to the quantity divided by the component quantity
                                            $bottleneckQuantity = min($quantityOfBundleShippedFromComponent, $bottleneckQuantity);

                                            // We need to make sure no other fulfillment lines get processed that were in fulfillment lines from bundle.
                                            $fulfillmentLinesIdsAlreadyProcessed[] = $fulfillmentLine->id;
                                        }
                                    });

                                    $quantity = $bottleneckQuantity;
                                }

                                // if some quantity may have been fulfilled or refunded
                                if ($quantity > $shopifyLine['fulfillable_quantity']) {
                                    $quantity = $shopifyLine['fulfillable_quantity'];
                                }
                                $line = collect($fulfillmentOrder['line_items'])->where('line_item_id',
                                    $fulfillmentLine->salesOrderLine->sales_channel_line_id)->first();

                                /*
                                 * If the line is not found in the fulfillment order, then we don't include it in submitting the fulfillment
                                 */
                                if (! $line) {
                                    return null;
                                }

                                $quantity = min($quantity, $line['fulfillable_quantity']);

                                return [
                                    'id' => $line['id'],
                                    'quantity' => $quantity,
                                ];
                            })->filter()->values()->toArray(),
                    ],
                ],

            ];
            customlog('submitShopifyFulfillments', 'fulfillment info sent to shopify for '.$this->salesOrderFulfillment->salesOrder->sales_order_number, $options, 7);
        }

        // there are no lines for fulfilling that means that the lines are refunded or already fulfilled
        if (empty($options['line_items_by_fulfillment_order'][0]['fulfillment_order_line_items'])) {
            $fulfillmentLineIds = $this->salesOrderFulfillment->salesOrderFulfillmentLines->pluck('salesOrderLine.sales_channel_line_id')->toArray();

            // check if the lines are refunded
            $refundedLineIds = collect($shopifyOrder->json_object['refunds'])->pluck('refund_line_items')->collapse()->pluck('line_item_id')->unique()->values()->toArray();
            // the lines have been refunded
            if (array_intersect($fulfillmentLineIds, $refundedLineIds)) {
                // TODO: what should we do?
                $this->salesOrderFulfillment->markAsSubmittedToSalesChannel(now());

                return ['success' => true, 'message' => 'the lines have been refunded (marked as submitted)'];
            }

            // check if the lines are fulfilled
            $orderFulfillments = collect($shopify->getFulfillments($shopifyOrder->shopify_order_id))->where('status', '=', 'success');
            // the fulfillment that has lines that we are trying to fulfill and it is not mapped before
            $shopifyFulfillment = $orderFulfillments->filter(fn ($fulfillment) => collect($fulfillment['line_items'])->whereIn('id', $fulfillmentLineIds)->isNotEmpty())
                ->whereNotIn('id', $this->shopifyOrderMappingRepository->getMappingsForTypeForShopifyOrder($shopifyOrder, ShopifyOrderMapping::LINK_TYPE_FULFILLMENTS)->pluck('link_id')->toArray())
                ->first();

            if ($shopifyFulfillment) {
                $this->salesOrderFulfillment->markAsSubmittedToSalesChannel(now());
                $shopifyOrder->mapFulfillmentToSkuFulfillment($shopifyFulfillment, $this->salesOrderFulfillment->id);

                return ['success' => true, 'message' => 'the lines have already been fulfilled (marked as submitted)'];
            }

            // the fulfillment has lines that we are trying to fulfill and it was already mapped
            $shopifyFulfillment = $orderFulfillments->filter(fn ($fulfillment) => collect($fulfillment['line_items'])->whereIn('id', $fulfillmentLineIds)->isNotEmpty())
                ->whereIn('id', $this->shopifyOrderMappingRepository->getMappingsForTypeForShopifyOrder($shopifyOrder, ShopifyOrderMapping::LINK_TYPE_FULFILLMENTS)->pluck('link_id')->toArray())
                ->first();

            if ($shopifyFulfillment) {
                $this->salesOrderFulfillment->markAsSubmittedToSalesChannel(now());

                return ['success' => true, 'message' => 'the lines have already been fulfilled (marked as submitted) and mapping already exists'];
            }
            //            try {
            //                ShopifyDownloadOrderrefresh($shopifyOrder);
            //            } catch (InsufficientStockException $e)
            //            {
            //                print "Insufficient stock for order {$shopifyOrder->shopify_order_id}, continuing...\n";
            //            }
            customlog('submitShopifyFulfillments', $this->salesOrderFulfillment->salesOrder->sales_order_number, [
                'shopifyFulfillmentOrders' => count($shopifyFulfillmentOrders),
                'openShopifyFulfillmentOrders' => count(collect($shopifyFulfillmentOrders)->whereIn('status', ['open', 'in_progress'])),
            ]);

            //customlog('syncNew', 'Fulfillment for ' . $this->salesOrderFulfillment->salesOrder->sales_order_number . ' has no fulfillable lines', $fulfillmentOrder);
            return ['success' => false, 'message' => 'the fulfillment does not have any fulfillable line and the shopify order does not have a fulfillment with these lines and the lines are not refunded'];
        }

        // send the fulfillment to Shopify
        try {
            $shopifyFulfillment = $shopify->createFulfillment($options);
        } catch (\Throwable $exception) {
            // some lines or all lines have been already fulfilled or unfulfillable(refunded)
            if (Str::endsWith($exception->getMessage(), 'already fulfilled')) {
                // trying to get the shopify fulfillment that contains these lines
                $orderFulfillments = collect($shopify->getFulfillments($shopifyOrder->shopify_order_id))->where('status', '!=', 'cancelled');
                // the fulfillment that has lines that we are trying to fulfill and it is not mapped before
                $shopifyFulfillment = $orderFulfillments->filter(fn ($fulfillment) => collect($fulfillment['line_items'])->whereIn('id', array_column($options['line_items'], 'id'))->isNotEmpty())
                    ->whereNotIn('id', $this->shopifyOrderMappingRepository->getMappingsForTypeForShopifyOrder($shopifyOrder, ShopifyOrderMapping::LINK_TYPE_FULFILLMENTS)->pluck('link_id')->toArray())
                    ->first();
                if (! $shopifyFulfillment) {
                    return ['success' => false, 'message' => 'the lines marked as fulfilled but the shopify order does not have a fulfillment with these lines'];
                }
            } else {
                return ['success' => false, 'message' => $exception->getMessage()];
            }
        }
        // the fulfillment submitted successfully (or it was already submitted)
        $this->salesOrderFulfillment->markAsSubmittedToSalesChannel(now());
        $shopifyOrder->mapFulfillmentToSkuFulfillment($shopifyFulfillment, $this->salesOrderFulfillment->id);

        return ['success' => true, 'message' => 'The fulfillment submitted successfully'];
    }
}
