<?php

namespace App\Services\Shopify;

use App\Integrations\Listings\ProductMapper;
use App\Jobs\Shopify\ShopifyMapSalesOrderLines;
use App\Models\IntegrationInstance;
use App\Models\ProductListing;
use App\Models\Shopify\ShopifyProduct;
use App\Models\Shopify\ShopifyWebhook;
use App\Models\TaskStatus\TaskStatus;
use App\Response;
use App\Services\StockTake\OpenStockTakeException;
use Carbon\Carbon;
use Illuminate\Support\Facades\DB;
use InvalidArgumentException;
use JetBrains\PhpStorm\ArrayShape;

/**
 * Class ShopifyListingMapper.
 */
class ShopifyListingMapper implements ProductMapper
{
    /**
     * @var IntegrationInstance
     */
    protected $integrationInstance;

    /**
     * ShopifyListingMapper constructor.
     */
    public function __construct(IntegrationInstance $integrationInstance)
    {
        $this->integrationInstance = $integrationInstance;
    }

    /**
     * @return string[]
     */
    public function validationRules(): array
    {
        return [
            'mapping' => 'required|array|min:1',
            'mapping.*.variant_id' => 'required_without:mapping.*.document_id|exists:shopify_products,variant_id',
            'mapping.*.document_id' => 'required_without:mapping.*.variant_id|exists:shopify_products,id',
            'mapping.*.product_id' => 'exists:products,id|nullable',
        ];
    }

    /**
     * @return mixed|void
     */
    #[ArrayShape(['mapped' => 'int', 'unmapped' => 'int', 'errors' => 'array'])]
    public function mapListings(array $mappings, ?TaskStatus $taskStatus = null)
    {
        $variantIds = collect($mappings)->pluck('variant_id')
            ->filter(function ($row) {
                return (bool) $row;
            })
            ->map(function ($id) {
                return (int) $id;
            })
            ->toArray();

        $documentIds = collect($mappings)->pluck('document_id')
            ->filter(function ($row) {
                return (bool) $row;
            })->toArray();

        $shopifyProducts = ShopifyProduct::with([])
            ->Where('integration_instance_id', $this->integrationInstance->id)
            ->whereIn('variant_id', $variantIds);
        if (count($documentIds) > 0) {
            $shopifyProducts = $shopifyProducts->orWhereIn('id', $documentIds);
        }

        $shopifyProducts = $shopifyProducts->get();
        // Fill in missing variant ids
        $mappings = array_map(function ($mapping) use ($shopifyProducts) {
            if (isset($mapping['variant_id'])) {
                return $mapping;
            }

            $match = $shopifyProducts->firstWhere('id', $mapping['document_id']);

            if (! $match) {
                throw new InvalidArgumentException('Invalid document id: '.$mapping['document_id']);
            }
            $mapping['variant_id'] = $match->variant_id;

            return $mapping;
        }, $mappings);

        $mapped = 0;
        $unmapped = 0;
        $errors = [];
        foreach ($mappings as $mapping) {
            if (empty($mapping['product_id'])) {
                $this->unmapByVariantId($mapping['variant_id']);
                $unmapped++;

                continue;
            }

            $productDocument = $shopifyProducts->firstWhere('variant_id', (int) $mapping['variant_id']);
            if (! $productDocument) {
                // Document does not exist, we skip (Kalvin)
                $errors[$mapping['variant_id']] = ["Skipping variant_id: {$mapping['variant_id']}, document doesn't exist.", Response::CODE_NOT_FOUND, $mapping['variant_id']];
                if ($taskStatus) {
                    $taskStatus->addErrorMessage("Skipping variant_id: {$mapping['variant_id']}, document doesn't exist.");
                }

                continue;
            }

            try {
                DB::transaction(function () use ($productDocument, $mapping) {
                    $this->mapListing($mapping, $productDocument);
                });
                $mapped++;
            } catch (OpenStockTakeException $openStockTakeException) {
                $errors[$mapping['variant_id']] = [$openStockTakeException->getMessage(), Response::CODE_OPEN_STOCK_TAKE, $mapping['variant_id']];
            }
        }

        return ['mapped' => $mapped, 'unmapped' => $unmapped, 'errors' => $errors];
    }

    /**
     * @throws \Exception
     */
    private function unmapByVariantId($variantId)
    {
        $productListing = ProductListing::with([])
            ->where('sales_channel_listing_id', $variantId)
            ->where('sales_channel_id', $this->integrationInstance->salesChannel->id)
            ->first();

        if ($productListing) {
            $this->unmapListings($productListing);
        }
    }

    private function mapListing(array $mapping, ShopifyProduct $productDocument)
    {
        /** @var ProductListing $productListing */
        $productListing = ProductListing::with([])->firstOrNew([
            'sales_channel_id' => $this->integrationInstance->salesChannel->id,
            'sales_channel_listing_id' => $productDocument->variant_id, ]);

        // sales_channel_listing_id based on integration instance
        $productListing->product_id = $mapping['product_id'];
        $productListing->document_id = $productDocument['id'];
        $productListing->document_type = \App\Models\Shopify\ShopifyProduct::class;

        $productListing->listing_sku = $productDocument['sku'];
        $productListing->title = $productDocument['title'];

        // set sales channel quantity
        if (ShopifyWebhook::hasInventoryLevelUpdateWebhook($this->integrationInstance->id)) {
            $productListing->sales_channel_qty = $productDocument['inventory_quantity'];
            if (@$productDocument['is_mysql']) {
                $productListing->sales_channel_qty_last_updated = $productDocument->updated_at->copy()->setTimezone('UTC');
            } else {
                $productListing->sales_channel_qty_last_updated = Carbon::parse($productDocument['updated_at'])->setTimezone('UTC');
            }
        }

        $productListing->save();
        $productDocument->product = $productListing->id;
        $productDocument->update();

        $productListingStatus = $productListing->wasRecentlyCreated ? ShopifyMapSalesOrderLines::STATUS_NEW : ShopifyMapSalesOrderLines::STATUS_UPDATED;
        dispatch_sync(( new ShopifyMapSalesOrderLines($this->integrationInstance, $productListing, $productListingStatus) )->onConnection('sync'));
    }

    public function unmapListings(ProductListing $productListing)
    {
        ShopifyProduct::where('product', $productListing->id)->update([
            'product' => null,
        ]);

        $productListing->delete();

        dispatch_sync(( new ShopifyMapSalesOrderLines($this->integrationInstance, $productListing, ShopifyMapSalesOrderLines::STATUS_DELETED) )->onConnection('sync'));
    }
}
