<?php

namespace App\Services\ShippingProvider;

use App\Exceptions\SalesOrder\SalesOrderFulfillmentDispatchException;
use App\Models\IntegrationInstance;
use App\Models\SalesOrderFulfillment;
use App\Notifications\SalesOrderFulfilledNotification;
use App\Services\SalesOrderFulfillment\SubmitTrackingInfo;
use Illuminate\Database\Eloquent\Collection;
use Illuminate\Support\Facades\Notification;
use RuntimeException;

abstract class ShippingProviderOrderManager
{

    /**
     * @param  IntegrationInstance  $instance
     * @param  ShippingProviderClient  $client
     * @param  ShippingProviderRepository  $orders
     */
    public function __construct(
        private readonly IntegrationInstance $instance,
        protected readonly ShippingProviderClient $client,
        private readonly ShippingProviderRepository $orders
    ){}

    /**
     * @param  SalesOrderFulfillment  $fulfillment
     * @return ShippingProviderOrder
     * @throws SalesOrderFulfillmentDispatchException
     */
    public function createFulfillment(SalesOrderFulfillment $fulfillment): ShippingProviderOrder{
        // Only fulfillments from a veracore warehouse should be submitted to veracore.
        if($fulfillment->warehouse_id != $this->instance->linkedWarehouse->id){
            throw new SalesOrderFulfillmentDispatchException(
                $fulfillment,
                "Fulfillment is not from a {$this->instance->name} warehouse."
            );
        }

        if($order = $this->orders->getOrderByFulfillmentId($fulfillment->id)){
            return $order;
        }

        try{
            $response = $this->client->sendFulfillmentOrder(
                payload: $this->makeFulfillmentOrderPayload($fulfillment),
            );
            return $this->orders->updateOrCreateByResponse($response, $fulfillment);
        } catch (RuntimeException $e){
            throw new SalesOrderFulfillmentDispatchException($fulfillment, $e->getMessage());
        }
    }


    /**
     * @return void
     */
    public function updateTrackingInfo(): void
    {

        if(empty($needingUpdates = $this->orders->getOrderIdsNeedingTrackingInfo())){
            // No orders need updates.
            return;
        }

        // Download the tracking info from shipping provider
        try{
            $orders = $this->client->getTrackingInfo($needingUpdates);
        }catch (RuntimeException){
            return;
        }

        $orders = $this->orders->updateAll($orders);

        // Update the SKU fulfillments with tracking info
        $fulfillments = $this->updateTrackingInfoForFulfillments($orders);

        // Submit tracking info to the sales channels
        /** @var SalesOrderFulfillment $fulfillment */
        foreach ($fulfillments as $fulfillment) {
            SubmitTrackingInfo::factory($fulfillment)?->submit();
            // Update the fulfillment status to fulfilled
            $fulfillment->salesOrder->updateFulfillmentStatus();

            // Notify the customer that their order has been fulfilled
            if ($fulfillment->salesOrder->salesChannel->emailsCustomers() && $fulfillment->fulfilled()) {
                Notification::send(
                    notifiables: [$fulfillment->salesOrder->customer],
                    notification: new SalesOrderFulfilledNotification($fulfillment->salesOrder)
                );
            }
        }
    }

    /**
     * @param  array  $orders
     * @return Collection|array<ShippingProviderOrder>
     */
    private function updateTrackingInfoForFulfillments(array $orders): Collection|array
    {

        $shippingMethods = $this->orders->getShippingMethodsIn($orders);

        $fulfillments = $this->orders->getFulfillmentsForOrders($orders);

        $fulfillmentData = array_map(function(SalesOrderFulfillment $current) use ($orders, $shippingMethods) {
            $matching = array_values(array_filter($orders, fn(ShippingProviderOrder $order) => $order->getOrderId() == $current->shipping_provider_id));

            if(empty($matching)) return false;

            /** @var ShippingProviderOrder $order */
            $order = $matching[0];

            $methodName = $order->getTrackingInfo()->shippingMethodName;
            $shippingMethod = array_values(array_filter($shippingMethods, fn($current) => data_get($current, 'full_name') == $methodName));

            // If the shipping method exists, we will use the ID for
            // the fulfillment. If not, we will use the name from the shipping provider.
            $foundShippingMethod = !empty($shippingMethod);

            return [
                'id' => $current->id,
                'tracking_number' => $order->getTrackingInfo()->trackingNumber,
                'fulfillment_type' => $current->fulfillment_type,
                'fulfilled_at' => $order->getTrackingInfo()->fulfillmentDate,
                'status' => SalesOrderFulfillment::STATUS_FULFILLED,
                'cost' => $order->getTrackingInfo()->cost,
                'fulfilled_shipping_method_id' => $foundShippingMethod ? $shippingMethod[0]['id'] : null,
                'fulfilled_shipping_method' => !$foundShippingMethod ? $methodName : null,
            ];
        }, $fulfillments);

        // Filter out any records not matched to a shipping provider order.
        // This should be very rare but in case there is an edge case.
        $fulfillmentData = array_filter($fulfillmentData, fn($current) => is_array($current));

        $this->orders->updateFulfillmentsWithTrackingInfo($fulfillmentData);

        // We need to return the updated fulfillments, so we can submit tracking info to the sales channels.
        return SalesOrderFulfillment::query()
            ->whereIn('id', collect($fulfillmentData)->pluck('id')->toArray())
            ->get();
    }



    /**
     * @param  SalesOrderFulfillment  $fulfillment
     * @return array
     */
    public abstract function makeFulfillmentOrderPayload(SalesOrderFulfillment $fulfillment): array;

}