<?php

namespace App\Services\ShippingProvider;

use App\Data\CreateShippingProviderIntegrationInstanceData;
use App\Data\UpdateShippingProviderIntegrationInstanceData;
use App\Models\IntegrationInstance;
use App\Repositories\ShippingProviders\ShippingProviderRepository;
use App\Repositories\WarehouseRepository;
use InvalidArgumentException;
use Throwable;

class ShippingProviderIntegrationManager
{

    /**
     * @param  ShippingProviderRepository  $providers
     * @param  WarehouseRepository  $warehouses
     */
    public function __construct(
        private readonly ShippingProviderRepository $providers,
        private readonly WarehouseRepository $warehouses
    ){}

    /**
     * @param  CreateShippingProviderIntegrationInstanceData  $data
     * @return IntegrationInstance
     */
    public function createIntegrationInstance(
        CreateShippingProviderIntegrationInstanceData $data
    ): IntegrationInstance
    {
        if ($this->providers->providerNameExists($data->integration_settings->connection_name)) {
            throw new InvalidArgumentException("Provider name ({$data->integration_settings->connection_name}) already exists");
        }

        $instance = $this->providers->create($data);

        // Convert the linked warehouse into a 3pl warehouse if it isn't set.
        $this->warehouses->convertTo3plIfDirect($data->integration_settings->linked_warehouse);

        return $instance;
    }

    /**
     * @param  IntegrationInstance  $integrationInstance
     * @param  UpdateShippingProviderIntegrationInstanceData  $data
     * @return IntegrationInstance
     */
    public function updateIntegrationInstance(
        IntegrationInstance $integrationInstance,
        UpdateShippingProviderIntegrationInstanceData $data
    ): IntegrationInstance {
        return $this->providers->update($integrationInstance, $data);
    }

    /**
     * @param  IntegrationInstance  $integrationInstance
     * @return void
     * @throws Throwable
     */
    public function deleteIntegrationInstance(IntegrationInstance $integrationInstance)
    {
        // TODO: Should we delete the shipping provider orders?
        $integrationInstance->delete();
    }

}