<?php

namespace App\Services\SalesOrderFulfillment;

use App\Http\Requests\FulfillSalesOrderRequest;
use App\Managers\ProductInventoryManager;
use App\Models\InventoryMovement;
use App\Models\SalesOrderFulfillmentLine;
use App\Models\SalesOrderLine;
use App\Response;
use Illuminate\Support\Facades\DB;

class UpdateSalesOrderFulfillmentService extends SalesOrderFulfillmentService
{
    public function execute(FulfillSalesOrderRequest $request)
    {
        DB::transaction(function () use ($request) {
            // Normal fulfill sales order lines
            if (! $this->updateFulfillment($request)) {
                DB::rollBack();

                return;
            }

            // if the warehouse is enable dropship, add warning to update it manually
            if ($this->salesOrderFulfillment->warehouse->is_dropship) {
                Response::instance()->addWarning(__('messages.sales_order.can_not_update_dropship_po'), Response::CODE_UPDATE_MANUALLY, 'sales_order_fulfillment_id');
            }

            Response::instance()->setMessage(__('messages.success.update', [
                'resource' => 'sales order fulfillment',
                'id' => $this->salesOrderFulfillment->id,
            ]))
                ->addData($this->salesOrderFulfillment);
        });
    }

    /**
     * Create Sales Order Fulfillment.
     */
    private function updateFulfillment(FulfillSalesOrderRequest $request): bool
    {
        $inputs = $request->validated();

        $this->salesOrderFulfillment->update($inputs);

        if (array_key_exists('fulfillment_lines', $inputs)) {
            if (empty($inputs['fulfillment_lines'])) {
                $this->salesOrderFulfillment->salesOrderFulfillmentLines()->delete();
            } else {
                $salesOrderFulfillmentLinesIds = [];

                foreach ($inputs['fulfillment_lines'] as $index => $fulfillmentLine) {
                    // update existing sales order fulfillment line
                    if (isset($fulfillmentLine['id'])) {
                        /** @var SalesOrderFulfillmentLine $salesOrderFulfillmentLine */
                        $salesOrderFulfillmentLine = $this->salesOrderFulfillment->salesOrderFulfillmentLines->firstWhere('id', $fulfillmentLine['id']);
                        $salesOrderFulfillmentLine->quantity = $fulfillmentLine['quantity'];
                        $salesOrderFulfillmentLine->save();
                        // reset inventory movements
                        $salesOrderFulfillmentLine->inventoryMovements()->delete();
                    } else { // create a new fulfillment line
                        $salesOrderFulfillmentLine = new SalesOrderFulfillmentLine();
                        $salesOrderFulfillmentLine->sales_order_line_id = $fulfillmentLine['sales_order_line_id'];
                        $salesOrderFulfillmentLine->quantity = $fulfillmentLine['quantity'];
                        $this->salesOrderFulfillment->salesOrderFulfillmentLines()->save($salesOrderFulfillmentLine);
                    }

                    $salesOrderFulfillmentLine->salesOrderLine->fulfilled_quantity = $fulfillmentLine['quantity'];
                    $salesOrderFulfillmentLine->salesOrderLine->update();

                    // reverse reserved quantity for the product sales order lines
                    if ($salesOrderFulfillmentLine->salesOrderLine->product_id && ! $salesOrderFulfillmentLine->salesOrderLine->is_dropship) {
                        if (! $salesOrderFulfillmentLine->negateReservationMovements()) {
                            Response::instance()->error(Response::HTTP_BAD_REQUEST)
                                ->addError(__('messages.sales_order.no_reservation_movements'), 'SalesOrderLine'.Response::CODE_RESERVE_NOT_FOUND, "fulfillment_lines.{$index}.sales_order_line_id");

                            return false;
                        }
                    }

                    $salesOrderFulfillmentLinesIds[] = $salesOrderFulfillmentLine->id;
                }

                // delete unused fulfillment lines
                $this->salesOrderFulfillment->salesOrderFulfillmentLines()->whereNotIn('id', $salesOrderFulfillmentLinesIds)->delete();
            }
        } else {
            // if fulfilled_at changed, we will update movements date of the sales order fulfillment lines
            if (isset($inputs['fulfilled_at'])) {
                $salesOrderFulfillmentLinesIds = $this->salesOrderFulfillment->salesOrderFulfillmentLines->pluck('id');

                InventoryMovement::query()->where('link_type', SalesOrderFulfillmentLine::class)
                    ->whereIn('link_id', $salesOrderFulfillmentLinesIds)
                    ->update(['inventory_movement_date' => $this->salesOrderFulfillment->fulfilled_at]);
            }
        }

        // mark sales order as fulfilled
        $this->salesOrderFulfillment->salesOrder->updateFulfillmentStatus($this->salesOrderFulfillment->fulfilled_at);

        // cache inventory for all sales order line products (if any fulfillment line is deleted)
        $productIds = SalesOrderLine::query()
            ->where('sales_order_id', $this->salesOrderFulfillment->sales_order_id)
            ->whereNotNull('product_id')
            ->pluck('product_id');
        (new ProductInventoryManager($productIds))->setUpdateAverageCost(false)->updateProductInventoryAndAvgCost();

        return true;
    }
}
