<?php

namespace App\Services\SalesOrder;

use App\Data\FinancialLineData;
use App\Data\SalesOrderLineData;
use App\Data\SalesOrderLineMappingData;
use App\Data\UpdateSalesOrderData;
use App\Data\UpdateSalesOrderPayloadData;
use App\Exceptions\SalesOrder\InvalidProductWarehouseRouting;
use App\Helpers;
use App\Http\Controllers\Traits\OrdersExports;
use App\Jobs\AutomatedSalesOrderFulfillmentJob;
use App\Jobs\SyncBackorderQueueCoveragesJob;
use App\Jobs\UpdateProductsInventoryAndAvgCost;
use App\Models\FinancialLine;
use App\Models\FinancialLineType;
use App\Models\InventoryMovement;
use App\Models\Product;
use App\Models\SalesChannel;
use App\Models\SalesOrder;
use App\Models\SalesOrderFulfillment;
use App\Models\SalesOrderFulfillmentLine;
use App\Models\SalesOrderLine;
use App\Models\Setting;
use App\Repositories\InventoryMovementRepository;
use App\Repositories\ProductRepository;
use App\Repositories\PurchaseOrderRepository;
use App\Repositories\SalesOrder\SalesOrderRepository;
use App\Repositories\SalesOrderFulfillmentRepository;
use App\Repositories\SalesOrderLineRepository;
use App\Repositories\SettingRepository;
use App\Repositories\WarehouseRepository;
use App\Services\InventoryManagement\BulkInventoryManager;
use App\Services\InventoryManagement\InventoryManager;
use App\Services\SalesOrder\Actions\CalculateFulfilledQuantity;
use App\Services\SalesOrder\Actions\CheckIfTryingToChangeWarehouseOnFulfilledSalesOrderLine;
use App\Services\SalesOrder\Actions\DeleteRemovedSalesOrderLines;
use App\Services\SalesOrder\Actions\DeleteSubmittedFulfillmentsDueToSalesOrderChange;
use App\Services\SalesOrder\Actions\DetectChangesThatShouldCauseSubmittedFulfillmentsToBeDeleted;
use App\Services\SalesOrder\Actions\FindAddressesForSalesOrder;
use App\Services\SalesOrder\Actions\FindCustomerForSalesOrder;
use App\Services\SalesOrder\Actions\FulfilledOnCancelledSalesOrderLineHandler;
use App\Services\SalesOrder\Actions\HandleSalesOrderLineCancellations;
use App\Services\SalesOrder\Actions\HandleTaxesForSalesOrderUpdate;
use App\Services\SalesOrder\Actions\LoadExistingSalesOrderLines;
use App\Services\SalesOrder\Actions\LoadProductsIntoSalesOrderLines;
use App\Services\SalesOrder\Actions\SyncInventoryMovementReferences;
use App\Services\SalesOrder\Actions\TransformBundlesSalesOrderLinesIntoComponentsForUpdate;
use App\Services\SalesOrder\Actions\ProcessSalesOrderLines;
use App\Services\SalesOrder\Concerns\ManagesDropshipOrders;
use App\Support\Concurrency\ConcurrencyManager;
use DB;
use Exception;
use Facades\App\Services\SalesOrder\Actions\ApproveSalesOrder;
use Facades\App\Services\SalesOrder\Actions\SetLineWarehouse;
use Illuminate\Contracts\Container\BindingResolutionException;
use Illuminate\Support\Collection;
use Illuminate\Support\Facades\Pipeline;
use Spatie\LaravelData\Attributes\DataCollectionOf;
use Spatie\LaravelData\DataCollection;
use Spatie\LaravelData\Optional;
use Throwable;

class SalesOrderManager
{
    use ManagesDropshipOrders;
    use OrdersExports;

    public function __construct(
        protected readonly SalesOrderRepository $orders,
        protected readonly PurchaseOrderRepository $purchaseOrders,
        protected readonly WarehouseRepository $warehouses,
        protected readonly ProductRepository $productRepository,
        protected readonly SalesOrderFulfillmentRepository $fulfillments,
        protected readonly InventoryMovementRepository $movements,
        protected readonly SalesOrderLineInventoryManager $lineInventoryManager,
        protected readonly ConcurrencyManager $concurrency,
        protected readonly SalesOrderLineRepository $orderLines,
        protected readonly ProductRepository $products,
    ) {
    }

    /**
     * @throws InvalidProductWarehouseRouting
     * @throws Throwable
     */
    public function createOrder(array $payload): SalesOrder
    {
        /** @var SalesOrder $order */
        $order = $this->concurrency->allOrNothing(function () use ($payload) {
            customlog('shopifyOrdersBenchmark', 'Sales Order Manager - createOrder - start');
            // We create the sales order
            $order = $this->orders->create($payload);

            customlog('shopifyOrdersBenchmark',
                $order->sales_order_number.': Sales Order Manager - create/prepareOrder - start');
            // We prepare the sales order for processing. This involves mapping
            // lines to warehouses based on the routing method chosen and
            // splitting externally fulfilled lines into their own lines.
            $order = $this->prepareOrder($order);

            customlog('shopifyOrdersBenchmark',
                $order->sales_order_number.': Sales Order Manager - create/other - start');
            // We approve the order if it's demanded and can be approved.
            // Note that a sales order is approved once and only a draft
            // order can be approved. Also, pre-audit trail orders are
            // not approved by default. They're approved at the point
            // of fulfillment.

            if ($this->requiresApproval($payload)) {
                $order = ApproveSalesOrder::approve($order, $this->unpaidOrdersMarkReserved($order));

                // Refresh product inventory cache
                $this->refreshProductInventoryCache($order);
            }

            // We create a dropship purchase order if any of the sales
            // order lines is a dropship line.
            if ($this->orderHasDropshippableLines($order)) {
                $order = $this->createDropshipPurchaseOrdersFor($order);
            }

            // We should handle the fulfillment status here.  If the fulfillment status is "out_of_sync", then the
            // handler should be skipped
            if ($order->fulfillment_status !== SalesOrder::FULFILLMENT_STATUS_OUT_OF_SYNC && $order->fulfillment_status != SalesOrder::FULFILLMENT_STATUS_FULFILLED && (!$this->requiresApproval($payload) || !$order->isReserved()) && !$order->isDraft()) {
                $order->updateFulfillmentStatus(null, false);
            }

            // We handle field caching for the order.
            $cacheableFields = $this->handleCacheableFields($order, $payload);

            customlog('shopifyOrdersBenchmark',
                $order->sales_order_number.': Sales Order Manager - createOrder - finish');

            return $cacheableFields;
        }, 5);

        if (!$order->salesChannel->integrationInstance->isShopify())
        {
            $order = $this->handleAutoFulfillment($order);
        }

        return $order;
    }

    /**
     * @throws Throwable
     */
    public function updateOrder(UpdateSalesOrderData $data): SalesOrder
    {
        /** @var SalesOrder $order */
        $order = $this->concurrency->allOrNothing(function () use ($data)
        {
            // Caching the original lines on the sales order before running the update.
            // This is used to determine if warehouses are changing so we handle inventory properly.
            $originalLines = collect($data->salesOrder->salesOrderLines->toArray());

            $data = Pipeline::send($data)
                ->through([
                    FulfilledOnCancelledSalesOrderLineHandler::class,
                    FindAddressesForSalesOrder::class,
                    FindCustomerForSalesOrder::class,
                    DeleteRemovedSalesOrderLines::class,
                    LoadProductsIntoSalesOrderLines::class,
                    TransformBundlesSalesOrderLinesIntoComponentsForUpdate::class,
                    LoadExistingSalesOrderLines::class,
                    HandleSalesOrderLineCancellations::class,
                    CheckIfTryingToChangeWarehouseOnFulfilledSalesOrderLine::class,
                    // Set warehouse id here?
                    HandleTaxesForSalesOrderUpdate::class,
                    DetectChangesThatShouldCauseSubmittedFulfillmentsToBeDeleted::class,
                    DeleteSubmittedFulfillmentsDueToSalesOrderChange::class,
                    CalculateFulfilledQuantity::class,
                    SyncInventoryMovementReferences::class,
                ])
                ->thenReturn();

            $updatedSalesOrder = $this->orders->updateOrder($data);

            $payload = $data->payload->toArray();

            if ($this->cancelingOrder($payload)) {
                // Check if original order was closed, if not we need to cancel it
                return $data->salesOrder->canceled_at ? $updatedSalesOrder : $updatedSalesOrder->cancel(
                    @$payload['canceled_at'] ?? null
                );
            }

            // We process the sales order. This involves mapping
            // lines to warehouses based on the routing method chosen,
            // splitting externally fulfilled lines into their own lines
            // and approving the sales order if approval is requested.
            $updatedSalesOrder = $this->prepareOrder($updatedSalesOrder);

            //uncancel order
            if (
                array_key_exists('canceled_at', $payload)
                && is_null($payload['canceled_at'])
                && $updatedSalesOrder->order_status === SalesOrder::STATUS_CLOSED
            ) {
                $updatedSalesOrder->order_status = SalesOrder::STATUS_DRAFT;
                $updatedSalesOrder->canceled_at  = null;
                $updatedSalesOrder->save();
                /*
                 * This extra step is needed so that the conditionals below will work correctly, especially to run
                 * through the approval process.
                 */
                $payload['order_status'] = SalesOrder::STATUS_OPEN;
            }

            if ($this->requiresApproval($payload)) {
                customlog('SKU-6240', $updatedSalesOrder->sales_order_number.' requires approval');
                $updatedSalesOrder = ApproveSalesOrder::approve($updatedSalesOrder, $this->unpaidOrdersMarkReserved($updatedSalesOrder), $originalLines);

                // Refresh product inventory cache
                $this->refreshProductInventoryCache($updatedSalesOrder);
            } elseif (($updatedSalesOrder->isOpen() || $updatedSalesOrder->isReserved()) && $this->payloadContainsKey($payload,
                    'sales_order_lines')) {
                customlog('SKU-6240', $updatedSalesOrder->sales_order_number.' does NOT requires approval');
                // Since this is an open order, we sync the inventory
                // reservations for the order.

                $updatedSalesOrder->warehousedProductLines()->each(function (SalesOrderLine $orderLine) use ($originalLines) {
                    $this->lineInventoryManager->refreshLineReservation(
                        orderLine: $orderLine,
                        originalLine: $originalLines->where('id', $orderLine->id)->first()
                    );
                });
                // Refresh product inventory cache.
                $this->refreshProductInventoryCache($updatedSalesOrder);
            } else {
                customlog('SKU-6240', $updatedSalesOrder->sales_order_number.' is not open or reserved', [
                    'order_status' => $updatedSalesOrder->order_status,
                    'is_open' => $updatedSalesOrder->isOpen(),
                    'is_reserved' => $updatedSalesOrder->isReserved(),
                    'payload' => $payload,
                ]);
            }

            // Handle dropship in sales order update.
            $updatedSalesOrder = $this->handleDropshipInOrderUpdate($updatedSalesOrder);

            if (
                $updatedSalesOrder->fulfillment_status != SalesOrder::FULFILLMENT_STATUS_OUT_OF_SYNC && // Don't update fulfillment status for out of sync orders
                !$updatedSalesOrder->isDraft() && // Don't update fulfillment status for draft orders
                !@$payload['on_hold'] // If not on hold, update fulfillment status
            ) {
                $updatedSalesOrder->updateFulfillmentStatus();
            }

            $this->handleCacheableFields($updatedSalesOrder, $payload);

            if (isset($payload['order_status']) && $payload['order_status'] === SalesOrder::STATUS_DRAFT
                && !$updatedSalesOrder->isDraft()) {
                // Reverting sales order to draft
                $response = $updatedSalesOrder->revertToDraft();
                if ($response) {
                    $updatedSalesOrder->order_status = SalesOrder::STATUS_DRAFT;
                    $updatedSalesOrder->save();
                }
            } elseif (isset($payload['order_status']) && $payload['order_status'] === SalesOrder::STATUS_RESERVED && !$updatedSalesOrder->isReserved() ||
                ($updatedSalesOrder->isReserved() && array_key_exists('sales_order_lines', $payload))) {
                $updatedSalesOrder->reserve(
                    !$updatedSalesOrder->isDraft() ? ApproveSalesOrderService::APPROVE_DROPSHIP_UPDATE_ONLY
                        : true,
                    $updatedSalesOrder->isDraft()
                );
            }

            customlog('shopifyOrdersBenchmark',
                $updatedSalesOrder->sales_order_number.': Sales Order Manager - updateOrder - finish');

            // Final check to see if order is unpaid and needs to be marked reserved
            if ($this->unpaidOrdersMarkReserved($updatedSalesOrder)) {
                $updatedSalesOrder->order_status = SalesOrder::STATUS_RESERVED;
                $updatedSalesOrder->save();
            }

            return $updatedSalesOrder;
        }, 1);

        $this->dispatchBackorderQueueCoveragesForSalesOrderUpdate($data);

        return $this->handleAutoFulfillment($order);
    }

    private function dispatchBackorderQueueCoveragesForSalesOrderUpdate(UpdateSalesOrderData $data): void
    {
        if ($data->productWarehousePairsNeedingCoverageUpdate instanceof Optional) {
            return;
        }
        $data->productWarehousePairsNeedingCoverageUpdate->toCollection()->unique()->groupBy('warehouse_id')->each(function (Collection $items, int $warehouseId) {
            dispatch(new SyncBackorderQueueCoveragesJob(productIds: $items->pluck('product_id')->toArray(), warehouseId: $warehouseId));
        });
    }

    private function handleAutoFulfillment(SalesOrder $order): SalesOrder
    {
        if ($order->isOpen() && $order->is_fully_fulfillable) {
            // Handle auto fulfillment
            dispatch(new AutomatedSalesOrderFulfillmentJob($order))->onQueue('automatedFulfillments');
        }

        return $order;
    }


    /**
     * @param  SalesOrder  $salesOrder
     * @return void
     */
    protected function refreshProductInventoryCache(SalesOrder $salesOrder): void
    {
        $job = new UpdateProductsInventoryAndAvgCost($salesOrder->nonDropshipProductLines->pluck('product_id')->toArray());
        $job->setUpdateAverageCost(false);
        dispatch($job);
    }

    /**
     * @throws InvalidProductWarehouseRouting
     * @throws Throwable
     */
    private function prepareOrder(SalesOrder $order): SalesOrder
    {
        // We split externally fulfilled lines into their
        // own sales order lines. This provides more clarity
        // to users about such lines, and they can convert
        // such lines into SKU fulfillments.
        $order = $this->splitPartiallyExternallyFulfilledLines($order);

        // TODO: could use a test here

        // We map warehouses to product lines based on
        // the warehouse selection method on the lines.
        return $this->processWarehouseSelectionForLines($order);
    }

    /**
     * @throws Throwable
     */
    private function handleCacheableFields(SalesOrder $order, array $payload = []): SalesOrder
    {
        // We cache the currency rate
        $order->cacheCurrencyRate();

        // We update the tax and discount allocation of the order.
        return $order->updateDiscountAndTaxAllocation($payload);
    }

    /**
     * @throws InvalidProductWarehouseRouting
     * @throws Throwable
     */
    private function processWarehouseSelectionForLines(SalesOrder $order): SalesOrder
    {
        // For situations where the user requires all lines in the order
        // to belong to the same warehouse, we enforce this with a different approach.
        if ($this->allLinesMustBeAtSameWarehouse()) {
            $priorityWarehouse = $this->warehouses->getPriorityStockWarehouseIdForSalesOrderLines(
                salesOrderLines: $order->productLines,
                shippingAddress: $order->shippingAddress
            );
            // For product lines with changing warehouse, we remove the inventory at the original warehouse,
            // except for lines with finalized fulfillments.
            $affectedLines = $order->productLines()
                ->whereDoesntHave('salesOrderFulfillmentLines.salesOrderFulfillment', function ($q) {
                    return $q->where('status', SalesOrderFulfillment::STATUS_FULFILLED);
                })
                ->whereNotNull('product_id')
                ->whereNull('warehouse_id')
                ->where('warehouse_id', '!=', $priorityWarehouse)
                ->get();

            // Update warehouse id for lines without fulfillments.
            $order->productLines()
                ->whereDoesntHave('salesOrderFulfillmentLines.salesOrderFulfillment', function ($q) {
                    return $q->where('status', SalesOrderFulfillment::STATUS_FULFILLED);
                })
                ->whereNotNull('product_id')
                ->whereNull('warehouse_id')
                ->each(function (SalesOrderLine $salesOrderLine) use ($priorityWarehouse) {
                    $salesOrderLine->warehouse_id = $priorityWarehouse;
                    $salesOrderLine->save();
                });


            // Update dropship lines
            $order
                ->dropshipLines()
                ->whereNull('warehouse_id')
                ->each(fn(SalesOrderLine $line) => SetLineWarehouse::setWarehouse($line));

            if ($affectedLines->isNotEmpty()) {
                foreach ($affectedLines as $orderLine) {
                    if ($orderLine->warehouse_id) {
                        InventoryManager::with(warehouseId: $orderLine->warehouse_id, product: $orderLine->product)
                            ->reverseNegativeEvent($orderLine);
                    }
                }
            }

            return $order;
        }
        $order->productLines()->each(/**
         * @throws Throwable
         */ function (SalesOrderLine $orderLine) {
            SetLineWarehouse::setWarehouse($orderLine);
        });

        return $order;
    }

    private function allLinesMustBeAtSameWarehouse(): bool
    {
        $setting = SettingRepository::getSettingByKey(Setting::KEY_NEVER_SPLIT_SALES_ORDERS_ACROSS_WAREHOUSES);

        return !empty($setting) && boolval($setting);
    }

    /**
     * @throws Exception
     */
    private function splitPartiallyExternallyFulfilledLines(SalesOrder $order): SalesOrder
    {
        $order->productLines()->each(/**
         * @throws Exception
         */ function (SalesOrderLine $orderLine) {
            if ($orderLine->externally_fulfilled_quantity > 0) {
                $orderLine->splitExternallyFulfilled();
            }
        });

        return $order;
    }

    private function requiresApproval(array $payload): bool
    {
        return isset($payload['order_status']) && ($payload['order_status'] == SalesOrder::STATUS_OPEN || $payload['order_status'] == SalesOrder::STATUS_RESERVED);
    }

    private function unpaidOrdersMarkReserved(SalesOrder $order): bool
    {
        // Return false if paid
        if ($order->payment_status == SalesOrder::PAYMENT_STATUS_PAID) {
            return false;
        }

        // Return false if setting not enabled
        if (!Helpers::setting(Setting::KEY_SO_UNPAID_ORDERS_AS_RESERVED)) {
            return false;
        }

        return true;
    }

    protected function payloadContainsKey(array $payload, string $key): bool
    {
        return isset($payload[$key]);
    }

    private function cancelingOrder(array $payload): bool
    {
        return isset($payload['canceled_at']) && $payload['canceled_at'];
    }

    public function lineReservation(SalesOrder $order, array $payload): SalesOrder
    {
        collect($payload['sales_order_lines'])
            ->each(/**
             * @throws Throwable
             */ function ($orderLine) use ($order) {
                /** @var SalesOrderLine $warehousedLine */
                $warehousedLine = $order->warehousedProductLines->where('id', $orderLine['id'])->first();
                if (!$warehousedLine) {
                    return;
                }

                $action = LineReservationAction::from($orderLine['action']);
                match ($action) {
                    LineReservationAction::REVERSE_RESERVATION => $this->lineInventoryManager
                        ->reverseLineReservation($warehousedLine, $orderLine['quantity']),
                    LineReservationAction::RESERVE => $this->lineInventoryManager
                        ->reserveInventoryForLine($warehousedLine, $orderLine['quantity'])
                };
            });
        $this->refreshProductInventoryCache($order);

        return $order->load('salesOrderLines');
    }

    public function syncExternallyFulfilled(array $ids = []): void
    {
        $productIdsUpdated = [];

        /*
         * Get reservation releases prior to start date.  We want to confirm the fulfillment date for these and if the
         * fulfillment is before the inventory start date, we want to delete the fulfillment and their reservations.
         */
        $this->movements->getReservationReleasesPriorToStartDate($ids)->each(/**
         * @throws Exception
         */ function (InventoryMovement $inventoryMovement) use (&$productIdsUpdated) {
            // The fulfillment was already deleted
            if (!$inventoryMovement->link) {
                return;
            }

            /** @var SalesOrderFulfillmentLine $salesOrderFulfillmentLine */
            $salesOrderFulfillmentLine = $inventoryMovement->link;

            /** @var SalesOrderFulfillment $salesOrderFuflillment */
            $salesOrderFulfillment = $salesOrderFulfillmentLine->salesOrderFulfillment;

            // TODO: Should this be fulfilled_at or created_at?
            if ($salesOrderFulfillment->fulfilled_at < Helpers::setting(Setting::KEY_INVENTORY_START_DATE)) {
                $salesOrderFulfillment->salesOrderFulfillmentLines->each(function (
                    SalesOrderFulfillmentLine $salesOrderFulfillmentLine
                ) use (&$productIdsUpdated) {
                    $salesOrderLine = $salesOrderFulfillmentLine->salesOrderLine;

                    // What happens if the sales order fulfillment line quantity < the line quantity?  split the line
                    $salesOrderLine->externally_fulfilled_quantity = $salesOrderFulfillmentLine->quantity;
                    $salesOrderLine->save();
                    if ($splitLine = $salesOrderLine->splitExternallyFulfilled()) {
                        $productIdsUpdated[] = $splitLine->product->id;
                    }

                    $productIdsUpdated[] = $salesOrderLine->product_id;
                });

                // This should delete any inventory movements as well
                $salesOrderFulfillment->delete(true);
            } else {
                // This is unexpected behavior so want to throw an exception to analyze the data for now if it happens
                throw new Exception('Fulfillment date is after inventory start date.  This is unexpected behavior.  Please analyze the data and fix the issue.');
                //$inventoryMovement->inventory_movement_date = $salesOrderFulfillment->fulfilled_at;
            }
        });

        /*
         * Delete any reservations prior to inventory start date.  First confirm there are no sales order fulfillment
         * lines for the order line
         */
        $this->movements->getReservationsPriorToStartDate($ids)->each(/**
         * @throws Exception
         */ function (InventoryMovement $inventoryMovement) use (&$productIdsUpdated) {
            /** @var SalesOrderLine $salesOrderLine */
            $salesOrderLine = $inventoryMovement->link;

            $productIdsUpdated[] = $salesOrderLine->product_id;

            if ($salesOrderLine->salesOrderFulfillmentLines->count()) {
                $inventoryMovement->inventory_movement_date = $salesOrderLine->salesOrderFulfillmentLines->first()->salesOrderFulfillment->fulfilled_at;
                $inventoryMovement->save();
                //throw new Exception('Sales order fulfillment lines still exist for sales order line ' . $salesOrderLine->id . '.  Please analyze the data and fix the issue.');
            } else {
                $inventoryMovement->delete();
            }
        });

        $productIdsUpdated = array_unique($productIdsUpdated);

        dispatch(new UpdateProductsInventoryAndAvgCost($productIdsUpdated));
    }

    /**
     * @throws Throwable
     */
    public function mapSalesOrderLine(SalesOrderLine $salesOrderLine): void
    {
        $this->updateOrder(UpdateSalesOrderData::from([
            'salesOrder' => $salesOrderLine->salesOrder,
            'payload' => UpdateSalesOrderPayloadData::from([
                'sales_order_lines' => SalesOrderLineData::collection([$salesOrderLine->toArray()]),
            ]),
        ]));
    }

    /**
     * @throws Exception
     * @throws Throwable
     */
    public function convertSalesOrderLineFromBundleToComponents(SalesOrderLine $salesOrderLine): void
    {
        if (!$salesOrderLine->bundle()) {
            return;
        }

        if ($salesOrderLine->canceled_quantity > 0) {
            throw new Exception('convertSalesOrderLineFromBundleToComponents: Cannot convert a canceled bundle line to components');
        }

        if ($shopifyOrder = $salesOrderLine->salesOrder->shopifyOrder) {
            // For shopify orders, we don't know if the bundle quantity was already expanded out to the component quantity or not, so we double-check
            if (!$shopifyOrderLine = $shopifyOrder->orderLineItems()->where('line_id',
                $salesOrderLine->sales_channel_line_id)->first()) {
                throw new Exception('convertSalesOrderLineFromBundleToComponents: Could not find shopify order line for sales order line '.$salesOrderLine->id);
            }
            if ($shopifyOrderLine->line_item['quantity'] != $salesOrderLine->quantity) {
                $salesOrderLine->quantity = $shopifyOrderLine->line_item['quantity'];
            }
        }

        DB::transaction(function () use ($salesOrderLine) {
            $bundleLine = $salesOrderLine;

            if ($bundleLine->salesOrderFulfillmentLines->count() > 0) {
                $bundleLine->salesOrderFulfillmentLines()->first()->salesOrderFulfillment->delete(true);
                //                $bundleLine->salesOrderFulfillmentLines->each(function (SalesOrderFulfillmentLine $salesOrderFulfillmentLine) {
                //                    $salesOrderFulfillmentLine->salesOrderFulfillment->delete(true);
                //                });
            }

            $bundleProduct    = $bundleLine->product;
            $bundleComponents = $bundleProduct->components;
            foreach ($bundleComponents as $bundleComponent) {
                $componentLine = $this->productRepository->saveComponentLineFromBundle($bundleLine, $bundleProduct,
                    $bundleComponent);

                $this->lineInventoryManager->refreshLineReservation(
                    orderLine: $componentLine,
                );
            }

            $bundleLine->delete();
        });
    }

    public function fixBundlesForSalesOrder(SalesOrder $salesOrder): void
    {
        $linesToFix = $salesOrder
            ->salesOrderLines()
            ->where(function ($query) {
                $query->whereColumn('externally_fulfilled_quantity', '<', 'quantity');
                $query->orWhere('quantity', 0);
            })
            ->whereHas('product', function ($query) {
                $query->where('type', Product::TYPE_BUNDLE);
            });

        if ($linesToFix->count() == 0) {
            customlog('fixBundlesForSalesOrder', 'No Lines to Fix for '.$salesOrder->sales_order_number);

            return;
        }

        $linesToFix = $linesToFix->get();

        customlog('fixBundlesForSalesOrder', [
            'sales_order_id' => $salesOrder->id,
            'sales_order_number' => $salesOrder->sales_order_number,
            'lines_to_fix' => $linesToFix->pluck('product.sku')->toArray(),
        ]);

        $linesToFix->each(/**
         * @throws Throwable
         */ function (SalesOrderLine $salesOrderLine) {
            if ($salesOrderLine->product?->type == Product::TYPE_BUNDLE) {
                $this->convertSalesOrderLineFromBundleToComponents($salesOrderLine);
            }
        });
    }

    public function generateSalesOrderInvoice($salesOrder): string
    {
        return $this->generateJasperReport($salesOrder, 'SKU_Sales_Order.jrxml', $salesOrder->sales_order_number);
    }

    /**
     * @throws Throwable
     * @throws InvalidProductWarehouseRouting
     */
    public function duplicate(SalesOrder $salesOrder): SalesOrder
    {
        $salesOrderLines = $salesOrder->salesOrderLines->makeHidden([
            'id',
            'sales_order_id',
            'sales_channel_line_id',
            'product_listing_id',
            'fulfilled_quantity',
            'externally_fulfilled_quantity',
            'canceled_quantity',
            'has_backorder',
            'sales_order',
            'updated_at',
            'created_at',
        ])->toArray();

        $newSalesOrder = $salesOrder->replicate();
        unset($newSalesOrder->sales_order_date);
        unset($newSalesOrder->currency_rate);
        unset($newSalesOrder->packing_slip_printed_at);
        unset($newSalesOrder->updated_at);
        unset($newSalesOrder->created_at);
        $newSalesOrder->sales_order_number = SalesOrder::getNextLocalNumber();
        $newSalesOrder->order_status       = SalesOrder::STATUS_DRAFT;
        $newSalesOrder->fulfillment_status = SalesOrder::FULFILLMENT_STATUS_UNFULFILLED;
        $newSalesOrder->payment_status     = SalesOrder::PAYMENT_STATUS_UNPAID;
        $newSalesOrder->sales_channel_id   = SalesChannel::LOCAL_CHANNEL_ID;
        $newSalesOrder->sales_order_lines  = $salesOrderLines;

        return $this->createOrder($newSalesOrder->toArray());
    }

    /**
     * @throws Throwable
     */
    public function convertToFinancialLine(SalesOrderLine $salesOrderLine, FinancialLineType $financialLineType): void
    {
        DB::transaction(function () use ($salesOrderLine, $financialLineType) {
            $salesOrder       = $salesOrderLine->salesOrder;
            $financialLineDto = FinancialLineData::from([
                'sales_order_id' => $salesOrder->id,
                'financial_line_type_id' => $financialLineType->id,
                'nominal_code_id' => $financialLineType->nominal_code_id,
                'description' => $salesOrderLine->description,
                'amount' => $salesOrderLine->amount,
                'quantity' => $salesOrderLine->quantity,
                'tax_allocation' => $salesOrderLine->tax_allocation,
                'allocate_to_products' => $financialLineType->allocate_to_products,
                'proration_strategy' => $financialLineType->proration_strategy,
            ]);
            FinancialLine::create($financialLineDto->toArray());

            $this->updateOrder(UpdateSalesOrderData::from([
                'salesOrder' => $salesOrder,
                'payload' => UpdateSalesOrderPayloadData::from([
                    'sales_order_lines' => SalesOrderLineData::collection($salesOrder->salesOrderLines->reject(function ($line) use ($salesOrderLine) {
                        return $line->id == $salesOrderLine->id;
                    })),
                ]),
            ]));
        });
    }

    /**
     * @throws BindingResolutionException
     * @throws Throwable
     */
    public function mapSalesOrderLinesToSalesChannelProducts(
        #[DataCollectionOf(SalesOrderLineMappingData::class)] DataCollection $data
    ): void {
        $data = $data->toCollection();

        // TODO: Bulk functionality for getPriorityWarehouseIdForProduct

        $salesOrderLineCollection = $this->orderLines->getForValues($data->pluck('id')->toArray(), 'id',
            SalesOrderLine::class, ['salesOrder.salesChannel.integrationInstance.warehouse']);
        $productCollection        = $this->products->getForValues($data->pluck('product_id')->toArray(), 'id',
            Product::class);

        $salesOrderLineCollection = $data->map(/**
         * @throws Exception
         */ function (SalesOrderLineMappingData $salesOrderLineMappingData) use (
            $salesOrderLineCollection,
            $productCollection
        ) {
            $salesOrderLine                          = $salesOrderLineCollection->where('id',
                $salesOrderLineMappingData->id)->first();
            $product                                 = $productCollection->where('id',
                $salesOrderLineMappingData->product_id)->first();
            $integrationWarehouse                    = $salesOrderLine->salesOrder->salesChannel->integrationInstance->warehouse;
            $salesOrderLineMappingData->warehouse_id = $integrationWarehouse?->id ?? $this->warehouses->getPriorityWarehouseIdForProduct($product,
                $salesOrderLine->quantity);
            return $salesOrderLineMappingData;
        });

        DB::transaction(function () use ($salesOrderLineCollection, $data) {
            batch()->update(new SalesOrderLine(), $salesOrderLineCollection->toArray(), 'id');

            // Fetch the fresh sales order lines with the warehouse ids / product ids now updated
            $salesOrderLineCollection = $this->orderLines->getForValues($data->pluck('id')->toArray(), 'id',
                SalesOrderLine::class);

            $salesOrderLineCollection = $this->orderLines->sanitizeLinesForAllocation($salesOrderLineCollection);

            // Reserve inventory for the newly mapped lines
            app(BulkInventoryManager::class)->bulkAllocateNegativeInventoryEvents($salesOrderLineCollection);
        });
    }
}
